<?php
// Debug mode
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();
include 'db.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

date_default_timezone_set('Asia/Kolkata');

/**
 * Resolve target date (YYYY-MM-DD). Defaults to today. Sanitizes bad input.
 */
$today = date('Y-m-d');
$targetDate = isset($_GET['d']) ? trim($_GET['d']) : $today;
$dt = DateTime::createFromFormat('Y-m-d', $targetDate);
if (!$dt || $dt->format('Y-m-d') !== $targetDate) {
    $targetDate = $today; // fallback on invalid input
}

/**
 * Determine reference "now" for status logic:
 * - if viewing today -> real now
 * - if viewing a past date -> 23:59:59 of that date (so it's "closed")
 * - if viewing a future date -> 00:00:00 of that date (so it's "not started")
 */
if ($targetDate === $today) {
    $refNow = time();
} elseif ($targetDate < $today) {
    $refNow = strtotime($targetDate . ' 23:59:59');
} else {
    $refNow = strtotime($targetDate . ' 00:00:00');
}

/**
 * Detect which date column to use for filtering user_bids
 * Prefers created_at, then bid_date; fallback = no date filter if neither exists.
 */
function detect_date_column(mysqli $conn): ?string {
    $col = null;
    $res = $conn->query("SHOW COLUMNS FROM user_bids LIKE 'created_at'");
    if ($res && $res->num_rows > 0) { $col = 'created_at'; }
    if (!$col) {
        $res2 = $conn->query("SHOW COLUMNS FROM user_bids LIKE 'bid_date'");
        if ($res2 && $res2->num_rows > 0) { $col = 'bid_date'; }
        if ($res2) { $res2->free(); }
    }
    if ($res) { $res->free(); }
    return $col;
}
$dateCol = detect_date_column($conn);

// Fetch games sorted by open_time and close_time in ascending order
$sql = "SELECT * FROM game56 ORDER BY open_time ASC, close_time ASC";
$result = $conn->query($sql);
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Bid Totals per Number</title>
  <style>
    body { font-family: Arial, sans-serif; background: #f0f2f5; margin: 0; padding: 0; }
    .navbar { background: #007bff; color: #fff; text-align: center; padding: 12px; position: relative; }
    .navbar h2 { font-size: 1.2rem; margin: 0; }
    .back-button {
      position: absolute; left: 8px; top: 50%; transform: translateY(-50%);
      background: #28a745; color: #fff; padding: 6px 10px; text-decoration: none;
      border-radius: 4px; font-size: 0.85rem;
    }
    .filter-bar {
      background: #fff; border-bottom: 1px solid #e5e5e5; padding: 10px 12px;
      display: flex; align-items: center; gap: 8px; justify-content: center;
    }
    .filter-bar form { display: flex; align-items: center; gap: 8px; flex-wrap: wrap; }
    .filter-bar label { font-weight: bold; }
    .filter-bar input[type="date"] { padding: 6px 8px; border: 1px solid #ccc; border-radius: 4px; }
    .btn {
      padding: 6px 10px; border: none; border-radius: 4px; cursor: pointer; font-size: 0.9rem;
    }
    .btn-primary { background: #007bff; color: #fff; }
    .btn-secondary { background: #6c757d; color: #fff; }
    .content { max-width: 1200px; margin: 20px auto; padding: 0 12px; display: flex; flex-wrap: wrap; justify-content: space-between; }
    .game-card {
      background: #fff; border-radius: 10px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);
      padding: 16px; margin-bottom: 20px; width: 30%; box-sizing: border-box; transition: transform 0.2s;
    }
    .game-card:hover { transform: scale(1.05); }
    .game-card h3 { margin-bottom: 10px; text-align: center; background: #007bff; color: #fff; padding: 8px; border-radius: 6px; }
    .status-message { color: #d9534f; font-weight: bold; text-align: center; margin-bottom: 10px; }
    .summary-bar { display: flex; justify-content: space-between; margin-bottom: 10px; font-size: 0.9rem; font-weight: bold; color: #333; }
    .totals { display: flex; flex-wrap: wrap; justify-content: space-between; }
    .totals div {
      width: 48%; background: #f8f9fa; margin-bottom: 6px; padding: 8px; border-radius: 5px;
      border: 1px solid #ddd; display: flex; justify-content: space-between;
    }
    .totals div span.number { font-weight: bold; }
    .totals div.max { background: #d4edda; }
    .totals div.min { background: #f8d7da; }
    .meta { text-align: center; font-size: 0.9rem; color: #333; margin: 10px 0; }
    @media (max-width: 1024px) { .game-card { width: 48%; } }
    @media (max-width: 640px) { .game-card { width: 100%; } }
  </style>
</head>
<body>

<div class="navbar">
  <a href="dashboard.php" class="back-button">Back</a>
  <h2>Bid Totals per Number (0–9)</h2>
</div>

<div class="filter-bar">
  <form method="get" action="">
    <label for="d">Select Date:</label>
    <input type="date" id="d" name="d" value="<?php echo htmlspecialchars($targetDate); ?>" required>
    <button type="submit" class="btn btn-primary">Apply</button>
    <a href="?d=<?php echo htmlspecialchars($today); ?>" class="btn btn-secondary">Today</a>
  </form>
</div>

<div class="meta">
  Showing data for: <strong><?php echo htmlspecialchars($targetDate); ?></strong>
  <?php if (!$dateCol): ?>
    <div style="color:#a94442;margin-top:6px;">
      Note: No date column (created_at/bid_date) found in <code>user_bids</code>; totals include all bids for each game.
    </div>
  <?php endif; ?>
</div>

<div class="content">
<?php if ($result && $result->num_rows > 0): ?>
  <?php while ($row = $result->fetch_assoc()): ?>
    <?php
      // Build timestamps for the selected date
      $open_ts      = strtotime($targetDate . ' ' . $row['open_time']);
      $open_bid_ts  = strtotime($targetDate . ' ' . $row['open_bid_last_time']);
      $close_bid_ts = strtotime($targetDate . ' ' . $row['close_bid_last_time']);
      $close_ts     = strtotime($targetDate . ' ' . $row['close_time']);

      // Determine the game status relative to selected date
      if ($targetDate > $today) {
          $status_text = "Game has not started";
      } elseif ($targetDate < $today) {
          $status_text = "Game has Closed";
      } else {
          // today -> compare with real time (refNow)
          if ($refNow < $open_bid_ts) {
              $status_text = "Game has not started";
          } elseif ($refNow < $open_ts) {
              $status_text = "Open bidding is closed";
          } elseif ($refNow < $close_bid_ts) {
              $status_text = "Game is Running";
          } elseif ($refNow < $close_ts) {
              $status_text = "Close bidding is closed";
          } else {
              $status_text = "Game has Closed";
          }
      }

      // Initialize totals 0-9
      $totals = array_fill(0, 10, 0.0);

      // Fetch bids for this game (filtered by date if column exists)
      $gameName = $row['game_name'];
      if ($dateCol) {
          $stmt = $conn->prepare("SELECT bid_details FROM user_bids WHERE game_name = ? AND DATE($dateCol) = ?");
          $stmt->bind_param('ss', $gameName, $targetDate);
      } else {
          $stmt = $conn->prepare("SELECT bid_details FROM user_bids WHERE game_name = ?");
          $stmt->bind_param('s', $gameName);
      }
      if ($stmt && $stmt->execute()) {
          $resBids = $stmt->get_result();
          while ($b = $resBids->fetch_assoc()) {
              $details = $b['bid_details'];
              if (!$details) { continue; }
              $data = json_decode($details, true);

              // Case 1: JSON object like {"2":"98"}
              if (is_array($data) && array_keys($data) !== range(0, count($data) - 1)) {
                  foreach ($data as $num => $amt) {
                      $num = intval($num);
                      if ($num >= 0 && $num <= 9) {
                          $totals[$num] += floatval($amt);
                      }
                  }
              }
              // Case 2: Array of amounts [500,300,...] or ["98"]
              elseif (is_array($data)) {
                  foreach ($data as $num => $amt) {
                      $num = intval($num);
                      if ($num >= 0 && $num <= 9) {
                          $totals[$num] += floatval($amt);
                      }
                  }
              }
          }
          $resBids->free();
      }
      if ($stmt) { $stmt->close(); }

      // Max and min (non-zero min)
      $maxVal = max($totals);
      $maxIndex = array_search($maxVal, $totals);

      $nonZero = array_filter($totals, fn($v) => $v > 0);
      $minVal = $nonZero ? min($nonZero) : 0;
      $minIndex = $nonZero ? array_search($minVal, $totals) : '-';

      // Helper for currency formatting (keep simple, no decimals if integer)
      $fmt = function($v) {
          return (fmod($v, 1.0) === 0.0) ? number_format((int)$v) : number_format($v, 2);
      };
    ?>
    <div class="game-card">
      <h3><?php echo htmlspecialchars($row['game_name']); ?> (<?php echo htmlspecialchars($row['game_number']); ?>)</h3>
      <p class="status-message"><?php echo $status_text; ?></p>

      <!-- Game Time Information -->
      <p><strong>Open Time: </strong><?php echo htmlspecialchars($row['open_time']); ?></p>
      <p><strong>Close Time: </strong><?php echo htmlspecialchars($row['close_time']); ?></p>

      <!-- Summary Bar -->
      <div class="summary-bar">
        <div>Highest bidding number = <?php echo htmlspecialchars((string)$maxIndex); ?> (₹<?php echo $fmt($maxVal); ?>)</div>
        <div>Low bidding number = <?php echo htmlspecialchars((string)$minIndex); ?> (₹<?php echo $fmt($minVal); ?>)</div>
      </div>

      <div class="totals">
        <?php for ($i = 0; $i <= 9; $i++):
          $class = '';
          if ($totals[$i] == $maxVal && $maxVal > 0) $class = 'max';
          if ($totals[$i] == $minVal && $minVal > 0) $class = 'min';
        ?>
          <div class="<?php echo $class; ?>">
            <span class="number"><?php echo $i; ?></span>
            <span>₹<?php echo $fmt($totals[$i]); ?></span>
          </div>
        <?php endfor; ?>
      </div>
    </div>
  <?php endwhile; ?>
<?php else: ?>
  <p style="text-align:center; margin-top:32px;">No games available.</p>
<?php endif; ?>
</div>

</body>
</html>
