<?php
// _bootstrap.php — common helpers + IMB config + DB include (mysqli $conn)

$DB_PATH = realpath(__DIR__ . '/../../db.php');
if (!$DB_PATH || !file_exists($DB_PATH)) {
  http_response_code(500);
  header('Content-Type: application/json');
  echo json_encode(['ok'=>false,'error'=>'db.php not found at Sai/db.php']);
  exit;
}
require_once $DB_PATH; // defines $conn (mysqli)

function jheader() { header('Content-Type: application/json; charset=utf-8'); }
function jexit($arr, $code = 200) { http_response_code($code); echo json_encode($arr, JSON_UNESCAPED_UNICODE); exit; }

function post($key, $default = null) { return isset($_POST[$key]) ? trim((string)$_POST[$key]) : $default; }
function getv($key, $default = null) { return isset($_GET[$key]) ? trim((string)$_GET[$key]) : $default; }

/* ---- IMB PG config ---- */
const IMB_BASE   = 'https://pay.imb.org.in';         // use the API URL they show in your dashboard; if it shows pqy.imb.org.in, swap this host
const IMB_CREATE = IMB_BASE . '/api/create-order';
const IMB_STATUS = IMB_BASE . '/api/check-order-status';
const IMB_TOKEN  = 'a612b3ad1626aadf156ab82f8960ad4f';

/* ---- logging ---- */
$LOG_DIR = realpath(__DIR__ . '/../../logs') ?: __DIR__;
define('IMB_LOG_FILE', $LOG_DIR . '/imb_webhook.log');

/* ---- HTTP helper ---- */
function http_post_form($url, array $payload, $timeout = 30) {
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_POST           => true,
    CURLOPT_POSTFIELDS     => http_build_query($payload),
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT        => $timeout,
    CURLOPT_HTTPHEADER     => ['Content-Type: application/x-www-form-urlencoded']
  ]);
  $res  = curl_exec($ch);
  $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
  $err  = curl_error($ch);
  curl_close($ch);
  return [$http, $res, $err];
}

/* ---- DB helpers ---- */
function tableExists(mysqli $conn, string $table) : bool {
  $stmt = $conn->prepare("SELECT 1 FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = ?");
  $stmt->bind_param('s', $table);
  $stmt->execute();
  $stmt->store_result();
  $ok = $stmt->num_rows > 0;
  $stmt->close();
  return $ok;
}

function getUserIdByMobile(mysqli $conn, string $mobile) : ?int {
    $digits = preg_replace('/\D+/', '', $mobile ?? '');
    if ($digits === '') return null;

    if (tableExists($conn, 'users_data')) {
        $stmt = $conn->prepare(
            "SELECT user_id FROM users_data WHERE REPLACE(REPLACE(REPLACE(mobile,'+',''),' ',''),'-','') COLLATE utf8mb4_general_ci LIKE CONCAT('%',?) LIMIT 1"
        );
        $stmt->bind_param('s', $digits);
        $stmt->execute(); 
        $stmt->bind_result($uid);
        if ($stmt->fetch()) { 
            $stmt->close(); 
            return (int)$uid; 
        }
        $stmt->close();
    }

    return null;
}


function creditWallet(mysqli $conn, int $userId, float $amount) : bool {
  if ($amount <= 0) return false;
  $stmt = $conn->prepare("SELECT id, balance FROM wallet WHERE user_id=? LIMIT 1");
  $stmt->bind_param('i',$userId);
  $stmt->execute(); $stmt->bind_result($wid,$bal);
  if ($stmt->fetch()) {
    $stmt->close();
    $newBal = (float)$bal + $amount;
    $u = $conn->prepare("UPDATE wallet SET balance=?, updated_at=NOW() WHERE id=?");
    $u->bind_param('di',$newBal,$wid);
    $ok = $u->execute(); $u->close();
    return $ok;
  }
  $stmt->close();
  $i = $conn->prepare("INSERT INTO wallet (user_id, balance, created_at, updated_at) VALUES (?,?,NOW(),NOW())");
  $i->bind_param('id',$userId,$amount);
  $ok = $i->execute(); $i->close();
  return $ok;
}

/* ---- Normalize IMB status payload into ["success"|"failed"|"pending", utr] ---- */
function imb_normalize_status(array $payload) : array {
  // flatten & upper for string checks
  $raw   = json_encode($payload);
  $upper = strtoupper($raw);

  // UTR / RRN extraction
  $utr = null;
  if (isset($payload['result']) && is_array($payload['result'])) {
    $r = $payload['result'];
    $utr = $r['utr'] ?? ($r['bank_rrn'] ?? ($r['rrn'] ?? null));
  }

  // many variants observed in wild
  $ok =   strpos($upper,'"STATUS":"SUCCESS"')         !== false
       || strpos($upper,'"PAYMENT_STATUS":"SUCCESS"') !== false
       || strpos($upper,'"TXNSTATUS":"SUCCESS"')      !== false
       || strpos($upper,'"TXNSTATUS":"COMPLETED"')    !== false
       || strpos($upper,'"PAYMENTSTATUS":"SUCCESS"')  !== false
       || strpos($upper,'"MESSAGE":"PAYMENT SUCCESS"')!== false;

  $fail = strpos($upper,'"FAILED"')   !== false
       || strpos($upper,'"CANCELLED"')!== false
       || strpos($upper,'"REVERSED"') !== false;

  if ($ok)   return ['success',$utr];
  if ($fail) return ['failed',$utr];
  return ['pending',$utr];
}

/* ---- Server-to-server: IMB check status (send both order_id & orderId) ---- */
function imb_check_status(string $orderId) : array {
  [$http, $res, $err] = http_post_form(IMB_STATUS, [
    'user_token' => IMB_TOKEN,
    'order_id'   => $orderId,   // some accounts expect this
    'orderId'    => $orderId    // others expect this
  ], 25);

  // log everything to diagnose merchant-specific payloads
  @file_put_contents(IMB_LOG_FILE, date('c')." STATUS http=$http err=$err body=$res\n", FILE_APPEND);

  if ($http !== 200 || !$res) {
    return ['ok'=>false,'http'=>$http,'error'=>$err ?: 'Empty response'];
  }
  $data = json_decode($res, true);
  if (!is_array($data)) {
    return ['ok'=>false,'http'=>$http,'error'=>'Invalid JSON','raw'=>$res];
  }
  return ['ok'=>true,'data'=>$data,'raw'=>$res];
}
