<?php
// api_processor.php - Optimized for UptimeRobot monitoring
// Upload this file to your Hostinger public_html folder

// Error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Set execution time limit
set_time_limit(30);

// Include your database connection
require_once 'db.php';  // Make sure this file exists with your DB connection

// Configuration
$maxRecords = 10;       // Process max 10 records per run
$apiTimeout = 5;        // 5 second timeout for API calls
$logFile = 'uptimerobot_log.txt';

/**
 * Log messages with timestamp
 */
function logMessage($message, $logFile) {
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[{$timestamp}] {$message}" . PHP_EOL;
    
    // Keep log file manageable (max 50KB)
    if (file_exists($logFile) && filesize($logFile) > 51200) {
        $lines = file($logFile);
        $recentLines = array_slice($lines, -50); // Keep last 50 lines
        file_put_contents($logFile, implode('', $recentLines));
    }
    
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    return $logEntry;
}

/**
 * Process API calls
 */
function processApiCalls($conn, $maxRecords, $apiTimeout, $logFile) {
    $processed = 0;
    $success = 0;
    $errors = 0;
    
    // Get records from database
    $sql = "SELECT mobile, DATE(created_at) AS filter_date
            FROM users_data
            WHERE mobile IS NOT NULL
            LIMIT ?";
            
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("i", $maxRecords);
        $stmt->execute();
        $stmt->bind_result($mobile, $filter_date);
        
        while ($stmt->fetch()) {
            $processed++;
            
            // Prepare API payload
            $payload = json_encode([
                'mobile'      => $mobile,
                'filter_date' => $filter_date
            ]);
            
            // Make API call
            $ch = curl_init('https://sai7755.com/Sai/bids_result_api.php');
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST => true,
                CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
                CURLOPT_POSTFIELDS => $payload,
                CURLOPT_TIMEOUT => $apiTimeout,
                CURLOPT_CONNECTTIMEOUT => 3,
                CURLOPT_USERAGENT => 'UptimeRobot-Monitor/1.0'
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);
            
            // Process response
            if ($curlError) {
                $errors++;
                logMessage("❌ Mobile: {$mobile} - CURL Error: {$curlError}", $logFile);
            } elseif ($httpCode !== 200) {
                $errors++;
                logMessage("❌ Mobile: {$mobile} - HTTP Error: {$httpCode}", $logFile);
            } else {
                $jsonResponse = json_decode($response, true);
                if ($jsonResponse && isset($jsonResponse['data'])) {
                    $success++;
                    $recordCount = is_array($jsonResponse['data']) ? count($jsonResponse['data']) : 0;
                    logMessage("✅ Mobile: {$mobile} - Success: {$recordCount} records", $logFile);
                } else {
                    $errors++;
                    logMessage("❌ Mobile: {$mobile} - Invalid JSON response", $logFile);
                }
            }
            
            // Small delay between calls
            usleep(100000); // 0.1 second
        }
        $stmt->close();
    } else {
        logMessage("❌ Database Error: " . $conn->error, $logFile);
        return ['processed' => 0, 'success' => 0, 'errors' => 1];
    }
    
    return ['processed' => $processed, 'success' => $success, 'errors' => $errors];
}

// =================== MAIN EXECUTION ===================

$startTime = microtime(true);

// Log start
$logEntry = logMessage("🚀 API Processing Started via UptimeRobot", $logFile);
echo $logEntry;

try {
    // Process API calls
    $results = processApiCalls($conn, $maxRecords, $apiTimeout, $logFile);
    
    $endTime = microtime(true);
    $executionTime = round(($endTime - $startTime), 2);
    
    // Log summary
    $summary = "📊 Completed: {$results['processed']} processed, {$results['success']} success, {$results['errors']} errors in {$executionTime}s";
    $logEntry = logMessage($summary, $logFile);
    echo $logEntry;
    
    // Output status for UptimeRobot
    http_response_code(200);
    echo "OK - API Processing Complete\n";
    echo "Processed: {$results['processed']}\n";
    echo "Success: {$results['success']}\n";
    echo "Errors: {$results['errors']}\n";
    echo "Time: {$executionTime}s\n";
    
} catch (Exception $e) {
    $errorMsg = "💥 Fatal Error: " . $e->getMessage();
    logMessage($errorMsg, $logFile);
    
    http_response_code(500);
    echo "ERROR - " . $e->getMessage();
}

// Show recent logs if requested
if (isset($_GET['logs']) && $_GET['logs'] === '1') {
    echo "\n\n=== RECENT LOGS ===\n";
    if (file_exists($logFile)) {
        $lines = file($logFile);
        $recentLines = array_slice($lines, -10); // Last 10 lines
        foreach ($recentLines as $line) {
            echo $line;
        }
    } else {
        echo "No log file found.\n";
    }
}

// Simple status page if accessed via browser
if (isset($_GET['status']) && $_GET['status'] === '1') {
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <title>API Processor Status</title>
        <meta http-equiv="refresh" content="60">
        <style>
            body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
            .container { background: white; padding: 20px; border-radius: 8px; max-width: 800px; }
            .status { padding: 10px; margin: 10px 0; border-radius: 5px; }
            .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
            .error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
            pre { background: #f8f9fa; padding: 15px; border-radius: 5px; overflow-x: auto; }
        </style>
    </head>
    <body>
        <div class="container">
            <h2>🔄 API Processor Status</h2>
            <p><strong>Last Check:</strong> <?= date('Y-m-d H:i:s') ?></p>
            
            <?php if (file_exists($logFile)): ?>
                <div class="status success">
                    ✅ Log file exists (<?= number_format(filesize($logFile)) ?> bytes)
                </div>
                
                <h3>📝 Recent Activity (Last 15 lines):</h3>
                <pre><?php
                    $lines = file($logFile);
                    $recentLines = array_slice($lines, -15);
                    foreach ($recentLines as $line) {
                        echo htmlspecialchars($line);
                    }
                ?></pre>
            <?php else: ?>
                <div class="status error">
                    ❌ No log file found - Script may not have run yet
                </div>
            <?php endif; ?>
            
            <p><a href="?status=1">🔄 Refresh</a> | <a href="api_processor.php">▶️ Run Once</a></p>
        </div>
    </body>
    </html>
    <?php
    exit;
}
?>