<?php
// api_trs.php — SD + DP + SP + TP + JD + HS + FS bid history by mobile + date (for 60s polling).

// Response: { ok, mobile, date, refresh_interval_sec, count, items:[ ... ] }

(isset($_GET['debug']) ? ini_set('display_errors', 1) : ini_set('display_errors', 0));
error_reporting(E_ALL);
date_default_timezone_set('Asia/Kolkata');

// CORS
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Methods: POST, OPTIONS');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store');

/* ---------- payout multipliers (per ₹1 stake) ---------- */
$PAYOUT = [
  'SD' => 10,
  'JD' => 100,
  'SP' => 160,
  'DP' => 320,
  'TP' => 1000,
  'HS' => 1000,   // both parts must match in the given session
  'FS' => 10000,  // both parts must match in the given session
];


/* ---------- hardened JSON error handling ---------- */
function json_fatal($msg, $code = 500) {
  http_response_code($code);
  if (ob_get_length()) { ob_end_clean(); }
  echo json_encode(['ok'=>false, 'error'=>$msg], JSON_UNESCAPED_UNICODE);
  exit;
}
set_error_handler(function($errno,$errstr,$errfile,$errline){
  json_fatal("PHP error: $errstr at $errfile:$errline", 500);
});
set_exception_handler(function($ex){
  json_fatal("Exception: ".$ex->getMessage(), 500);
});
register_shutdown_function(function(){
  $e = error_get_last();
  if ($e && in_array($e['type'], [E_ERROR,E_PARSE,E_CORE_ERROR,E_COMPILE_ERROR])) {
    json_fatal("Fatal error: ".$e['message']." at ".$e['file'].":".$e['line'], 500);
  }
});

require __DIR__ . '/db.php';
require_once __DIR__ . '/lib/wallet_helper.php';
if (!isset($conn) || !($conn instanceof mysqli)) json_fatal('DB not connected', 500);

/* ---------- helpers ---------- */
function read_json_body() {
  $raw = file_get_contents('php://input');
  $j = json_decode($raw, true);
  return is_array($j) ? $j : [];
}
function normalize_date($in) {
  $in = trim((string)$in);
  if ($in === '') return date('Y-m-d');
  if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $in)) return $in;                // YYYY-MM-DD
  if (preg_match('/^(\d{2})[\/-](\d{2})[\/-](\d{4})$/', $in, $m)) {       // DD-MM-YYYY or DD/MM/YYYY
    return $m[3].'-'.$m[2].'-'.$m[1];
  }
  return date('Y-m-d');
}
/** DateTime from "HH:MM:SS" for a specific Y-m-d (IST). */
function dt_from_hms_for_date(?string $hms, string $ymd): ?DateTime {
  if (!$hms) return null;
  $dt = DateTime::createFromFormat('Y-m-d H:i:s', "$ymd $hms", new DateTimeZone('Asia/Kolkata'));
  return $dt ?: null;
}

/** Sum-of-digits last digit (0–9) for 1–3 digit strings. */
function sd_from_panna($s){
  $sum = 0;
  for ($i=0,$n=strlen($s); $i<$n; $i++){
    $sum += (int)$s[$i];
  }
  return $sum % 10;
}

/**
 * SD pairs (robust):
 * - JSON map: {"0":200,"5":100}
 * - Quoted map variants
 * - Compact tokens like "0 [500]" or "191 [500]" (191→1 via sum-of-digits)
 * - Also supports "0 500", "191-500", "191:500", separated by comma/space/newline
 * Returns [digit(int)=>amount(float), ...]
 */
function parse_sd_pairs($raw){
  $out = [];
  if (!is_string($raw) || $raw === '') return $out;
  $s = trim($raw);

  // 1) JSON-ish: {'0':200, '5':100} or {"0":200}
  if ($s[0] === '{' && substr($s,-1) === '}') {
    $json = preg_replace('/([{\s,])(\')([^\'"]+)(\')(\s*:)/', '$1"$3"$5', $s);
    $json = preg_replace('/:\s*\'([^\']*)\'/', ':"$1"', $json);
    $data = json_decode($json, true);
    if (is_array($data)) {
      foreach ($data as $k => $v) {
        if (preg_match('/^\d$/', (string)$k)) {
          $out[(int)$k] = ($out[(int)$k] ?? 0) + (float)$v;
        }
      }
    }
  }

  // 2) "digit" : "amount" pairs in text (fallback)
  if (!$out && preg_match_all('/"(\d)"\s*:\s*"?(\d+(?:\.\d+)?)"?/', $s, $m, PREG_SET_ORDER)) {
    foreach ($m as $hit) {
      $d = (int)$hit[1];
      $out[$d] = ($out[$d] ?? 0) + (float)$hit[2];
    }
  }

  // 3) Compact tokens like "191 [500]" or "0 [200]"
  if (preg_match_all('/\b(\d{1,3})\s*\[\s*(\d+(?:\.\d+)?)\s*\]/', $s, $m, PREG_SET_ORDER)) {
    foreach ($m as $hit) {
      $k = $hit[1]; $amt = (float)$hit[2];
      $digit = (strlen($k) === 1) ? (int)$k : sd_from_panna($k);
      $out[$digit] = ($out[$digit] ?? 0) + $amt;
    }
  }

  // 4) Loose "a b" or "a-b" or "a:b" tokens (e.g., "0 500", "191-200")
  if (preg_match_all('/\b(\d{1,3})\s*(?:[:-]|\s)\s*(\d+(?:\.\d+)?)\b/', $s, $m2, PREG_SET_ORDER)) {
    foreach ($m2 as $hit) {
      $k = $hit[1]; $amt = (float)$hit[2];
      $digit = (strlen($k) === 1) ? (int)$k : sd_from_panna($k);
      $out[$digit] = ($out[$digit] ?? 0) + $amt;
    }
  }

  ksort($out);
  return $out;
}

/** DP pairs (map): { "110": 70, "112": 50 } -> ["110"=>70,"112"=>50] */
function parse_dp_pairs($raw){
  $out = [];
  if (!is_string($raw) || $raw === '') return $out;
  $s = trim($raw);

  $data = json_decode($s, true);
  if (is_array($data) && !isset($data[0])) {
    foreach ($data as $k=>$v){
      if (preg_match('/^\d{2,3}$/', (string)$k)) {
        $out[(string)$k] = ($out[(string)$k] ?? 0) + (float)$v;
      }
    }
  } elseif (preg_match_all('/"(\d{2,3})"\s*:\s*"?(\d+(?:\.\d+)?)"?/', $s, $m, PREG_SET_ORDER)) {
    foreach ($m as $hit){
      $out[$hit[1]] = ($out[$hit[1]] ?? 0) + (float)$hit[2];
    }
  }

  ksort($out, SORT_NUMERIC);
  return $out;
}


/** SP/TP array-of-objects */
function parse_sp_tp_entries_all($raw){
  $rows=[]; if(!is_string($raw)||$raw==='') return $rows;

  $j=json_decode($raw,true);
  if(is_array($j)){
    // A) array of objects
    if(isset($j[0]) && is_array($j[0])){
      $acc=[];
      foreach($j as $e){
        if(!is_array($e)) continue;
        $num=(string)($e['number']??$e['num']??'');
        if($num==='') continue;
        $amt=(float)($e['amount']??$e['amt']??0);
        $sess=strtoupper((string)($e['session']??'')); // O/OPEN/C/CLOSE
        $sess = ($sess==='C' || $sess==='CLOSE') ? 'CLOSE' : (($sess==='O'||$sess==='OPEN'||$sess==='OP')?'OPEN':null);
        $acc[$num.'|'.($sess?:'-')] = ($acc[$num.'|'.($sess?:'-')]??0) + $amt;
      }
      foreach ($acc as $k => $sum) {
  $parts = array_pad(explode('|', $k, 2), 2, '-');  // ensure indexes 0 and 1 exist
  $n = $parts[0];
  $s = $parts[1];
  $rows[] = ['num' => $n, 'amt' => $sum, 'sess' => ($s === '-' ? null : $s)];
}

      return $rows;
    }
    // B) object/map
    foreach($j as $k=>$v){
      if(preg_match('/^\d{2,3}$/',(string)$k)){
        $rows[]=['num'=>(string)$k,'amt'=>(float)$v,'sess'=>null];
      }
    }
    if($rows) return $rows;
  }

  // C) last resort — textual `"123": 50` fragments
  if (preg_match_all('/"(\d{2,3})"\s*:\s*"?(\d+(?:\.\d+)?)"?/', $raw, $m, PREG_SET_ORDER)) {
  foreach ($m as $h) {
    $n = $h[1] ?? null;
    $a = isset($h[2]) ? (float)$h[2] : 0.0;
    if ($n !== null) {
      $rows[] = ['num' => $n, 'amt' => $a, 'sess' => null];
    }
  }
}

  return $rows;
}

/* --- JD parser that returns per-number amounts --- */
function parse_jd_items($raw, $fallbackTotal = null){
  $out = [];
  if (!is_string($raw) || trim($raw) === '') return [];

  $s = trim(stripslashes($raw));
  $j = json_decode($s, true);

  if (is_array($j)) {
    // a) [{number:"12", amount:50}, ...]
    if (isset($j[0]) && is_array($j[0])) {
      foreach ($j as $e) {
        if (!is_array($e)) continue;
        $n = $e['number'] ?? $e['num'] ?? $e['jd'] ?? $e['jodi'] ?? '';
        $n = preg_replace('/\D+/', '', (string)$n);
        if ($n === '') continue;
        $n   = str_pad($n, 2, '0', STR_PAD_LEFT);
        $amt = (float)($e['amount'] ?? $e['amt'] ?? $e['stake'] ?? 0);
        $out[$n] = ($out[$n] ?? 0) + $amt;
      }
    } else {
      // b) {"12": 100, "34": 50}
      foreach ($j as $k => $v) {
        if (preg_match('/^\d{1,2}$/', (string)$k)) {
          $n = str_pad((string)$k, 2, '0', STR_PAD_LEFT);
          $out[$n] = ($out[$n] ?? 0) + (float)$v;
        }
      }
      // c) ["12","34"]  => split fallback total equally
      if (!$out && array_keys($j) === range(0, count($j)-1)) {
        $nums = [];
        foreach ($j as $v) {
          if (preg_match('/^\d{1,2}$/', (string)$v)) {
            $nums[] = str_pad((string)$v, 2, '0', STR_PAD_LEFT);
          }
        }
        if ($nums) {
          $per = (is_numeric($fallbackTotal) && $fallbackTotal > 0)
                 ? ((float)$fallbackTotal / count($nums)) : 0.0;
          foreach ($nums as $n) $out[$n] = ($out[$n] ?? 0) + $per;
        }
      }
    }
  }

  // d) Text pairs: 12:50 / 12=50 / "12" 50 / 12-50
  if (!$out && preg_match_all('/\b(\d{1,2})\b\D+(\d+(?:\.\d+)?)/', $s, $m, PREG_SET_ORDER)) {
    foreach ($m as $h) {
      $n = str_pad($h[1], 2, '0', STR_PAD_LEFT);
      $out[$n] = ($out[$n] ?? 0) + (float)$h[2];
    }
  }

  // e) Plain list without amounts: "12,34" or "12 34"
  if (!$out) {
    if (preg_match_all('/\b(\d{1,2})\b/', $s, $m)) {
      $nums = array_values(array_unique(array_map(function($n){
        return str_pad($n, 2, '0', STR_PAD_LEFT);
      }, $m[1])));
      if ($nums) {
        $per = (is_numeric($fallbackTotal) && $fallbackTotal > 0)
               ? ((float)$fallbackTotal / count($nums)) : 0.0;
        foreach ($nums as $n) $out[$n] = ($out[$n] ?? 0) + $per;
      }
    }
  }

  $rows = [];
  foreach ($out as $n => $a) $rows[] = ['num' => $n, 'amt' => (float)$a];
  return $rows;
}


/** JD digits — keep leading zero (e.g., "01"). */
function parse_jd_digits($raw){
  $out = [];
  if (!is_string($raw) || $raw==='') return $out;
  $t = trim($raw);

  if (preg_match('/^\d{1,2}$/', $t)) return [str_pad($t,2,'0',STR_PAD_LEFT)];
  if (preg_match('/^\s*\d{1,2}(\s*[, ]\s*\d{1,2})+\s*$/', $t)) {
    foreach (preg_split('/\s*[, ]\s*/', $t) as $n) {
      if ($n==='') continue;
      $out[] = str_pad($n,2,'0',STR_PAD_LEFT);
    }
    return $out;
  }
  $j = json_decode($t, true);
  if (is_array($j)) {
    if (isset($j[0])) {
      foreach ($j as $e) {
        if (!is_array($e)) continue;
        $n = isset($e['number']) ? (string)$e['number'] : (string)($e['num'] ?? '');
        if ($n!=='') $out[] = str_pad($n,2,'0',STR_PAD_LEFT);
      }
      if ($out) return $out;
    }
    foreach ($j as $k=>$v) {
      if (preg_match('/^\d{1,2}$/', (string)$k)) $out[] = str_pad((string)$k,2,'0',STR_PAD_LEFT);
    }
    if ($out) return $out;
  }
  if (preg_match_all('/\b\d{1,2}\b/', $t, $m)) {
    foreach ($m[0] as $n) $out[] = str_pad($n,2,'0',STR_PAD_LEFT);
  }
  return $out;
}

/* ---- SD: robust extractor (handles 0 reliably and 194..203 codes) ---- */
/* ---- SD: ultra-robust extractor (never drops 0) ---- */
function sd_extract_pairs($raw, $stake = null): array {
  if (!is_string($raw) || $raw === '') return [];

  // Normalise: unescape, decode HTML entities, remove NBSPs, fix trailing commas
  $s = trim(stripslashes($raw));
  $s = html_entity_decode($s, ENT_QUOTES | ENT_HTML5, 'UTF-8');       // &quot;0&quot; -> "0"
  $s = preg_replace('/\x{00A0}|\x{2007}|\x{202F}/u', ' ', $s);       // NBSP family -> space
  $s = preg_replace('/,(\s*[}\]])/', '$1', $s);                       // {"0":46,} -> {"0":46}

  $out = [];

  // A) Try real JSON (and a coerced single-quoted variant)
  foreach ([$s, preg_replace('/([{\s,])\'([^\'"]+)\'\s*:/', '$1"$2":', $s)] as $cand) {
    $j = json_decode($cand, true);
    if (is_array($j) && $j) {
      foreach ($j as $k => $v) {
        if (preg_match('/^\d$/', (string)$k)) {
          $d = (int)$k;
          $out[$d] = ($out[$d] ?? 0) + (float)$v;
        }
      }
      if ($out) { ksort($out); return $out; }
    }
  }

  // B) Object-like fragments anywhere: {"0":46}, {0:"46"}, '0':46, 0:46
  if (preg_match_all('/[{,]\s*[\'"]?([0-9])[\'"]?\s*:\s*[\'"]?(\d+(?:\.\d+)?)[\'"]?\s*(?=,|})/u',
                     '{'.$s.'}', $m, PREG_SET_ORDER)) {
    foreach ($m as $h) {
      $d = (int)$h[1];
      $out[$d] = ($out[$d] ?? 0) + (float)$h[2];
    }
  }

  // C) Compact codes 194..203 => digit 0..9 (optional [amount]); falls back to $stake
  if (preg_match_all('/\b(19[4-9]|20[0-3])\b[^\d\[]*(?:\[\s*(\d+(?:\.\d+)?)\s*\])?/u', $s, $m2, PREG_SET_ORDER)) {
    foreach ($m2 as $h) {
      $d   = (int)$h[1] - 194;
      $amt = ($h[2] !== '' ? (float)$h[2] : (float)$stake);
      if ($amt > 0) $out[$d] = ($out[$d] ?? 0) + $amt;
    }
  }

  // D) Plain lists -> assign $stake per digit ([0], [0,3], "0 3", etc.)
  if (!$out) {
    $j = json_decode($s, true);
    if (is_array($j) && array_keys($j) === range(0, count($j)-1)) {
      foreach ($j as $d) if (is_numeric($d) && $d >= 0 && $d <= 9) $out[(int)$d] = (float)$stake;
    } elseif (preg_match('/^\s*\[\s*([0-9])\s*\]\s*$/', $s, $mm)) {
      $out[(int)$mm[1]] = (float)$stake;
    } else {
      foreach (preg_split('/\D+/', $s) as $tok) {
        if ($tok !== '' && ctype_digit($tok)) {
          $d = (int)$tok; if ($d >= 0 && $d <= 9) $out[$d] = (float)$stake;
        }
      }
    }
  }

  // E) Final safety: literal zero in any form => use $stake
  if (!$out && preg_match('/^\s*(?:\[?\s*[\'"]?0[\'"]?\s*\]?)\s*$/u', $s)) {
    $out = [0 => (float)$stake];
  }

  ksort($out);
  return $out;
}



/* ---------- input ---------- */
$body    = read_json_body();
$mobile  = preg_replace('/\D+/', '', (string)($body['mobile'] ?? ''));
$selDate = normalize_date($body['date'] ?? $body['current_date'] ?? $body['currentDate'] ?? '');

if (!$mobile) {
  echo json_encode(['ok'=>false,'error'=>'mobile required'], JSON_UNESCAPED_UNICODE);
  exit;
}

/* ---------- resolve user ---------- */
$user_id = null;
$stmt = $conn->prepare("SELECT user_id FROM users_data WHERE mobile = ? LIMIT 1");
if (!$stmt) json_fatal('Prepare failed: '.$conn->error, 500);
$stmt->bind_param('s', $mobile);
$stmt->execute();
$stmt->bind_result($user_id);
$stmt->fetch();
$stmt->close();

if (!$user_id) {
  echo json_encode(['ok'=>true,'mobile'=>$mobile,'date'=>$selDate,'refresh_interval_sec'=>60,'items'=>[]], JSON_UNESCAPED_UNICODE);
  exit;
}

$items = [];
$seq = 0; 
$now   = new DateTime('now', new DateTimeZone('Asia/Kolkata'));

/* ---------- SD ---------- */
$sql_sd = "
SELECT
  b.id AS bid_id,
  b.game_id, b.game_name, b.bid_details, b.bid_session, b.total_bid_amount,
  DATE(b.created_at) AS cdate,
  DATE_FORMAT(b.created_at,'%d/%m/%Y') AS cdate_display,
  DATE_FORMAT(b.created_at,'%W') AS cday_display,
  TIME_FORMAT(COALESCE(g.open_time, r.open_time),  '%h:%i %p') AS open12,
  TIME_FORMAT(COALESCE(g.close_time, r.close_time), '%h:%i %p') AS close12,
  TIME_FORMAT(COALESCE(g.open_time, r.open_time),  '%H:%i:%s') AS open_hms,
  TIME_FORMAT(COALESCE(g.close_time, r.close_time), '%H:%i:%s') AS close_hms,
  COALESCE(r.winning_open_time_sd,  NULL) AS win_o,
  COALESCE(r.winning_close_time_sd, NULL) AS win_c,
  g.id AS game_no,
  UNIX_TIMESTAMP(CONVERT_TZ(b.created_at, '+00:00', '+05:30')) AS ts
FROM user_bids b
LEFT JOIN game56 g ON g.id = b.game_id
LEFT JOIN sd_rewards r ON r.game_id = b.game_id AND r.reward_date = ?
WHERE b.user_id = ?
  AND DATE(CONVERT_TZ(b.created_at, '+00:00', '+05:30')) = ?
ORDER BY b.created_at DESC, b.id DESC";
$stmt = $conn->prepare($sql_sd);
if (!$stmt) json_fatal('Prepare failed: '.$conn->error, 500);
$stmt->bind_param('sis', $selDate, $user_id, $selDate);
$stmt->execute();
$stmt->store_result();
$stmt->bind_result(
  $bid_id, $game_id, $game_name, $bid_details, $bid_session, $total_bid_amount,
  $cdate, $cdate_display, $cday_display,
  $open12, $close12, $open_hms, $close_hms,
  $win_o, $win_c, $game_no, $ts
);

while ($stmt->fetch()) {
  // 1) Parse all normal formats first — identical to trs.php
  $pairs = sd_extract_pairs((string)$bid_details, (float)$total_bid_amount);

  // 2) FINAL RESCUE for “only 0” bids (glitches like stray quotes/commas/NBSP)
  if (!$pairs) {
    $bd2 = trim(stripslashes((string)$bid_details));
    $bd2 = preg_replace('/,(\s*[}\]])/', '$1', $bd2);                 // {"0":46,} -> {"0":46}
    $bd2 = preg_replace('/\x{00A0}|\x{2007}|\x{202F}/u', ' ', $bd2);   // NBSPs

    // Strict: {"0":46} | {'0':46} | {0:46} | "0":46 | '0':"46"
    if (preg_match('/^\s*\{\s*["\']?0["\']?\s*:\s*["\']?(\d+(?:\.\d+)?)["\']?\s*\}\s*$/u', $bd2, $m)) {
      $pairs = [0 => (float)$m[1]];
    }

    // Super-forgiving: the string is basically just a zero (0, "0", '0', [0])
    if (!$pairs && preg_match('/^\s*(?:\[?\s*["\']?0["\']?\s*\]?)\s*$/u', $bd2)) {
      $pairs = [0 => (float)$total_bid_amount];
    }

    // Last-chance: there is a “0” key and a single number near it
    if (!$pairs && preg_match('/["\']?\s*0\s*["\']?\s*[:=]\s*["\']?(\d+(?:\.\d+)?)["\']?/u', $bd2, $m)) {
      $pairs = [0 => (float)$m[1]];
    }
  }

  if (!$pairs) continue;

  $openDT  = dt_from_hms_for_date($open_hms ?: null,  $selDate);
  $closeDT = dt_from_hms_for_date($close_hms ?: null, $selDate);
  $winO = ($win_o !== null && $win_o !== '') ? (int)$win_o : null;
  $winC = ($win_c !== null && $win_c !== '') ? (int)$win_c : null;

  foreach ($pairs as $digit => $amt) {
    $sess = strtoupper((string)$bid_session) === 'CLOSE' ? 'CLOSE' : 'OPEN';
    $timeDisp = ($sess === 'OPEN') ? ($open12 ?? '') : ($close12 ?? '');

   $resO = ($winO !== null) ? (((int)$digit === $winO) ? 'You win' : 'Better luck') : 'Pending';
$resC = ($winC !== null) ? (((int)$digit === $winC) ? 'You win' : 'Better luck') : 'Pending';

$payout = 0.0;
if ($sess === 'OPEN'  && $winO !== null && (int)$digit === $winO)  $payout = (float)$amt * $PAYOUT['SD'];
if ($sess === 'CLOSE' && $winC !== null && (int)$digit === $winC)  $payout = (float)$amt * $PAYOUT['SD'];
// AUTO-CREDIT on win (idempotent)
if ($payout > 0) {
  require_once __DIR__ . '/lib/wallet_helper.php';
  wallet_win_credit($user_id, (float)$payout, 'SD', (int)$bid_id, (string)$digit, $sess);
}

    $items[] = [
      '_ts'=>(int)$ts, '_pk'=>(int)$bid_id, '_sub'=>(int)$digit, '_seq'=>++$seq, 'badge'=>'SD',
      'game'=>(string)$game_name, 'game_no'=>(int)($game_no ?? 0),
      'date_disp'=>(string)$cdate_display.' ('.(string)$cday_display.')',
      'date'=>(string)$cdate,
      'bid_id'=>'SD_'.(int)$bid_id,
      'digits'=>(int)$digit,
      'amount'=>round((float)$amt, 2),
      'time'=>(string)$timeDisp,
      'session'=>$sess,
      'win_open'=>($winO===null?'':(string)$winO),
      'win_close'=>($winC===null?'':(string)$winC),
      'result_open'=>$resO, 'result_close'=>$resC,
      'result'=>($sess==='OPEN' ? $resO : $resC),
      'win_amount' => round($payout, 2),

    ];
  }
}

$stmt->close();

/* ---------- DP ---------- */
$sql_dp = "
SELECT
  b.id AS bid_id,
  b.game_id, b.game_name, b.bid_details, b.bid_session,
  DATE(b.created_at) AS cdate,
  DATE_FORMAT(b.created_at,'%d/%m/%Y') AS cdate_display,
  DATE_FORMAT(b.created_at,'%W') AS cday_display,
  TIME_FORMAT(g.open_time,  '%h:%i %p') AS open12,
  TIME_FORMAT(g.close_time, '%h:%i %p') AS close12,
  TIME_FORMAT(g.open_time,  '%H:%i:%s') AS open_hms,
  TIME_FORMAT(g.close_time, '%H:%i:%s') AS close_hms,
  COALESCE(r.winning_open_time_dp,  b.winning_open_time_dp)  AS win_o,
  COALESCE(r.winning_close_time_dp, b.winning_close_time_dp) AS win_c,
  g.id AS game_no,
  UNIX_TIMESTAMP(CONVERT_TZ(b.created_at, '+00:00', '+05:30')) AS ts
FROM user_bids_dp b
LEFT JOIN game56 g     ON g.id = b.game_id
LEFT JOIN dp_rewards r ON r.game_id = b.game_id AND r.reward_date = ?
WHERE b.user_id = ?
  AND DATE(CONVERT_TZ(b.created_at, '+00:00', '+05:30')) = ?
ORDER BY b.created_at DESC, b.id DESC";
$stmt = $conn->prepare($sql_dp);
if (!$stmt) json_fatal('Prepare failed: '.$conn->error, 500);
$stmt->bind_param('sis', $selDate, $user_id, $selDate);
$stmt->execute();
$stmt->store_result();
$stmt->bind_result(
  $bid_id, $game_id, $game_name, $bid_details, $bid_session,
  $cdate, $cdate_display, $cday_display,
  $open12, $close12, $open_hms, $close_hms,
  $win_o, $win_c, $game_no, $ts
);
while ($stmt->fetch()) {
  $pairs = parse_dp_pairs($bid_details);
  if (!$pairs) continue;

  $openDT  = dt_from_hms_for_date($open_hms ?: null,  $selDate);
  $closeDT = dt_from_hms_for_date($close_hms ?: null, $selDate);
  $winO = ($win_o !== null && $win_o !== '') ? (string)$win_o : null;
  $winC = ($win_c !== null && $win_c !== '') ? (string)$win_c : null;

  foreach ($pairs as $panna => $amt) {
    $sess = strtoupper((string)$bid_session) === 'CLOSE' ? 'CLOSE' : 'OPEN';
    $timeDisp = ($sess === 'OPEN') ? ($open12 ?? '') : ($close12 ?? '');
    $resO = ($winO !== null) ? (((string)$panna === $winO) ? 'You win' : 'Better luck') : 'Pending';
$resC = ($winC !== null) ? (((string)$panna === $winC) ? 'You win' : 'Better luck') : 'Pending';

$payout = 0.0;
if ($sess === 'OPEN'  && $winO !== null && (string)$panna === $winO)  $payout = (float)$amt * $PAYOUT['DP'];
if ($sess === 'CLOSE' && $winC !== null && (string)$panna === $winC)  $payout = (float)$amt * $PAYOUT['DP'];
if ($payout > 0) {
  wallet_win_credit($user_id, (float)$payout, 'DP', (int)$bid_id, (string)$panna, $sess);
}

    $items[] = [
      '_ts'=>(int)$ts, '_pk'=>(int)$bid_id, '_sub'=>intval($panna), '_seq'=>++$seq, 'badge'=>'DP',
      'game'=>(string)$game_name, 'game_no'=>(int)($game_no ?? 0),
      'date_disp'=>(string)$cdate_display.' ('.(string)$cday_display.')',
      'date'=>(string)$cdate,
      'bid_id'=>'DP_'.(int)$bid_id.'_'.(string)$panna,
      'digits'=>(string)$panna,
      'amount'=>round((float)$amt, 2),
      'time'=>(string)$timeDisp,
      'session'=>$sess,
      'win_open'=>($winO===null?'':$winO),
      'win_close'=>($winC===null?'':$winC),
      'result_open'=>$resO, 'result_close'=>$resC,
      'result'=>($sess==='OPEN' ? $resO : $resC),
      'win_amount' => round($payout, 2),

    ];
  }
}
$stmt->close();

/* ---------- SP ---------- */
$sql_sp = "
SELECT
  b.id AS bid_id,
  b.game_id, b.game_name, b.bid_details, b.bid_session,
  DATE(b.created_at) AS cdate,
  DATE_FORMAT(b.created_at,'%d/%m/%Y') AS cdate_display,
  DATE_FORMAT(b.created_at,'%W') AS cday_display,
  TIME_FORMAT(g.open_time,  '%h:%i %p') AS open12,
  TIME_FORMAT(g.close_time, '%h:%i %p') AS close12,
  TIME_FORMAT(g.open_time,  '%H:%i:%s') AS open_hms,
  TIME_FORMAT(g.close_time, '%H:%i:%s') AS close_hms,
  COALESCE(r.winning_open_time_sp,  NULL) AS win_o,
  COALESCE(r.winning_close_time_sp, NULL) AS win_c,
  g.id AS game_no,
  UNIX_TIMESTAMP(CONVERT_TZ(b.created_at, '+00:00', '+05:30')) AS ts
FROM user_bids_sp b
LEFT JOIN game56 g     ON g.id = b.game_id
LEFT JOIN sp_rewards r ON r.game_id = b.game_id AND r.reward_date = ?
WHERE b.user_id = ?
  AND DATE(CONVERT_TZ(b.created_at, '+00:00', '+05:30')) = ?
ORDER BY b.created_at DESC, b.id DESC";
$stmt = $conn->prepare($sql_sp);
if (!$stmt) json_fatal('Prepare failed: '.$conn->error, 500);
$stmt->bind_param('sis', $selDate, $user_id, $selDate);
$stmt->execute();
$stmt->store_result();
$stmt->bind_result(
  $bid_id, $game_id, $game_name, $bid_details, $bid_session,
  $cdate, $cdate_display, $cday_display,
  $open12, $close12, $open_hms, $close_hms,
  $win_o, $win_c, $game_no, $ts
);
while ($stmt->fetch()) {
  $entries = parse_sp_tp_entries_all($bid_details);
  if (!$entries) continue;

  $openDT  = dt_from_hms_for_date($open_hms ?: null,  $selDate);
  $closeDT = dt_from_hms_for_date($close_hms ?: null, $selDate);
  $winO = ($win_o !== null && $win_o !== '') ? (string)$win_o : null;
  $winC = ($win_c !== null && $win_c !== '') ? (string)$win_c : null;

  foreach ($entries as $e) {
    $num   = (string)$e['num'];
    $amt   = (float)$e['amt'];
    $sessE = $e['sess'] ?: strtoupper((string)$bid_session);
    $sess  = ($sessE==='CLOSE') ? 'CLOSE' : 'OPEN';
    $timeDisp = ($sess==='OPEN') ? ($open12 ?? '') : ($close12 ?? '');

    $resO = ($winO !== null) ? (($num === $winO) ? 'You win' : 'Better luck') : 'Pending';
$resC = ($winC !== null) ? (($num === $winC) ? 'You win' : 'Better luck') : 'Pending';

$payout = 0.0;
if ($sess === 'OPEN'  && $winO !== null && $num === $winO)  $payout = (float)$amt * $PAYOUT['SP'];
if ($sess === 'CLOSE' && $winC !== null && $num === $winC)  $payout = (float)$amt * $PAYOUT['SP'];
if ($payout > 0) {
  wallet_win_credit($user_id, (float)$payout, 'SP', (int)$bid_id, (string)$num, $sess);
}
   $items[] = [
      '_ts'=>(int)$ts, '_pk'=>(int)$bid_id, '_sub'=>intval($num), '_seq'=>++$seq, 'badge'=>'SP',
      'game'=>(string)$game_name, 'game_no'=>(int)($game_no ?? 0),
      'date_disp'=>(string)$cdate_display.' ('.(string)$cday_display.')',
      'date'=>(string)$cdate,
      'bid_id'=>'SP_'.(int)$bid_id.'_'. $num,
      'digits'=>(string)$num,
      'amount'=>round($amt, 2),
      'time'=>(string)$timeDisp,
      'session'=>$sess,
      'win_open'=>($winO===null?'':$winO),
      'win_close'=>($winC===null?'':$winC),
      'result_open'=>$resO, 'result_close'=>$resC,
      'result'=>($sess==='OPEN' ? $resO : $resC),
      'win_amount' => round($payout, 2),

    ];
  }
}
$stmt->close();

/* ---------- TP ---------- */
$sql_tp = "
SELECT
  b.id AS bid_id,
  b.game_id, b.game_name, b.bid_details, b.bid_session,
  DATE(b.created_at) AS cdate,
  DATE_FORMAT(b.created_at,'%d/%m/%Y') AS cdate_display,
  DATE_FORMAT(b.created_at,'%W') AS cday_display,
  TIME_FORMAT(g.open_time,  '%h:%i %p') AS open12,
  TIME_FORMAT(g.close_time, '%h:%i %p') AS close12,
  TIME_FORMAT(g.open_time,  '%H:%i:%s') AS open_hms,
  TIME_FORMAT(g.close_time, '%H:%i:%s') AS close_hms,
  COALESCE(r.winning_open_time_tp,  NULL) AS win_o,
  COALESCE(r.winning_close_time_tp, NULL) AS win_c,
  g.id AS game_no,
  UNIX_TIMESTAMP(CONVERT_TZ(b.created_at, '+00:00', '+05:30')) AS ts
FROM user_bids_tp b
LEFT JOIN game56 g     ON g.id = b.game_id
LEFT JOIN tp_rewards r ON r.game_id = b.game_id AND r.reward_date = ?
WHERE b.user_id = ?
  AND DATE(CONVERT_TZ(b.created_at, '+00:00', '+05:30')) = ?
ORDER BY b.created_at DESC, b.id DESC";
$stmt = $conn->prepare($sql_tp);
if (!$stmt) json_fatal('Prepare failed: '.$conn->error, 500);
$stmt->bind_param('sis', $selDate, $user_id, $selDate);
$stmt->execute();
$stmt->store_result();
$stmt->bind_result(
  $bid_id, $game_id, $game_name, $bid_details, $bid_session,
  $cdate, $cdate_display, $cday_display,
  $open12, $close12, $open_hms, $close_hms,
  $win_o, $win_c, $game_no, $ts
);
while ($stmt->fetch()) {
  $entries = parse_sp_tp_entries_all($bid_details);
  if (!$entries) continue;

  $openDT  = dt_from_hms_for_date($open_hms ?: null,  $selDate);
  $closeDT = dt_from_hms_for_date($close_hms ?: null, $selDate);
  $winO = ($win_o !== null && $win_o !== '') ? (string)$win_o : null;
  $winC = ($win_c !== null && $win_c !== '') ? (string)$win_c : null;

  foreach ($entries as $e) {
    $num   = (string)$e['num'];
    $amt   = (float)$e['amt'];
    $sessE = $e['sess'] ?: strtoupper((string)$bid_session);
    $sess  = ($sessE==='CLOSE') ? 'CLOSE' : 'OPEN';
    $timeDisp = ($sess==='OPEN') ? ($open12 ?? '') : ($close12 ?? '');

    $resO = ($winO !== null) ? (($num === $winO) ? 'You win' : 'Better luck') : 'Pending';
$resC = ($winC !== null) ? (($num === $winC) ? 'You win' : 'Better luck') : 'Pending';

$payout = 0.0;
if ($sess === 'OPEN'  && $winO !== null && $num === $winO)  $payout = (float)$amt * $PAYOUT['TP'];
if ($sess === 'CLOSE' && $winC !== null && $num === $winC)  $payout = (float)$amt * $PAYOUT['TP'];
if ($payout > 0) {
  wallet_win_credit($user_id, (float)$payout, 'TP', (int)$bid_id, (string)$num, $sess);
}


    $items[] = [
      '_ts'=>(int)$ts, '_pk'=>(int)$bid_id, '_sub'=>intval($num), '_seq'=>++$seq, 'badge'=>'TP',
      'game'=>(string)$game_name, 'game_no'=>(int)($game_no ?? 0),
      'date_disp'=>(string)$cdate_display.' ('.(string)$cday_display.')',
      'date'=>(string)$cdate,
      'bid_id'=>'TP_'.(int)$bid_id.'_'. $num,
      'digits'=>(string)$num,
      'amount'=>round($amt, 2),
      'time'=>(string)$timeDisp,
      'session'=>$sess,
      'win_open'=>($winO===null?'':$winO),
      'win_close'=>($winC===null?'':$winC),
      'result_open'=>$resO, 'result_close'=>$resC,
      'result'=>($sess==='OPEN' ? $resO : $resC),
      'win_amount' => round($payout, 2),

    ];
  }
}
$stmt->close();

/* ---------- JD (Jodi) ---------- */
$sql_jd = "
SELECT
  b.id AS bid_id,
  b.game_id, b.game_name, b.bid_details, b.total_bid_amount, b.bid_session,
  DATE(b.created_at) AS cdate,
  DATE_FORMAT(b.created_at,'%d/%m/%Y') AS cdate_display,
  DATE_FORMAT(b.created_at,'%W') AS cday_display,
  TIME_FORMAT(COALESCE(g.open_time, r.open_time),  '%h:%i %p') AS open12,
  TIME_FORMAT(COALESCE(g.close_time, r.close_time), '%h:%i %p') AS close12,
  TIME_FORMAT(COALESCE(g.open_time, r.open_time),  '%H:%i:%s') AS open_hms,
  TIME_FORMAT(COALESCE(g.close_time, r.close_time), '%H:%i:%s') AS close_hms,
  COALESCE(jr.win_open_jd,  r.winning_open_time_jd)  AS win_o,
  COALESCE(jr.win_close_jd, r.winning_close_time_jd) AS win_c,
  jr.res_open  AS res_o,
  jr.res_close AS res_c,
  g.id AS game_no,
  UNIX_TIMESTAMP(CONVERT_TZ(b.created_at, '+00:00', '+05:30')) AS ts
FROM user_bids_jd b
LEFT JOIN game56 g     ON g.id = b.game_id
LEFT JOIN jd_rewards r ON r.game_id = b.game_id AND r.reward_date = ?
LEFT JOIN (
    SELECT r2.id AS rrid,
           SUBSTRING_INDEX(GROUP_CONCAT(j2.win_open_jd  ORDER BY j2.created_at DESC), ',', 1) AS win_open_jd,
           SUBSTRING_INDEX(GROUP_CONCAT(j2.win_close_jd ORDER BY j2.created_at DESC), ',', 1) AS win_close_jd,
           SUBSTRING_INDEX(GROUP_CONCAT(j2.res_open     ORDER BY j2.created_at DESC), ',', 1) AS res_open,
           SUBSTRING_INDEX(GROUP_CONCAT(j2.res_close    ORDER BY j2.created_at DESC), ',', 1) AS res_close
    FROM jd_rewards r2
    LEFT JOIN jd_result j2 ON j2.reward_id = r2.id
    WHERE r2.reward_date = ?
    GROUP BY r2.id
) jr ON jr.rrid = r.id
WHERE b.user_id = ?
  AND DATE(CONVERT_TZ(b.created_at, '+00:00', '+05:30')) = ?
ORDER BY b.created_at DESC, b.id DESC";
$stmt = $conn->prepare($sql_jd);
if (!$stmt) json_fatal('Prepare failed: '.$conn->error, 500);
$stmt->bind_param('ssis', $selDate, $selDate, $user_id, $selDate);
$stmt->execute();
$stmt->store_result();
$stmt->bind_result(
  $bid_id, $game_id, $game_name, $bid_details, $total_bid_amount, $bid_session,
  $cdate, $cdate_display, $cday_display,
  $open12, $close12, $open_hms, $close_hms,
  $win_o, $win_c, $res_o, $res_c, $game_no, $ts
);
while ($stmt->fetch()) {
  // NEW: per-number amounts
  $itemsJD = parse_jd_items((string)$bid_details, (float)$total_bid_amount);
  if (!$itemsJD) continue;

  $openDT  = dt_from_hms_for_date($open_hms ?: null,  $selDate);
  $closeDT = dt_from_hms_for_date($close_hms ?: null, $selDate);

  $winO = ($win_o !== null && $win_o !== '') ? str_pad((string)$win_o, 2, '0', STR_PAD_LEFT) : null;
  $winC = ($win_c !== null && $win_c !== '') ? str_pad((string)$win_c, 2, '0', STR_PAD_LEFT) : null;

  foreach ($itemsJD as $e) {
    $num = (string)$e['num'];          // '01', '12', ...
    $amt = (float)$e['amt'];           // per-number amount

    $sess     = strtoupper((string)$bid_session) === 'CLOSE' ? 'CLOSE' : 'OPEN';
    $timeDisp = ($sess === 'OPEN') ? ($open12 ?? '') : ($close12 ?? '');

    // Prefer explicit text if present; else compute from winners
    $resO = ($res_o !== null && $res_o !== '') ? (string)$res_o
           : (($winO !== null) ? (($num === $winO) ? 'You win' : 'Better luck') : 'Pending');

    $resC = ($res_c !== null && $res_c !== '') ? (string)$res_c
           : (($winC !== null) ? (($num === $winC) ? 'You win' : 'Better luck') : 'Pending');

$payout = 0.0;
if ($sess === 'OPEN'  && $winO !== null && $num === $winO)  $payout = (float)$amt * $PAYOUT['JD'];
if ($sess === 'CLOSE' && $winC !== null && $num === $winC)  $payout = (float)$amt * $PAYOUT['JD'];
if ($payout > 0) {
  require_once __DIR__ . '/lib/wallet_helper.php';
  wallet_win_credit($user_id, (float)$payout, 'JD', (int)$bid_id, (string)$num, $sess);
}


    $items[] = [
      '_ts'=>(int)$ts, '_pk'=>(int)$bid_id, '_sub'=>intval($num), '_seq'=>++$seq, 'badge'=>'JD',
      'game'=>(string)$game_name, 'game_no'=>(int)($game_no ?? 0),
      'date_disp'=>(string)$cdate_display.' ('.(string)$cday_display.')',
      'date'=>(string)$cdate,
      'bid_id'=>'JD_'.(int)$bid_id.'_'.$num,
      'digits'=>$num,
      'amount'=>round($amt, 2),              // <-- per-number amount
      'time'=>$timeDisp,
      'session'=>$sess,
      'win_open'=>($winO===null?'':$winO),
      'win_close'=>($winC===null?'':$winC),
      'result_open'=>$resO, 'result_close'=>$resC,
      'result'=>($sess==='OPEN' ? $resO : $resC),
      'win_amount' => round($payout, 2),

    ];
  }
}

$stmt->close();

/* ---------- HS ---------- */
$sql_hs = "
SELECT
  b.id AS bid_id,
  b.game_id, COALESCE(b.game_name, g.game_name) AS game_name,
  b.bid_open_hs AS bid_od,
  b.bid_close_hs AS bid_cp,
  b.total_bid_amount, b.bid_session,
  DATE(b.created_at) AS cdate,
  DATE_FORMAT(b.created_at,'%d/%m/%Y') AS cdate_display,
  DATE_FORMAT(b.created_at,'%W')        AS cday_display,
  TIME_FORMAT(g.open_time,  '%h:%i %p') AS open12,
  TIME_FORMAT(g.close_time, '%h:%i %p') AS close12,
  TIME_FORMAT(g.open_time,  '%H:%i:%s') AS open_hms,
  TIME_FORMAT(g.close_time, '%H:%i:%s') AS close_hms,
  r.winning_open_time_od_hs  AS w_open_od,
  r.winning_open_time_cp_hs  AS w_open_cp,
  r.winning_close_time_od_hs AS w_close_od,
  r.winning_close_time_cp_hs AS w_close_cp,
  g.id AS game_no,
  UNIX_TIMESTAMP(CONVERT_TZ(b.created_at, '+00:00', '+05:30')) AS ts
FROM user_bids_hs b
LEFT JOIN game56 g     ON g.id = b.game_id
LEFT JOIN hs_rewards r ON r.game_id = b.game_id AND r.reward_date = ?
WHERE b.user_id = ?
  AND DATE(CONVERT_TZ(b.created_at, '+00:00', '+05:30')) = ?
ORDER BY b.created_at DESC, b.id DESC";
$stmt = $conn->prepare($sql_hs);
if (!$stmt) json_fatal('Prepare failed: '.$conn->error, 500);
$stmt->bind_param('sis', $selDate, $user_id, $selDate);
$stmt->execute();
$stmt->store_result();
$stmt->bind_result(
  $bid_id, $game_id, $game_name, $bid_od, $bid_cp, $hs_total_amt, $bid_session,
  $cdate, $cdate_display, $cday_display,
  $open12, $close12, $open_hms, $close_hms,
  $w_open_od, $w_open_cp, $w_close_od, $w_close_cp,
  $game_no, $ts
);
while ($stmt->fetch()) {
  $openDT  = dt_from_hms_for_date($open_hms ?: null,  $selDate);
  $closeDT = dt_from_hms_for_date($close_hms ?: null, $selDate);

  $sess = strtoupper((string)$bid_session) === 'CLOSE' ? 'CLOSE' : 'OPEN';
  $timeDisp = ($sess === 'OPEN') ? ($open12 ?? '') : ($close12 ?? '');

  $resOpen  = (($w_open_od !== null && $w_open_od !== '') && ($w_open_cp !== null && $w_open_cp !== ''))
            ? (( (string)$bid_od === (string)$w_open_od && (string)$bid_cp === (string)$w_open_cp ) ? 'You win' : 'Better luck')
            : 'Pending';

$resClose = (($w_close_od !== null && $w_close_od !== '') && ($w_close_cp !== null && $w_close_cp !== ''))
            ? (( (string)$bid_od === (string)$w_close_od && (string)$bid_cp === (string)$w_close_cp ) ? 'You win' : 'Better luck')
            : 'Pending';

$payout = 0.0;
if ($sess === 'OPEN'  && $resOpen  === 'You win') $payout = (float)$hs_total_amt * $PAYOUT['HS'];
if ($sess === 'CLOSE' && $resClose === 'You win') $payout = (float)$hs_total_amt * $PAYOUT['HS'];
// Auto-credit HS (idempotent)
if ($payout > 0) {
  $sub = (string)$bid_od . '|' . (string)$bid_cp;   // make the win ref unique per pair
  wallet_win_credit($user_id, (float)$payout, 'HS', (int)$bid_id, $sub, $sess);
}


    $items[] = [
    '_ts'=>(int)$ts, '_pk'=>(int)$bid_id, '_sub'=>intval($bid_od), '_seq'=>++$seq, 'badge'=>'HS',
    'game'=>(string)$game_name, 'game_no'=>(int)($game_no ?? 0),
    'date_disp'=>(string)$cdate_display.' ('.(string)$cday_display.')',
    'date'=>(string)$cdate,
    'bid_id'=>'HS_'.(int)$bid_id,

    // keep raw parts
    'digits'=>(string)$bid_od,          // (left part; kept for backward-compat)
    'bid_cp'=>(string)$bid_cp,          // (right part; already present)
    // add display-friendly combined value for the app card
    'digits_display'=>(string)$bid_od.' | '.(string)$bid_cp,

    'amount'=>round((float)$hs_total_amt, 2),
    'time'=>(string)$timeDisp,
    'session'=>$sess,

    // keep existing strings, but also expose split values for UI if needed
    'win_open'=> (($w_open_od!==null && $w_open_od!=='') || ($w_open_cp!==null && $w_open_cp!=='')) ? ('OD: '.(string)($w_open_od?:'—').' | CP: '.(string)($w_open_cp?:'—')) : '',
    'win_close'=>(($w_close_od!==null && $w_close_od!=='') || ($w_close_cp!==null && $w_close_cp!=='')) ? ('OD: '.(string)($w_close_od?:'—').' | CP: '.(string)($w_close_cp?:'—')) : '',
    'win_open_od'=> (string)($w_open_od ?? ''),   // NEW
    'win_open_cp'=> (string)($w_open_cp ?? ''),   // NEW
    'win_close_od'=> (string)($w_close_od ?? ''), // NEW
    'win_close_cp'=> (string)($w_close_cp ?? ''), // NEW

    'result_open'=>$resOpen, 'result_close'=>$resClose,
    'result'=>($sess==='OPEN' ? $resOpen : $resClose),
    'win_amount' => round($payout, 2),

  ];

}
$stmt->close();

/* ---------- FS (Full Sangam) — same shape as HS ---------- */
/* ---------- FS (Full Sangam) — match tables exactly ---------- */
$sql_fs = "
SELECT
  b.id AS bid_id,
  g.game_name AS game_name,
  b.bid_open_fs  AS bid_op,
  b.bid_close_fs AS bid_cp,
  b.total_bid_amount AS bid_amount,
  b.bid_session,
  DATE(b.created_at)                          AS cdate,
  DATE_FORMAT(b.created_at,'%d/%m/%Y')        AS cdate_display,
  DATE_FORMAT(b.created_at,'%W')              AS cday_display,
  TIME_FORMAT(g.open_time,  '%h:%i %p')       AS open12,
  TIME_FORMAT(g.close_time, '%h:%i %p')       AS close12,
  TIME_FORMAT(g.open_time,  '%H:%i:%s')       AS open_hms,
  TIME_FORMAT(g.close_time, '%H:%i:%s')       AS close_hms,
  r.winning_open_op_fs   AS w_open_op,
  r.winning_open_cp_fs   AS w_open_cp,
  r.winning_close_op_fs  AS w_close_op,
  r.winning_close_cp_fs  AS w_close_cp,
  g.id AS game_no,
  UNIX_TIMESTAMP(CONVERT_TZ(b.created_at, '+00:00', '+05:30')) AS ts
FROM user_bids_fs b
LEFT JOIN game56    g ON g.id = b.game_id
LEFT JOIN fs_rewards r ON r.game_id = b.game_id AND r.reward_date = ?
WHERE b.user_id = ?
  AND DATE(CONVERT_TZ(b.created_at, '+00:00', '+05:30')) = ?
ORDER BY b.created_at DESC, b.id DESC";

$stmt = $conn->prepare($sql_fs);
if (!$stmt) json_fatal('Prepare failed: '.$conn->error, 500);
$stmt->bind_param('sis', $selDate, $user_id, $selDate);
$stmt->execute();
$stmt->store_result();
$stmt->bind_result(
  $bid_id, $fs_game_name, $bid_op, $bid_cp, $fs_amount, $fs_session,
  $fs_cdate, $fs_cdate_display, $fs_cday_display,
  $fs_open12, $fs_close12, $fs_open_hms, $fs_close_hms,
  $w_open_op, $w_open_cp, $w_close_op, $w_close_cp,
  $fs_game_no, $fs_ts
);

while ($stmt->fetch()) {
  $openDT  = dt_from_hms_for_date($fs_open_hms ?: null, $selDate);
  $closeDT = dt_from_hms_for_date($fs_close_hms ?: null, $selDate);

  // normalize session to OPEN/CLOSE (your enum stores lower-case)
  $sessRaw = strtoupper((string)$fs_session);
  $sess    = ($sessRaw === 'CLOSE') ? 'CLOSE' : 'OPEN';
  $timeDisp = ($sess === 'OPEN') ? ($fs_open12 ?? '') : ($fs_close12 ?? '');

 $resOpen  = (($w_open_op !== null && $w_open_op !== '') && ($w_open_cp !== null && $w_open_cp !== ''))
            ? (( (string)$bid_op === (string)$w_open_op && (string)$bid_cp === (string)$w_open_cp ) ? 'You win' : 'Better luck')
            : 'Pending';

$resClose = (($w_close_op !== null && $w_close_op !== '') && ($w_close_cp !== null && $w_close_cp !== ''))
            ? (( (string)$bid_op === (string)$w_close_op && (string)$bid_cp === (string)$w_close_cp ) ? 'You win' : 'Better luck')
            : 'Pending';

$payout = 0.0;
if ($sess === 'OPEN'  && $resOpen  === 'You win') $payout = (float)$fs_amount * $PAYOUT['FS'];
if ($sess === 'CLOSE' && $resClose === 'You win') $payout = (float)$fs_amount * $PAYOUT['FS'];
// Auto-credit FS (idempotent)
if ($payout > 0) {
  $sub = (string)$bid_op . '|' . (string)$bid_cp;   // unique per pair
  wallet_win_credit($user_id, (float)$payout, 'FS', (int)$bid_id, $sub, $sess);
}


  $items[] = [
    '_ts'      => (int)$fs_ts,
    '_pk'      => (int)$bid_id,
    '_sub'     => (int)$bid_op,
    '_seq'=>++$seq,
    'badge'    => 'FS',
    'game'     => (string)$fs_game_name,
    'game_no'  => (int)($fs_game_no ?? 0),
    'date_disp'=> (string)$fs_cdate_display.' ('.(string)$fs_cday_display.')',
    'date'     => (string)$fs_cdate,
    'bid_id'   => 'FS_'.(int)$bid_id,

    // left/right parts + a friendly combined display
    'digits'         => (string)$bid_op,                 // open pana
    'bid_cp'         => (string)$bid_cp,                 // close pana
    'digits_display' => (string)$bid_op.' | '.(string)$bid_cp,

    'amount'  => round((float)$fs_amount, 2),
    'time'    => (string)$timeDisp,
    'session' => $sess,

    // winners (split and combined strings, like HS block)
    'win_open'     => (($w_open_op!==null && $w_open_op!=='') || ($w_open_cp!==null && $w_open_cp!=='')) ? ('OP: '.(string)($w_open_op?:'—').' | CP: '.(string)($w_open_cp?:'—')) : '',
    'win_close'    => (($w_close_op!==null && $w_close_op!=='') || ($w_close_cp!==null && $w_close_cp!=='')) ? ('OP: '.(string)($w_close_op?:'—').' | CP: '.(string)($w_close_cp?:'—')) : '',
    'win_open_op'  => (string)($w_open_op  ?? ''),
    'win_open_cp'  => (string)($w_open_cp  ?? ''),
    'win_close_op' => (string)($w_close_op ?? ''),
    'win_close_cp' => (string)($w_close_cp ?? ''),

    'result_open'  => $resOpen,
    'result_close' => $resClose,
    'result'       => ($sess === 'OPEN' ? $resOpen : $resClose),
    'win_amount' => round($payout, 2),

  ];
}
$stmt->close();


/* ---------- sort combined items by newest first across ALL types ---------- */
usort($items, function($a, $b) {
  // 1) Newest timestamp (IST) first
  $c = ($b['_ts'] ?? 0) <=> ($a['_ts'] ?? 0);
  if ($c !== 0) return $c;

  // 2) If same second, newer DB row first
  $c = ($b['_pk'] ?? 0) <=> ($a['_pk'] ?? 0);
  if ($c !== 0) return $c;

  // 3) If still tied (e.g., multiple sub-entries from same row), preserve later insertion first
  return ($b['_seq'] ?? 0) <=> ($a['_seq'] ?? 0);
});


foreach ($items as &$it) { unset($it['_ts'], $it['_pk'], $it['_seq'], $it['_sub']); }


/* ---------- output ---------- */
if (ob_get_length()) { ob_end_clean(); }
echo json_encode([
  'ok'                    => true,
  'mobile'                => $mobile,
  'date'                  => $selDate,
  'refresh_interval_sec'  => 60,
  'count'                 => count($items),
  'items'                 => $items
], JSON_UNESCAPED_UNICODE);
