<?php
// bid_parsers.php
//
// Helpers to parse bid_details for SD/DP/SP/TP/JD in multiple shapes.
//
// Supported shapes (anywhere details JSON is expected):
//  1) JSON object:            {"7":50,"3":20}   or {"123":40,"456":60}
//  2) JSON array of objects:  [{"digit":"7","amount":50},{"number":"3","amount":20}]
//  3) JSON array of strings:  ["7","3","5"]  (use total to split evenly where provided)
//  4) Delimited strings:      "7:50,3:20"  or "7=50;3=20" or "7|50 3|20"
//                             "123:40,456:60"
//                             "7,3,5" (use total to split evenly)
// Notes:
//  • SD/JD helpers accept a $totalAmount to split evenly when individual amounts
//    are not present.
//  • DP/SP/TP helpers only return entries when an amount can be determined;
//    otherwise they return [] (your caller will keep the aggregate row).

// -----------------------------
// Internal helpers
// -----------------------------

/** Trim, drop BOM, and return decoded JSON or null */
function _bp_json_decode($raw) {
  if (!is_string($raw) || $raw === '') return null;
  // strip BOM
  $raw = preg_replace('/^\xEF\xBB\xBF/', '', trim($raw));
  // if it looks like JSON try to decode
  if ($raw === '' || (!in_array($raw[0], ['{','['], true))) {
    return null;
  }
  $j = json_decode($raw, true);
  return (json_last_error() === JSON_ERROR_NONE) ? $j : null;
}

/** True if array is associative */
function _bp_is_assoc(array $arr) {
  if ($arr === []) return false;
  return array_keys($arr) !== range(0, count($arr)-1);
}

/** Normalize a numeric string: keep digits only, optionally left-pad */
function _bp_clean_num($s, $padLen = 0) {
  $t = preg_replace('/\D+/', '', (string)$s);
  if ($padLen > 0 && $t !== '') {
    $t = str_pad($t, $padLen, '0', STR_PAD_LEFT);
  }
  return $t;
}

/**
 * Parse "pairs" from a free-form string.
 * Accepts: "A:10,B:20"  "A=10;B=20"  "A|10 B|20"  "A 10, B 20"
 * Returns ['A'=>10.0,'B'=>20.0] or [] if nothing usable.
 */
function _bp_parse_pair_string($s) {
  if (!is_string($s) || trim($s) === '') return [];
  $s   = trim($s);
  // Normalize separators to commas
  $s   = str_replace(["\n","\r",";","|","  "], [",",",",",",":"," "], $s);
  $s   = preg_replace('/\s*,\s*/', ',', $s);
  $out = [];
  foreach (explode(',', $s) as $chunk) {
    $chunk = trim($chunk);
    if ($chunk === '') continue;
    // Accept "num:amt" or "num=amt" or "num amt"
    if (strpos($chunk, ':') !== false) {
      list($k,$v) = array_map('trim', explode(':', $chunk, 2));
    } elseif (strpos($chunk, '=') !== false) {
      list($k,$v) = array_map('trim', explode('=', $chunk, 2));
    } else {
      // try "num amt"
      $parts = preg_split('/\s+/', $chunk);
      if (count($parts) >= 2) {
        $k = $parts[0]; $v = $parts[1];
      } else {
        // just a bare number with no amount -> skip here
        continue;
      }
    }
    $k = trim($k);
    $v = (float)preg_replace('/[^\d.]+/', '', $v);
    if ($k !== '' && $v > 0) $out[$k] = $v;
  }
  return $out;
}

/** Distribute a total evenly across n items; last item gets the remainder */
function _bp_split_even($n, $total) {
  $n = (int)$n;
  $total = (float)$total;
  if ($n <= 0 || $total <= 0) return [];
  $per = floor(($total / $n) * 100) / 100; // 2dp floor to keep within total
  $out = array_fill(0, $n, $per);
  // add remainder to the last item to match total
  $sum = array_sum($out);
  $out[$n-1] = round($out[$n-1] + ($total - $sum), 2);
  return $out;
}

// -----------------------------
// Public helpers used by your API
// -----------------------------

/**
 * SD: returns ['digit'=>amount,...]
 * $detailsJson can be any supported shape; if no per-digit amounts,
 * we split $totalAmount evenly.
 */
function sd_extract_pairs($detailsJson, $totalAmount) {
  $pairs = [];

  // 1) Try JSON
  $j = _bp_json_decode($detailsJson);
  if (is_array($j)) {
    if (_bp_is_assoc($j)) {
      // {"7":50,"3":20}
      foreach ($j as $k => $v) {
        $d = _bp_clean_num($k, 1);
        $a = (float)$v;
        if ($d !== '' && $a > 0) $pairs[$d] = $a;
      }
      if ($pairs) return $pairs;
    } else {
      // Array: could be ["7","3"] or [{"digit":"7","amount":50}, ...]
      $tmp = [];
      foreach ($j as $row) {
        if (is_array($row)) {
          $d = isset($row['digit'])  ? _bp_clean_num($row['digit'], 1)
             : (isset($row['number']) ? _bp_clean_num($row['number'], 1) : '');
          $a = isset($row['amount']) ? (float)$row['amount']
             : (isset($row['amt'])    ? (float)$row['amt'] : 0.0);
          if ($d !== '' && $a > 0) $tmp[$d] = $a;
        } else {
          $d = _bp_clean_num($row, 1);
          if ($d !== '') $tmp[$d] = null; // mark, will split later if needed
        }
      }
      if ($tmp) {
        // if amounts present, use them; else split evenly
        if (count(array_filter($tmp, fn($v) => $v !== null)) > 0) {
          foreach ($tmp as $d => $a) if ($a !== null) $pairs[$d] = (float)$a;
          if ($pairs) return $pairs;
        } else {
          $digits = array_keys($tmp);
          $split  = _bp_split_even(count($digits), (float)$totalAmount);
          foreach ($digits as $i => $d) $pairs[$d] = $split[$i];
          return $pairs;
        }
      }
    }
  }

  // 2) Try free-form string
  $map = _bp_parse_pair_string((string)$detailsJson);
  if ($map) {
    foreach ($map as $k => $v) {
      $d = _bp_clean_num($k, 1);
      if ($d !== '' && $v > 0) $pairs[$d] = (float)$v;
    }
    if ($pairs) return $pairs;
  }

  // 3) Comma-only digits: "7,3,5" -> split evenly
  $onlyDigits = array_filter(
    array_map(fn($x) => _bp_clean_num($x, 1), preg_split('/[,\s]+/', (string)$detailsJson))
  );
  if ($onlyDigits) {
    $onlyDigits = array_values(array_unique($onlyDigits));
    $split      = _bp_split_even(count($onlyDigits), (float)$totalAmount);
    foreach ($onlyDigits as $i => $d) $pairs[$d] = $split[$i];
  }

  return $pairs;
}

/**
 * DP: returns ['123'=>amount,...] when amounts are discoverable.
 * If amounts are not present, return [] so caller keeps aggregate row.
 */
function parse_dp_pairs($detailsJson) {
  $out = [];

  $j = _bp_json_decode($detailsJson);
  if (is_array($j)) {
    if (_bp_is_assoc($j)) {
      foreach ($j as $k => $v) {
        $num = _bp_clean_num($k); // 3-digit typical
        $a   = (float)$v;
        if ($num !== '' && $a > 0) $out[$num] = $a;
      }
      if ($out) return $out;
    } else {
      foreach ($j as $row) {
        if (is_array($row)) {
          $num = isset($row['number']) ? _bp_clean_num($row['number'])
               : (isset($row['num'])    ? _bp_clean_num($row['num']) : '');
          $a   = isset($row['amount']) ? (float)$row['amount']
               : (isset($row['amt'])    ? (float)$row['amt'] : 0.0);
          if ($num !== '' && $a > 0) $out[$num] = $a;
        } else {
          // bare numbers without amounts -> not usable (no total passed)
        }
      }
      if ($out) return $out;
    }
  }

  // pair string like "123:40,456:60"
  $map = _bp_parse_pair_string((string)$detailsJson);
  if ($map) {
    foreach ($map as $k => $v) {
      $num = _bp_clean_num($k);
      if ($num !== '' && $v > 0) $out[$num] = (float)$v;
    }
  }
  return $out;
}

/**
 * SP/TP: returns a list of ['num'=>string,'amt'=>float].
 * Only returns entries with amounts (does not split totals here).
 */
function parse_sp_tp_entries_all($detailsJson) {
  $rows = [];

  $j = _bp_json_decode($detailsJson);
  if (is_array($j)) {
    if (_bp_is_assoc($j)) {
      // {"222":55,"555":64}
      foreach ($j as $k => $v) {
        $num = _bp_clean_num($k);
        $a   = (float)$v;
        if ($num !== '' && $a > 0) $rows[] = ['num' => $num, 'amt' => $a];
      }
      if ($rows) return $rows;
    } else {
      // [{"number":"123","amount":50}, ...]  or ["222","555"] (no amounts)
      foreach ($j as $row) {
        if (is_array($row)) {
          $num = isset($row['number']) ? _bp_clean_num($row['number'])
               : (isset($row['num'])    ? _bp_clean_num($row['num']) : '');
          $a   = isset($row['amount']) ? (float)$row['amount']
               : (isset($row['amt'])    ? (float)$row['amt'] : 0.0);
          if ($num !== '' && $a > 0) $rows[] = ['num' => $num, 'amt' => $a];
        } else {
          // bare numbers -> ignore (no total to split here)
        }
      }
      if ($rows) return $rows;
    }
  }

  // Pair string "123:50,456:20"
  $map = _bp_parse_pair_string((string)$detailsJson);
  if ($map) {
    foreach ($map as $k => $v) {
      $num = _bp_clean_num($k);
      if ($num !== '' && $v > 0) $rows[] = ['num' => $num, 'amt' => (float)$v];
    }
  }

  return $rows;
}

/**
 * JD: returns list of ['num'=>string(2),'amt'=>float]
 * If no amounts are present in details, split $totalAmount evenly.
 */
function parse_jd_items($detailsJson, $totalAmount) {
  $items = [];

  // 1) JSON
  $j = _bp_json_decode($detailsJson);
  if (is_array($j)) {
    if (_bp_is_assoc($j)) {
      // {"27":30,"05":20}
      foreach ($j as $k => $v) {
        $num = _bp_clean_num($k, 2);
        $a   = (float)$v;
        if ($num !== '' && $a > 0) $items[] = ['num' => $num, 'amt' => $a];
      }
      if ($items) return $items;
    } else {
      // [{"number":"27","amount":30}, "05", "99"]
      $bare = [];
      foreach ($j as $row) {
        if (is_array($row)) {
          $num = isset($row['number']) ? _bp_clean_num($row['number'], 2)
               : (isset($row['num'])    ? _bp_clean_num($row['num'], 2) : '');
          $a   = isset($row['amount']) ? (float)$row['amount']
               : (isset($row['amt'])    ? (float)$row['amt'] : 0.0);
          if ($num !== '' && $a > 0)     $items[] = ['num' => $num, 'amt' => $a];
          elseif ($num !== '' && $a == 0) $bare[]  = $num;
        } else {
          $n = _bp_clean_num($row, 2);
          if ($n !== '') $bare[] = $n;
        }
      }
      if ($items) return $items;
      if ($bare) {
        $bare   = array_values(array_unique($bare));
        $split  = _bp_split_even(count($bare), (float)$totalAmount);
        foreach ($bare as $i => $n) $items[] = ['num' => $n, 'amt' => $split[$i]];
        return $items;
      }
    }
  }

  // 2) Pair string: "27:30,05:20"
  $map = _bp_parse_pair_string((string)$detailsJson);
  if ($map) {
    foreach ($map as $k => $v) {
      $num = _bp_clean_num($k, 2);
      if ($num !== '' && $v > 0) $items[] = ['num' => $num, 'amt' => (float)$v];
    }
    if ($items) return $items;
  }

  // 3) Only numbers: "27,05,99" -> split total
  $only = array_filter(
    array_map(fn($x) => _bp_clean_num($x, 2), preg_split('/[,\s]+/', (string)$detailsJson))
  );
  if ($only) {
    $only  = array_values(array_unique($only));
    $split = _bp_split_even(count($only), (float)$totalAmount);
    foreach ($only as $i => $n) $items[] = ['num' => $n, 'amt' => $split[$i]];
  }

  return $items;
}
