<?php
// bids_result_api.php — unified live bid history API with 2-second file cache

header('Content-Type: application/json; charset=utf-8');
ini_set('display_errors', 1);
ini_set('log_errors', 1);
error_reporting(E_ALL);

try {
    // 1) Parse & validate JSON input
    $in = json_decode(file_get_contents('php://input'), true);
    if (!$in || json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Invalid JSON: ' . json_last_error_msg());
    }
    if (empty($in['mobile']) || !preg_match('/^\d{10}$/', $in['mobile'])) {
        throw new Exception('Missing or invalid "mobile"');
    }
    $mobile = $in['mobile'];

    // 2) Date filter (defaults to today)
    $filter_date = date('Y-m-d');
    if (!empty($in['filter_date'])) {
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $in['filter_date'])) {
            throw new Exception('Invalid "filter_date"');
        }
        $filter_date = $in['filter_date'];
    }

    // 3) Prepare cache
    $cacheDir  = __DIR__ . '/cache';
    if (!is_dir($cacheDir)) {
        mkdir($cacheDir, 0755, true);
    }
    $cacheFile = "{$cacheDir}/bids_{$mobile}_{$filter_date}.json";

    // 4) Serve from cache if <2s old
    if (file_exists($cacheFile) && (time() - filemtime($cacheFile) < 2)) {
        echo file_get_contents($cacheFile);
        exit;
    }

    // 5) DB connection
    $db = new PDO(
        'mysql:host=localhost;dbname=u672932721_sai_app;charset=utf8mb4',
        'u672932721_sai_app_user',
        'Admin_66666',
        [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]
    );
    $db->exec("SET NAMES utf8mb4");

    // 6) Lookup user_id
    $stmt = $db->prepare("SELECT user_id FROM users_data WHERE mobile = ? LIMIT 1");
    $stmt->execute([$mobile]);
    $uid = $stmt->fetchColumn();
    if (!$uid) {
        throw new Exception('User not found');
    }
    
    function isPendingValue($val) {
    if (is_null($val)) {
        return true;
    }
    $t = trim($val);
    return $t === ''        // empty
        || $t === '-'       // ASCII hyphen
        || $t === '–'       // en-dash
        || $t === 'Pending'
        || strtoupper($t) === 'NULL';
}

    // 7) Status formatter — improved to catch NULL, '', '-', '–'
    function formatStatus($session, $openRes, $closeRes) {
    $raw = (strtolower($session) === 'open') ? $openRes : $closeRes;

    if (isPendingValue($raw)) {
        return 'Pending';
    }

    if (strpos($raw, '🏆') !== false
        && preg_match('/([\d,]+(?:\.\d{2})?)/', $raw, $m)
    ) {
        $win = floatval(str_replace(',', '', $m[1]));
        return "You Won ₹{$win} 😊";
    }

    return "Better Luck Next Time!!! 😞";
}

    $out = [];

    // 8a) Single-digit–style games: SD, DP, TP, SP, JD
    foreach (['sd','dp','tp','sp','jd'] as $sg) {
        $label  = strtoupper($sg);
        $sql = "
            SELECT
                '{$label}' AS subgame,
                r.id AS result_id,
                g.id AS game_id,
                g.game_name,
                DATE_FORMAT(rw.reward_date, '%d/%m/%Y') AS game_date,
                CONCAT('{$label}_', r.id) AS bid_id,
                r.bid_number AS digit,
                r.bid_amount AS amount,
                r.bid_session AS session,
                DATE_FORMAT(CONVERT_TZ(NOW(), '+00:00', '+05:30'), '%h:%i %p') AS transaction_time,
                r.res_open,
                r.res_close,
                r.id AS sort_order
            FROM {$sg}_result r
            JOIN {$sg}_rewards rw ON rw.id = r.reward_id
            JOIN game56 g ON g.id = rw.game_id
            WHERE r.user_id = :uid
              AND DATE(rw.reward_date) = :fdate
        ";
        $stmt = $db->prepare($sql);
        $stmt->execute([':uid'=>$uid, ':fdate'=>$filter_date]);
        while ($row = $stmt->fetch()) {
            $out[] = [
                'subgame'          => $row['subgame'],
                'game_id'          => (int)$row['game_id'],
                'game_name'        => $row['game_name'],
                'game_date'        => $row['game_date'],
                'transaction_time' => $row['transaction_time'],
                'bid_id'           => $row['bid_id'],
                'digit'            => (string)$row['digit'],
                'amount'           => (float)$row['amount'],
                'session'          => ucfirst($row['session']),
                'result'           => formatStatus($row['session'], $row['res_open'], $row['res_close']),
                'sort_order'       => (int)$row['sort_order']
            ];
        }
    }

    // 8b) Half Sangam (HS)
    $hsSql = "
        SELECT
            'HS' as subgame,
            r.id as result_id,
            g.id as game_id,
            g.game_name,
            DATE_FORMAT(rw.reward_date, '%d/%m/%Y') AS game_date,
            CONCAT('HS_', r.id) AS bid_id,
            CONCAT(r.bid_od_hs,'|',r.bid_cp_hs) AS digit,
            r.bid_amount AS amount,
            r.bid_session AS session,
            DATE_FORMAT(CONVERT_TZ(NOW(), '+00:00', '+05:30'), '%h:%i %p') AS transaction_time,
            r.res_open_od,
            r.res_open_cp,
            r.res_close_od,
            r.res_close_cp,
            r.id as sort_order
        FROM hs_result r
        JOIN hs_rewards rw ON rw.id = r.reward_id
        JOIN game56 g ON g.id = rw.game_id
        WHERE r.user_id = :uid
          AND DATE(rw.reward_date) = :fdate
    ";
    $stmt = $db->prepare($hsSql);
    $stmt->execute([':uid'=>$uid, ':fdate'=>$filter_date]);
    while ($row = $stmt->fetch()) {
        // count pending and total win
        $pending = 0; $win = 0;
        foreach ([
            $row['res_open_od'], $row['res_open_cp'],
            $row['res_close_od'], $row['res_close_cp']
        ] as $res) {
            if (is_null($res)
                || trim($res) === ''
                || trim($res) === '-'
                || trim($res) === '–'
            ) {
                $pending++;
            } elseif (strpos($res,'🏆')!==false
                && preg_match('/([\d,]+(?:\.\d{2})?)/',$res,$m)
            ) {
                $win += floatval(str_replace(',','',$m[1]));
            }
        }
        if ($pending === 4) {
            $status = 'Pending';
        } elseif ($win > 0) {
            $status = "You Won ₹{$win} 😊";
        } else {
            $status = "Better Luck Next Time!!! 😞";
        }
        $out[] = [
            'subgame'          => 'HS',
            'game_id'          => (int)$row['game_id'],
            'game_name'        => $row['game_name'],
            'game_date'        => $row['game_date'],
            'transaction_time' => $row['transaction_time'],
            'bid_id'           => $row['bid_id'],
            'digit'            => $row['digit'],
            'amount'           => (float)$row['amount'],
            'session'          => ucfirst($row['session']),
            'result'           => $status,
            'sort_order'       => (int)$row['sort_order']
        ];
    }

    // 8c) Full Sangam (FS)
    $fsSql = "
        SELECT
            'FS' as subgame,
            r.id as result_id,
            g.id as game_id,
            g.game_name,
            DATE_FORMAT(rw.reward_date, '%d/%m/%Y') AS game_date,
            CONCAT('FS_', r.id) AS bid_id,
            CONCAT(r.bid_open_fs,'|',r.bid_close_fs) AS digit,
            r.bid_amount AS amount,
            r.bid_session AS session,
            DATE_FORMAT(CONVERT_TZ(NOW(), '+00:00', '+05:30'), '%h:%i %p') AS transaction_time,
            r.res_open_op,
            r.res_open_cp,
            r.res_close_op,
            r.res_close_cp,
            r.id as sort_order
        FROM fs_result r
        JOIN fs_rewards rw ON rw.id = r.reward_id
        JOIN game56 g ON g.id = rw.game_id
        WHERE r.user_id = :uid
          AND DATE(rw.reward_date) = :fdate
    ";
    $stmt = $db->prepare($fsSql);
    $stmt->execute([':uid'=>$uid, ':fdate'=>$filter_date]);
    while ($row = $stmt->fetch()) {
        $pending = 0; $win = 0;
        foreach ([
            $row['res_open_op'], $row['res_open_cp'],
            $row['res_close_op'], $row['res_close_cp']
        ] as $res) {
            if (is_null($res)
                || trim($res) === ''
                || trim($res) === '-'
                || trim($res) === '–'
            ) {
                $pending++;
            } elseif (strpos($res,'🏆')!==false
                && preg_match('/([\d,]+(?:\.\d{2})?)/',$res,$m)
            ) {
                $win += floatval(str_replace(',','',$m[1]));
            }
        }
        if ($pending === 4) {
            $status = 'Pending';
        } elseif ($win > 0) {
            $status = "You Won ₹{$win} 😊";
        } else {
            $status = "Better Luck Next Time!!! 😞";
        }
        $out[] = [
            'subgame'          => 'FS',
            'game_id'          => (int)$row['game_id'],
            'game_name'        => $row['game_name'],
            'game_date'        => $row['game_date'],
            'transaction_time' => $row['transaction_time'],
            'bid_id'           => $row['bid_id'],
            'digit'            => $row['digit'],
            'amount'           => (float)$row['amount'],
            'session'          => ucfirst($row['session']),
            'result'           => $status,
            'sort_order'       => (int)$row['sort_order']
        ];
    }

    // 9) Sort by sort_order descending
    usort($out, function($a, $b){
        return $b['sort_order'] - $a['sort_order'];
    });

    // 10) Build and cache response
    foreach ($out as &$row) {
        unset($row['sort_order']);
    }
    $response = [
        'status' => 'success',
        'count'  => count($out),
        'data'   => $out
    ];

    $json = json_encode($response, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT);
    file_put_contents($cacheFile, $json);
    echo $json;

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'status'  => 'error',
        'message' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT);
}
