<?php
// bids_sd_api.php — Passbook endpoint (bids, topups, withdrawals, win credits)

ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
ini_set('html_errors', 0);
error_reporting(0);

header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/db.php';            // $conn = new mysqli(...)
$conn->set_charset('utf8mb4');
require_once __DIR__ . '/bid_parsers.php';   // helper parsers you already use

/* ---------- payout multipliers (per ₹1 stake) ---------- */
$PAYOUT = [
  'SD' => 10,
  'JD' => 100,
  'SP' => 160,
  'DP' => 320,
  'TP' => 1000,
  'HS' => 1000,
  'FS' => 10000,
];

/* ---------- 1) Read & validate request ---------- */
$body  = file_get_contents('php://input');
$input = json_decode($body, true);
if (!is_array($input)) {
  http_response_code(400);
  echo json_encode(['status'=>'error','message'=>'Invalid JSON']);
  exit;
}

$mobile = preg_replace('/\D+/', '', (string)($input['mobile'] ?? ''));
if (!preg_match('/^\d{10,12}$/', $mobile)) {
  http_response_code(400);
  echo json_encode(['status'=>'error','message'=>'Missing or invalid "mobile".']);
  exit;
}

/* ---------- 2) Resolve user_id ---------- */
$stmt = $conn->prepare("SELECT user_id FROM users_data WHERE mobile = ? LIMIT 1");
$stmt->bind_param('s', $mobile);
$stmt->execute();
$stmt->bind_result($uid);
if (!$stmt->fetch()) {
  $stmt->close();
  http_response_code(404);
  echo json_encode(['status'=>'error','message'=>'User not found']);
  exit;
}
$stmt->close();

/* ---------- 3) Build base (bids) query ---------- */
/* NOTE: keep all your existing bid tables. Amounts are NEGATIVE for bids placed. */
$sql = "
  (SELECT w.created_at AS date, 'Wallet Snapshot' AS particular, w.balance AS amount
     FROM wallet AS w
    WHERE w.user_id = {$uid}
    LIMIT 0) /* disabled output; using proper topups below */

  UNION ALL
  (SELECT ub.created_at, CONCAT('Single Digit Bid: ', ub.bid_details),        -ub.total_bid_amount
     FROM user_bids     AS ub WHERE ub.user_id = {$uid})

  UNION ALL
  (SELECT ub.created_at, CONCAT('Double Panna Bid: ', ub.bid_details),        -ub.total_bid_amount
     FROM user_bids_dp  AS ub WHERE ub.user_id = {$uid})

  UNION ALL
  (SELECT ub.created_at, CONCAT('Triple Panna Bid: ', ub.bid_details),        -ub.total_bid_amount
     FROM user_bids_tp  AS ub WHERE ub.user_id = {$uid})

  UNION ALL
  (SELECT ub.created_at, CONCAT('Jodi Digit Bid: ', ub.bid_details),          -ub.total_bid_amount
     FROM user_bids_jd  AS ub WHERE ub.user_id = {$uid})

  UNION ALL
  (SELECT ub.created_at, CONCAT('Single Panna Bid: ', ub.bid_details),        -ub.total_bid_amount
     FROM user_bids_sp  AS ub WHERE ub.user_id = {$uid})

  UNION ALL
  (SELECT ub.created_at,
          CONCAT('Half Sangam Bid: ', ub.bid_open_hs, ' | ', ub.bid_close_hs),
          -ub.total_bid_amount
     FROM user_bids_hs  AS ub WHERE ub.user_id = {$uid})

  UNION ALL
  (SELECT ub.created_at,
          CONCAT('Full Sangam Bid: ', ub.bid_open_fs, ' | ', ub.bid_close_fs),
          -ub.total_bid_amount
     FROM user_bids_fs  AS ub WHERE ub.user_id = {$uid})

  /* ---------- Wallet Adds via IMB (CREDIT) ---------- */
  UNION ALL
  (SELECT wo.created_at,
          CONCAT('Wallet Add (IMB) • OID ', wo.order_id),
          wo.amount
     FROM wallet_orders AS wo
    WHERE REPLACE(REPLACE(REPLACE(wo.mobile,'+',''),' ',''),'-','') = '{$mobile}'
      AND wo.status = 'SUCCESS')

  /* ---------- Withdraw Requests (DEBIT) ---------- */
  UNION ALL
  (SELECT wr.created_at,
          CONCAT('Withdraw ', wr.method, ' (', wr.status, ')'),
          -wr.amount
     FROM withdrawal_requests AS wr
    WHERE wr.user_id = {$uid})

  ORDER BY date DESC
";

$res = $conn->query($sql);
if (!$res) {
  http_response_code(500);
  echo json_encode(['status'=>'error','message'=>'SQL error: '.$conn->error]);
  exit;
}

/* ---------- 4) Format rows, expand aggregate bid lines ---------- */
date_default_timezone_set('Asia/Kolkata');

function push_row(array &$data, string $utcCreated, string $label, float $amt): void {
  // Treat DB timestamps as UTC; display in IST
  $dt = new DateTime($utcCreated, new DateTimeZone('UTC'));
  $dt->setTimezone(new DateTimeZone('Asia/Kolkata'));
  $data[] = [
    'date'       => $dt->format('Y-m-d'),
    'time'       => $dt->format('g:i A'),
    'particular' => $label,
    'amount'     => round($amt, 2),
  ];
}

$data = [];
while ($row = $res->fetch_row()) {
  $utc   = (string)$row[0];  // created_at
  $label = (string)$row[1];
  $amt   = (float)$row[2];

  // ---- SD (Single Digit) ----
  if (strpos($label, 'Single Digit Bid: ') === 0) {
    $json = trim(substr($label, strlen('Single Digit Bid: ')));
    $pairs = sd_extract_pairs($json, abs($amt));  // returns [digit => perDigitAmt]
    if (is_array($pairs) && $pairs) {
      foreach ($pairs as $d => $a) {
        push_row($data, $utc, "Single Digit Bid: $d", -(float)$a);
      }
      continue;
    }
  }

  // ---- DP (Double Panna) ----
  if (strpos($label, 'Double Panna Bid: ') === 0) {
    $json = trim(substr($label, strlen('Double Panna Bid: ')));
    $pairs = parse_dp_pairs($json);               // ['128'=>amt, ...]
    if (is_array($pairs) && $pairs) {
      foreach ($pairs as $num => $a) {
        push_row($data, $utc, "Double Panna Bid: {$num}", - (float)$a);
      }
      continue;
    }
  }

  // ---- SP (Single Panna) ----
  if (strpos($label, 'Single Panna Bid: ') === 0) {
    $json = trim(substr($label, strlen('Single Panna Bid: ')));
    $arr = json_decode($json, true);              // [{"number":"123","amount":50}, ...]
    if (is_array($arr) && $arr) {
      foreach ($arr as $o) {
        $num = (string)($o['number'] ?? '');
        $a   = (float)($o['amount'] ?? 0);
        if ($num !== '' && $a > 0) {
          push_row($data, $utc, "Single Panna Bid: {$num}", -$a);
        }
      }
      continue;
    }
  }

  // ---- TP (Triple Panna) ----
  if (strpos($label, 'Triple Panna Bid: ') === 0) {
    $json = trim(substr($label, strlen('Triple Panna Bid: ')));
    $arr  = json_decode($json, true);             // {"222":55,"555":64}
    if (is_array($arr) && $arr) {
      foreach ($arr as $num => $a) {
        push_row($data, $utc, "Triple Panna Bid: {$num}", - (float)$a);
      }
      continue;
    }
  }

  // ---- JD (Jodi Digit) ----
  if (strpos($label, 'Jodi Digit Bid: ') === 0) {
    $json = trim(substr($label, strlen('Jodi Digit Bid: ')));
    $items = parse_jd_items($json, abs($amt));    // [ {'num':'12','amt':X}, ... ]
    if (is_array($items) && $items) {
      foreach ($items as $e) {
        $num = (string)($e['num'] ?? '');
        $a   = (float)($e['amt'] ?? 0);
        if ($num !== '' && $a > 0) {
          push_row($data, $utc, "Jodi Digit Bid: {$num}", -$a);
        }
      }
      continue;
    }
  }

  // ---- Everything else (HS/FS, Wallet Add, Withdraw, fallback) ----
  push_row($data, $utc, $label, $amt);
}

/* ---------- 5) Append win credits (computed) ---------- */
function add_credit(array &$data, string $utcCreated, string $label, float $amt): void {
  if ($amt <= 0) return;
  $dt = new DateTime($utcCreated, new DateTimeZone('UTC'));
  $dt->setTimezone(new DateTimeZone('Asia/Kolkata'));
  $data[] = [
    'date'       => $dt->format('Y-m-d'),
    'time'       => $dt->format('g:i A'),
    'particular' => $label,
    'amount'     => round($amt, 2),
  ];
}

/* ---- SD wins ---- */
$q = "
  SELECT b.created_at, b.game_name, b.bid_details, b.bid_session, b.total_bid_amount,
         r.winning_open_time_sd  AS win_o,
         r.winning_close_time_sd AS win_c
    FROM user_bids b
    LEFT JOIN sd_rewards r
      ON r.game_id = b.game_id
     AND r.reward_date = DATE(CONVERT_TZ(b.created_at, '+00:00', '+05:30'))
   WHERE b.user_id = {$uid}
";
if ($rs = $conn->query($q)) {
  while ($R = $rs->fetch_assoc()) {
    $pairs = sd_extract_pairs((string)$R['bid_details'], (float)$R['total_bid_amount']);
    if (!$pairs) continue;
    $sess = (strtoupper((string)$R['bid_session']) === 'CLOSE') ? 'CLOSE' : 'OPEN';
    $winO = ($R['win_o'] !== null && $R['win_o'] !== '') ? (int)$R['win_o'] : null;
    $winC = ($R['win_c'] !== null && $R['win_c'] !== '') ? (int)$R['win_c'] : null;

    foreach ($pairs as $d => $amt) {
      $ok = ($sess === 'OPEN'  && $winO !== null && (int)$d === $winO)
         || ($sess === 'CLOSE' && $winC !== null && (int)$d === $winC);
      if ($ok) {
        add_credit($data, $R['created_at'], "Credit - SD Win ({$R['game_name']} $sess • $d)", (float)$amt * $PAYOUT['SD']);
      }
    }
  }
}

/* ---- DP wins ---- */
$q = "
  SELECT b.created_at, b.game_name, b.bid_details, b.bid_session,
         r.winning_open_time_dp  AS win_o,
         r.winning_close_time_dp AS win_c
    FROM user_bids_dp b
    LEFT JOIN dp_rewards r
      ON r.game_id = b.game_id
     AND r.reward_date = DATE(CONVERT_TZ(b.created_at, '+00:00', '+05:30'))
   WHERE b.user_id = {$uid}
";
if ($rs = $conn->query($q)) {
  while ($R = $rs->fetch_assoc()) {
    $pairs = parse_dp_pairs((string)$R['bid_details']);
    if (!$pairs) continue;
    $sess = (strtoupper((string)$R['bid_session']) === 'CLOSE') ? 'CLOSE' : 'OPEN';
    $winO = ($R['win_o'] !== null && $R['win_o'] !== '') ? (string)$R['win_o'] : null;
    $winC = ($R['win_c'] !== null && $R['win_c'] !== '') ? (string)$R['win_c'] : null;

    foreach ($pairs as $num => $amt) {
      $ok = ($sess === 'OPEN'  && $winO !== null && (string)$num === $winO)
         || ($sess === 'CLOSE' && $winC !== null && (string)$num === $winC);
      if ($ok) {
        add_credit($data, $R['created_at'], "Credit - DP Win ({$R['game_name']} $sess • $num)", (float)$amt * $PAYOUT['DP']);
      }
    }
  }
}

/* ---- SP wins ---- */
$q = "
  SELECT b.created_at, b.game_name, b.bid_details, b.bid_session,
         r.winning_open_time_sp  AS win_o,
         r.winning_close_time_sp AS win_c
    FROM user_bids_sp b
    LEFT JOIN sp_rewards r
      ON r.game_id = b.game_id
     AND r.reward_date = DATE(CONVERT_TZ(b.created_at, '+00:00', '+05:30'))
   WHERE b.user_id = {$uid}
";
if ($rs = $conn->query($q)) {
  while ($R = $rs->fetch_assoc()) {
    $rows = parse_sp_tp_entries_all((string)$R['bid_details']);
    if (!$rows) continue;
    $sess = (strtoupper((string)$R['bid_session']) === 'CLOSE') ? 'CLOSE' : 'OPEN';
    $winO = ($R['win_o'] !== null && $R['win_o'] !== '') ? (string)$R['win_o'] : null;
    $winC = ($R['win_c'] !== null && $R['win_c'] !== '') ? (string)$R['win_c'] : null;

    foreach ($rows as $e) {
      $num = (string)$e['num']; $amt = (float)$e['amt'];
      $ok = ($sess === 'OPEN'  && $winO !== null && $num === $winO)
         || ($sess === 'CLOSE' && $winC !== null && $num === $winC);
      if ($ok) {
        add_credit($data, $R['created_at'], "Credit - SP Win ({$R['game_name']} $sess • $num)", $amt * $PAYOUT['SP']);
      }
    }
  }
}

/* ---- TP wins ---- */
$q = "
  SELECT b.created_at, b.game_name, b.bid_details, b.bid_session,
         r.winning_open_time_tp  AS win_o,
         r.winning_close_time_tp AS win_c
    FROM user_bids_tp b
    LEFT JOIN tp_rewards r
      ON r.game_id = b.game_id
     AND r.reward_date = DATE(CONVERT_TZ(b.created_at, '+00:00', '+05:30'))
   WHERE b.user_id = {$uid}
";
if ($rs = $conn->query($q)) {
  while ($R = $rs->fetch_assoc()) {
    $rows = parse_sp_tp_entries_all((string)$R['bid_details']);
    if (!$rows) continue;
    $sess = (strtoupper((string)$R['bid_session']) === 'CLOSE') ? 'CLOSE' : 'OPEN';
    $winO = ($R['win_o'] !== null && $R['win_o'] !== '') ? (string)$R['win_o'] : null;
    $winC = ($R['win_c'] !== null && $R['win_c'] !== '') ? (string)$R['win_c'] : null;

    foreach ($rows as $e) {
      $num = (string)$e['num']; $amt = (float)$e['amt'];
      $ok = ($sess === 'OPEN'  && $winO !== null && $num === $winO)
         || ($sess === 'CLOSE' && $winC !== null && $num === $winC);
      if ($ok) {
        add_credit($data, $R['created_at'], "Credit - TP Win ({$R['game_name']} $sess • $num)", $amt * $PAYOUT['TP']);
      }
    }
  }
}

/* ---- HS wins ---- */
$q = "
  SELECT b.created_at, b.game_name, b.bid_open_hs AS bid_od, b.bid_close_hs AS bid_cp,
         b.total_bid_amount, b.bid_session,
         r.winning_open_time_od_hs  AS w_open_od,  r.winning_open_time_cp_hs  AS w_open_cp,
         r.winning_close_time_od_hs AS w_close_od, r.winning_close_time_cp_hs AS w_close_cp
    FROM user_bids_hs b
    LEFT JOIN hs_rewards r
      ON r.game_id = b.game_id
     AND r.reward_date = DATE(CONVERT_TZ(b.created_at, '+00:00', '+05:30'))
   WHERE b.user_id = {$uid}
";
if ($rs = $conn->query($q)) {
  while ($R = $rs->fetch_assoc()) {
    $sess = (strtoupper((string)$R['bid_session']) === 'CLOSE') ? 'CLOSE' : 'OPEN';
    $openWin  = ($R['w_open_od'] !== '' && $R['w_open_cp'] !== '') && ((string)$R['bid_od'] === (string)$R['w_open_od']) && ((string)$R['bid_cp'] === (string)$R['w_open_cp']);
    $closeWin = ($R['w_close_od'] !== '' && $R['w_close_cp'] !== '') && ((string)$R['bid_od'] === (string)$R['w_close_od']) && ((string)$R['bid_cp'] === (string)$R['w_close_cp']);
    $ok = ($sess === 'OPEN' && $openWin) || ($sess === 'CLOSE' && $closeWin);
    if ($ok) {
      add_credit($data, $R['created_at'], "Credit - HS Win ({$R['game_name']} $sess • {$R['bid_od']} | {$R['bid_cp']})", (float)$R['total_bid_amount'] * $PAYOUT['HS']);
    }
  }
}

/* ---- FS wins ---- */
$q = "
  SELECT b.created_at, g.game_name, b.bid_open_fs AS bid_op, b.bid_close_fs AS bid_cp,
         b.total_bid_amount, b.bid_session,
         r.winning_open_op_fs  AS w_open_op,  r.winning_open_cp_fs  AS w_open_cp,
         r.winning_close_op_fs AS w_close_op, r.winning_close_cp_fs AS w_close_cp
    FROM user_bids_fs b
    LEFT JOIN game56 g ON g.id = b.game_id
    LEFT JOIN fs_rewards r
      ON r.game_id = b.game_id
     AND r.reward_date = DATE(CONVERT_TZ(b.created_at, '+00:00', '+05:30'))
   WHERE b.user_id = {$uid}
";
if ($rs = $conn->query($q)) {
  while ($R = $rs->fetch_assoc()) {
    $sess = (strtoupper((string)$R['bid_session']) === 'CLOSE') ? 'CLOSE' : 'OPEN';
    $openWin  = ($R['w_open_op'] !== '' && $R['w_open_cp'] !== '') && ((string)$R['bid_op'] === (string)$R['w_open_op']) && ((string)$R['bid_cp'] === (string)$R['w_open_cp']);
    $closeWin = ($R['w_close_op'] !== '' && $R['w_close_cp'] !== '') && ((string)$R['bid_op'] === (string)$R['w_close_op']) && ((string)$R['bid_cp'] === (string)$R['w_close_cp']);
    $ok = ($sess === 'OPEN' && $openWin) || ($sess === 'CLOSE' && $closeWin);
    if ($ok) {
      add_credit($data, $R['created_at'], "Credit - FS Win ({$R['game_name']} $sess • {$R['bid_op']} | {$R['bid_cp']})", (float)$R['total_bid_amount'] * $PAYOUT['FS']);
    }
  }
}

/* ---------- 6) Sort (desc) & output ---------- */
usort($data, function($a,$b){
  $ta = strtotime($a['date'].' '.$a['time']);  // already IST
  $tb = strtotime($b['date'].' '.$b['time']);
  return $tb <=> $ta;
});

echo json_encode(['status'=>'success','data'=>$data], JSON_UNESCAPED_UNICODE);
