#!/usr/bin/php
<?php
// daemon_sd_update.php — regenerate cache every 2 seconds

ini_set('display_errors', 1);
error_reporting(E_ALL);
set_time_limit(0);
date_default_timezone_set('Asia/Kolkata');

// 1) Auth
if (empty($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// 2) DB
require_once __DIR__ . '/db.php';
if ($conn->connect_error) {
    die("DB Connect Error: " . $conn->connect_error);
}
$conn->set_charset('utf8mb4');

// Helper: parse bids into [digit, stake] pairs
function explode_ticket($raw, $stake) {
    if (empty($raw)) return [];
    $j = json_decode($raw, true);
    if (json_last_error() === JSON_ERROR_NONE && is_array($j)) {
        $out = [];
        foreach ($j as $d => $a) {
            $out[] = [(int)$d, (float)$a];
        }
        return $out;
    }
    if (strlen($raw)>2 && $raw[0]=='[' && substr($raw,-1)==']' && substr_count($raw,',')===0) {
        $in = trim(substr($raw,1,-1));
        if (is_numeric($in)) return [[(int)$in, $stake]];
    }
    $out = [];
    foreach (preg_split('/\D+/', $raw) as $d) {
        if ($d !== '' && is_numeric($d)) $out[] = [(int)$d, $stake];
    }
    return $out;
}

// Helper: calculate open/close results
function calculateResult($digit, $amt, $session, $winO, $winC, $openT, $closeT) {
    $now = time();
    $rO = $rC = '–';
    if ($session==='open' && $winO !== null) {
        $rO = $now >= strtotime($openT)
               ? ($digit==$winO ? "🏆 ₹".number_format($amt*10,2) : "❌ Better luck")
               : 'Pending';
    }
    if ($session==='close' && $winC !== null) {
        $rC = $now >= strtotime($closeT)
               ? ($digit==$winC ? "🏆 ₹".number_format($amt*10,2) : "❌ Better luck")
               : 'Pending';
    }
    return [$rO, $rC];
}

$filter_date = date('Y-m-d');
$cache_dir   = __DIR__ . '/cache';
@mkdir($cache_dir, 0755);
$cache_file  = "$cache_dir/sd_results_{$filter_date}.json";

while (true) {
    // 1) fetch data
    $sql = "
      SELECT
        g.game_name, g.open_time, g.close_time,
        DATE_FORMAT(g.open_time,'%l:%i %p') AS fmt_open,
        DATE_FORMAT(g.close_time,'%l:%i %p') AS fmt_close,
        u.username, ub.bid_details, ub.total_bid_amount,
        ub.bid_session, r.winning_open_time_sd,
        r.winning_close_time_sd, ub.created_at
      FROM user_bids ub
      JOIN users_data u ON u.user_id=ub.user_id
      JOIN game56 g ON g.id=ub.game_id
      LEFT JOIN sd_rewards r
        ON r.game_id=g.id AND r.reward_date=?
      WHERE DATE(ub.created_at)=?
      ORDER BY ub.created_at DESC
    ";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param('ss', $filter_date, $filter_date);
    $stmt->execute();
    $res = $stmt->get_result();

    $all = [];
    while ($row = $res->fetch_assoc()) {
        foreach (explode_ticket($row['bid_details'], (float)$row['total_bid_amount']) as $t) {
            list($dig,$amt) = $t;
            list($rO,$rC) = calculateResult(
                $dig, $amt, $row['bid_session'],
                $row['winning_open_time_sd'],
                $row['winning_close_time_sd'],
                $row['open_time'],
                $row['close_time']
            );
            $all[] = [
                'game'   => $row['game_name'],
                'open'   => $row['fmt_open'],
                'close'  => $row['fmt_close'],
                'user'   => $row['username'],
                'digit'  => $dig,
                'amount' => $amt,
                'session'=> ucfirst($row['bid_session']),
                'winO'   => $row['winning_open_time_sd'] ?: '–',
                'winC'   => $row['winning_close_time_sd'] ?: '–',
                'resO'   => $rO,
                'resC'   => $rC,
                'time'   => $row['created_at'],
            ];
        }
    }
    $stmt->close();

    // 2) write cache
    file_put_contents($cache_file, json_encode($all, JSON_PRETTY_PRINT));

    // 3) wait 2 seconds
    sleep(2);
}
