<?php
// /Sai/deposit_api.php — credit money to a user’s wallet (atomic & idempotent)

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Content-Type: application/json; charset=UTF-8');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
  http_response_code(200);
  exit; // preflight
}

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/lib/wallet_helper.php';  // <-- helper with wallet_credit()

// 1) Parse body (accept JSON or form)
$in     = json_decode(file_get_contents('php://input'), true) ?: $_REQUEST;
$mobile = trim($in['mobile'] ?? '');
$amount = isset($in['amount']) ? (float)$in['amount'] : 0.0;
/**
 * Optional but recommended: upstream transaction reference (for idempotency)
 * e.g. UPI txn id, order id, gateway payment id etc.
 */
$ref_id = isset($in['txn_id']) ? trim($in['txn_id']) : null;
if ($ref_id === null || $ref_id === '') {
  // make an idempotent local id (mobile + amount + minute bucket)
  $ref_id = 'DEP-' . $mobile . '-' . number_format($amount, 2, '.', '') . '-' . date('YmdHi');
}
// Basic validation
if (!preg_match('/^\d{10,15}$/', $mobile)) {
  http_response_code(400);
  echo json_encode(['success' => false, 'message' => 'Invalid mobile.']);
  exit;
}
if ($amount < 10) { // keep UI rule: minimum ₹10
  http_response_code(400);
  echo json_encode(['success' => false, 'message' => 'Minimum amount is ₹10.']);
  exit;
}

// 2) Find user_id for the mobile
$stmt = $conn->prepare("SELECT user_id FROM users_data WHERE mobile = ? LIMIT 1");
$stmt->bind_param('s', $mobile);
$stmt->execute();
$stmt->bind_result($user_id);
$found = $stmt->fetch();
$stmt->close();

if (!$found) {
  http_response_code(404);
  echo json_encode(['success' => false, 'message' => 'User not found.']);
  exit;
}

// 3) Credit the wallet (atomic + idempotent)
// reason: 'deposit', game: null, ref_id: upstream txn (prevents double-credit)
$ok = wallet_credit((int)$user_id, (float)$amount, 'deposit', null, $ref_id);
if (!$ok) {
  http_response_code(500);
  echo json_encode(['success' => false, 'message' => 'Failed to credit wallet.']);
  exit;
}

// 4) Return fresh balance
$bal = 0.00;
$sb  = $conn->prepare("SELECT balance FROM wallet WHERE user_id = ? LIMIT 1");
$sb->bind_param('i', $user_id);
$sb->execute();
$sb->bind_result($bal);
$sb->fetch();
$sb->close();

echo json_encode([
  'success' => true,
  'message' => 'Deposit successful.',
  'data'    => ['balance' => (float)$bal]
]);

$conn->close();
