<?php
// ─────────────────────────────────────────────────────────────────────────────
// double_panna.php
// ─────────────────────────────────────────────────────────────────────────────

// 1) Session + hide on‑screen errors, log instead
session_start();
ini_set('display_errors','0');
ini_set('log_errors','1');
ini_set('error_log', __DIR__.'/double_panna_error.log');
error_reporting(E_ALL);

// 2) DB
include 'db.php'; // must define $conn = new mysqli(...)

// 3) Auth
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}
$user_id = (int) $_SESSION['user_id'];

// 4) game_id check + mark subgame
if (empty($_GET['game_id'])) {
    die("No game selected.");
}
$game_id = (int) $_GET['game_id'];

if ($stmt = $conn->prepare("UPDATE game56 SET sub_games='double_panna' WHERE id=?")) {
    $stmt->bind_param("i",$game_id);
    $stmt->execute();
    $stmt->close();
}

// 5) Fetch open_time/close_time
$open_time = $close_time = '';
if ($stmt = $conn->prepare("SELECT open_time, close_time FROM game56 WHERE id=?")) {
    $stmt->bind_param("i",$game_id);
    $stmt->execute();
    $stmt->bind_result($open_time,$close_time);
    $stmt->fetch();
    $stmt->close();
}

// 6) Determine session using timestamps (IST)
date_default_timezone_set('Asia/Kolkata');
$today   = date('Y-m-d');
$nowTs   = time();
$openTs  = strtotime("$today $open_time");
$closeTs = strtotime("$today $close_time");
// if close <= open, assume overnight
if ($closeTs <= $openTs) {
    $closeTs += 86400;
}
$isCloseSession = ($nowTs >= $openTs && $nowTs <= $closeTs);

// 7) Button classes
$openBtnClass  = $isCloseSession ? 'btn-default' : 'btn-blue';
$closeBtnClass = $isCloseSession ? 'btn-blue'    : 'btn-default';

// 8) Wallet balance
$wallet_balance = 0.00;
if ($res = $conn->prepare("SELECT balance FROM wallet WHERE user_id=?")) {
    $res->bind_param("i",$user_id);
    $res->execute();
    $res->bind_result($wallet_balance);
    $res->fetch();
    $res->close();
}

// 9) Double panna definitions
$doublePanas = [
    "0 DOUBLE PANA’S" => ["118","226","244","299","334","488","550","668","677"],
    "1 DOUBLE PANA’S" => ["100","119","155","227","335","344","399","588","669"],
    "2 DOUBLE PANA’S" => ["110","200","228","255","336","499","660","688","778"],
    "3 DOUBLE PANA’S" => ["166","229","300","337","355","445","599","779","788"],
    "4 DOUBLE PANA’S" => ["112","220","266","338","400","446","455","699","770"],
    "5 DOUBLE PANA’S" => ["113","122","177","339","366","447","500","799","889"],
    "6 DOUBLE PANA’S" => ["600","114","277","330","448","466","556","880","899"],
    "7 DOUBLE PANA’S" => ["115","133","188","223","377","449","557","566","700"],
    "8 DOUBLE PANA’S" => ["116","224","233","288","440","477","558","800","990"],
    "9 DOUBLE PANA’S" => ["117","144","199","225","388","559","577","667","900"],
];

// 10) Build unique sorted list
$allNumbers = [];
foreach ($doublePanas as $nums) {
    $allNumbers = array_merge($allNumbers, $nums);
}
$allNumbers = array_unique($allNumbers, SORT_NUMERIC);
sort($allNumbers, SORT_NUMERIC);

// 11) Handle POST bid
$message = '';
if ($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['place_bid'])) {
    $selected = $_POST['selected_number'] ?? '';
    $bidAmt   = floatval($_POST['bid_amount'] ?? 0);

    if (!$selected) {
        $message = "<div class='alert alert-danger'>Please select a number.</div>";
    }
    elseif ($bidAmt < 10) {
        $message = "<div class='alert alert-danger'>Minimum bid is ₹10.</div>";
    }
    elseif ($bidAmt > $wallet_balance) {
        $message = "<div class='alert alert-danger'>Insufficient wallet balance.</div>";
    }
    else {
        $conn->begin_transaction();
        try {
            // 11.1) insert bid
            $sql = "
              INSERT INTO user_bids_dp
                (user_id, game_id, bid_details, total_bid_amount, bid_session, game_name)
              VALUES (?,?,?,?,?, (SELECT game_name FROM game56 WHERE id=?))
            ";
            if (!($ins = $conn->prepare($sql))) {
                throw new Exception($conn->error);
            }
            $session = $isCloseSession ? 'close' : 'open';
            $ins->bind_param("isdssi",
                $user_id,
                $game_id,
                $selected,
                $bidAmt,
                $session,
                $game_id
            );
            $ins->execute();
            $ins->close();

            // 11.2) deduct wallet
            if (!($upd = $conn->prepare("UPDATE wallet SET balance = balance - ? WHERE user_id=?"))) {
                throw new Exception($conn->error);
            }
            $upd->bind_param("di",$bidAmt,$user_id);
            $upd->execute();
            $upd->close();

            $conn->commit();
            $wallet_balance -= $bidAmt;
            $message = "<div class='alert alert-success'>
                          Successfully placed ₹".number_format($bidAmt,2).
                        " on {$selected} ({$session}).</div>";
        }
        catch (Exception $e) {
            $conn->rollback();
            $message = "<div class='alert alert-danger'>
                          Error placing bid: ".htmlspecialchars($e->getMessage())."
                        </div>";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Double Panna Bidding</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <style>
    body { background:#f8f9fa; margin:0; font-family:Arial,sans-serif; color:#333; }
    .back-button {
      position:absolute; top:20px; left:20px;
      background:#28a745; color:#fff; padding:8px 12px;
      text-decoration:none; border-radius:4px; font-size:14px;
    }
    .back-button:hover { background:#218838; }
    .header { text-align:center; padding:20px; }
    .wallet { font-size:1.2rem; font-weight:bold; margin-bottom:10px; }
    .toggle {
      text-align:center; margin-bottom:15px;
    }
    .toggle .btn {
      padding:8px 16px; margin:0 5px;
      border:none; border-radius:4px; cursor:pointer;
    }
    .btn-default { background:#ccc; color:#333; }
    .btn-blue    { background:#007BFF; color:#fff; }
    .grid {
      display:grid;
      grid-template-columns:repeat(auto-fit,minmax(280px,1fr));
      gap:20px; padding:20px;
    }
    .card {
      background:#fff; border:2px solid #f36; border-radius:8px;
      padding:15px; box-shadow:0 2px 5px rgba(0,0,0,0.1);
    }
    .card h2 { margin-top:0; color:#007BFF; font-size:1.2rem; }
    .numbers { margin:10px 0; font-size:1rem; line-height:1.5; }
    .form-container {
      max-width:500px; margin:30px auto; background:#fff; padding:20px;
      border-radius:8px; box-shadow:0 2px 5px rgba(0,0,0,0.1);
    }
    .form-container h3 { margin-top:0; color:#007BFF; text-align:center; }
    .alert { padding:0.75rem 1.25rem; margin-bottom:1rem; border-radius:4px; }
    .alert-success { background:#d4edda; color:#155724; }
    .alert-danger  { background:#f8d7da; color:#721c24; }
    label { display:block; margin:0.5rem 0; font-weight:bold; }
    select,input[type=number] {
      width:100%; padding:0.5rem; font-size:1rem; border:1px solid #ccc;
      border-radius:4px; box-sizing:border-box; margin-bottom:1rem;
    }
    button[type=submit] {
      width:100%; padding:0.75rem; font-size:1rem; color:#fff;
      background:#007BFF; border:none; border-radius:4px; cursor:pointer;
    }
    button[type=submit]:hover { background:#0056b3; }
  </style>
</head>
<body>

  <a href="dashboard.php" class="back-button">← Dashboard</a>

  <div class="header">
    <h1>Double Panna Bidding</h1>
    <div class="wallet">
      Wallet Balance: ₹<?=number_format($wallet_balance,2)?>
    </div>
    <div class="toggle">
      <button type="button" class="btn <?=$openBtnClass?>">Open</button>
      <button type="button" class="btn <?=$closeBtnClass?>">Close</button>
    </div>
  </div>

  <div class="grid">
    <?php foreach($doublePanas as $title=>$nums): ?>
      <div class="card">
        <h2><?=$title?></h2>
        <div class="numbers"><?=implode(", ",$nums)?></div>
      </div>
    <?php endforeach; ?>
  </div>

  <div class="form-container">
    <h3>Place Your Bid</h3>
    <?=$message?>
    <form method="POST">
      <label for="selected_number">Select Number:</label>
      <select name="selected_number" id="selected_number" required>
        <option value="">-- Choose --</option>
        <?php foreach($allNumbers as $n): ?>
          <option value="<?=htmlspecialchars($n)?>">
            <?=htmlspecialchars($n)?>
          </option>
        <?php endforeach; ?>
      </select>

      <label for="bid_amount">Bid Amount (₹):</label>
      <input type="number" name="bid_amount" id="bid_amount"
             step="0.01" min="10" required>

      <button type="submit" name="place_bid">Place Bid</button>
    </form>
  </div>

</body>
</html>

<?php
// close connection
$conn->close();
?>
