<?php
// dp_api.php — Double Panna JSON API

// ensure you’re using IST
date_default_timezone_set('Asia/Kolkata');

// 1) DEV: show all errors (disable in production)
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

// 2) JSON response header
header('Content-Type: application/json; charset=utf-8');

// 3) Include your DB connection
require_once __DIR__ . '/db.php';

// 4) Read & merge JSON body into $_REQUEST
$raw = file_get_contents('php://input');
if ($json = @json_decode($raw, true)) {
    $_REQUEST = array_merge($_REQUEST, $json);
}

// 5) Pull & validate inputs
$mobile       = trim($_REQUEST['mobile']       ?? '');
$game_id      = (int)($_REQUEST['game_id']     ?? 0);
$game_name    = trim($_REQUEST['game_name']    ?? '');
$bidsArr      = $_REQUEST['bids']              ?? [];

if (
    $mobile === '' ||
    $game_id <= 0 ||
    $game_name === '' ||
    !is_array($bidsArr)
) {
    http_response_code(400);
    exit(json_encode([
      'status'  => false,
      'message' => 'Required: mobile, game_id, game_name, bids (array)'
    ]));
}

try {
    // 6) Lookup user_id by mobile
    $stmt = $conn->prepare("
      SELECT user_id
        FROM users_data
       WHERE mobile = ?
       LIMIT 1
    ");
    $stmt->bind_param('s', $mobile);
    $stmt->execute();
    $stmt->bind_result($user_id);
    if (!$stmt->fetch()) {
        http_response_code(404);
        exit(json_encode(['status'=>false,'message'=>'User not found.']));
    }
    $stmt->close();

    // 7) Lookup game record by game_number + game_name
   $stmt = $conn->prepare("
  SELECT 
    id,
    open_time,
    close_time,
    open_bid_last_time,
    close_bid_last_time
  FROM game56
  WHERE id = ?
    AND LOWER(game_name) = LOWER(?)
  LIMIT 1
");
$stmt->bind_param('is', $game_id, $game_name);
$stmt->execute();
$stmt->bind_result(
  $game_id,
  $open_time,
  $close_time,
  $open_bid_last_time,
  $close_bid_last_time
);
    if (!$stmt->fetch()) {
        http_response_code(404);
        exit(json_encode(['status'=>false,'message'=>'Game not found or name/ID mismatch.']));
    }
    $stmt->close();
    
    // helper to turn "HH:MM:SS" into "h:mm A IST"
$fmt = function(string $t): string {
    return date('g:i A', strtotime($t)) . ' IST';
};

$open_f          = $fmt($open_time);
$close_f         = $fmt($close_time);
$open_bid_last_f = $fmt($open_bid_last_time);
$close_bid_last_f= $fmt($close_bid_last_time);


    // 8) Fetch current wallet balance
    $stmt = $conn->prepare("
      SELECT balance
        FROM wallet
       WHERE user_id = ?
    ");
    $stmt->bind_param('i', $user_id);
    $stmt->execute();
    $stmt->bind_result($wallet_balance);
    $stmt->fetch();
    $stmt->close();

    // 9) Clean & sum bids
    $min_bid   = 10;
    $total     = 0;
    $cleanBids = [];
    foreach ($bidsArr as $item) {
        if (!is_array($item) || !isset($item['number'], $item['amount'])) {
            continue;
        }
        $n = preg_replace('/\D/', '', (string)$item['number']);
        $a = floatval($item['amount']);
        if ($n === '' || $a < $min_bid) continue;
        $cleanBids[$n] = $a;
        $total += $a;
    }
    if (empty($cleanBids)) {
        http_response_code(400);
        exit(json_encode([
          'status'=>false,
          'message'=>"Please place at least one bid of ₹{$min_bid}.",
        ]));
    }
    if ($total > $wallet_balance) {
        http_response_code(400);
        exit(json_encode([
          'status'=>false,
          'message'=>'Insufficient wallet balance.',
          'data'=>['wallet_balance'=>$wallet_balance]
        ]));
    }

    // 10) Compute session (open vs close) in IST
    date_default_timezone_set('Asia/Kolkata');
   // 10) Compute session using the “bid_last_time” fields
$today        = date('Y-m-d');
$nowTs        = time();
$openLastTs   = strtotime("$today $open_bid_last_time");
$closeLastTs  = strtotime("$today $close_bid_last_time");
if ($closeLastTs <= $openLastTs) {
    // if your close cutoff is past midnight
    $closeLastTs += 86400;
}

if ($nowTs <= $openLastTs) {
    $session = 'open';
} elseif ($nowTs <= $closeLastTs) {
    $session = 'close';
} else {
    // bidding window is over
    http_response_code(403);
    exit(json_encode([
      'status'  => false,
      'message' => 'Bidding for this game is closed'
    ]));
}


    // 11) Begin transaction
    $conn->begin_transaction();

    // 12) Insert into user_bids_dp
    $detailsJson = json_encode($cleanBids, JSON_NUMERIC_CHECK);
    $ins = $conn->prepare("
      INSERT INTO user_bids_dp
        (user_id, game_id, bid_details, total_bid_amount, bid_session, game_name, created_at)
      VALUES (?,?,?,?,?,?, NOW())
    ");
    $ins->bind_param(
      'iisdss',
      $user_id,
      $game_id,
      $detailsJson,
      $total,
      $session,
      $game_name
    );
    $ins->execute();
    $ins->close();

    // 13) Deduct from wallet
    $upd = $conn->prepare("
      UPDATE wallet
         SET balance = balance - ?
       WHERE user_id = ?
    ");
    $upd->bind_param('di', $total, $user_id);
    $upd->execute();
    $upd->close();

    // 14) Commit
    $conn->commit();

    // 15) Response
    $newBalance = $wallet_balance - $total;
    echo json_encode([
  'status'  => true,
  'message' => "Placed ₹{$total} on double panna ({$session}).",
  'data'    => [
    // wallet & session
    'wallet_balance'           => $newBalance,
    'session'                  => $session,

    // game-times
    'game_open_time'           => $open_time,
    'game_close_time'          => $close_time,
    'game_open_time_fmt'       => $open_f,
    'game_close_time_fmt'      => $close_f,

    // bid-cutoff times
    'bid_open_last_time'       => $open_bid_last_time,
    'bid_close_last_time'      => $close_bid_last_time,
    'bid_open_last_time_fmt'   => $open_bid_last_f,
    'bid_close_last_time_fmt'  => $close_bid_last_f,
  ]
]);


} catch (Exception $e) {
    $conn->rollback();
    http_response_code(500);
    echo json_encode([
      'status'  => false,
      'message' => 'Server error: ' . $e->getMessage()
    ]);
}

$conn->close();
