<?php
// Debugging (remove in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();
include 'db.php';

// Redirect if not logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

date_default_timezone_set('Asia/Kolkata');

// Resolve target date
$today = date('Y-m-d');
$targetDate = isset($_GET['d']) ? trim($_GET['d']) : $today;
$dt = DateTime::createFromFormat('Y-m-d', $targetDate);
if (!$dt || $dt->format('Y-m-d') !== $targetDate) {
    $targetDate = $today;
}

// Determine reference "now"
if ($targetDate === $today) {
    $refNow = time();
} elseif ($targetDate < $today) {
    $refNow = strtotime($targetDate . ' 23:59:59');
} else {
    $refNow = strtotime($targetDate . ' 00:00:00');
}

// Debug: Check table structure
function debug_table_structure(mysqli $conn, $tableName) {
    echo "<div style='background: #f8f9fa; padding: 10px; margin: 10px; border-left: 4px solid #007bff;'>";
    echo "<h4>Debug: $tableName table structure</h4>";
    
    $result = $conn->query("SHOW COLUMNS FROM $tableName");
    if ($result) {
        echo "<ul>";
        while ($row = $result->fetch_assoc()) {
            echo "<li>" . $row['Field'] . " (" . $row['Type'] . ")</li>";
        }
        echo "</ul>";
        $result->free();
    } else {
        echo "<p>Error: " . $conn->error . "</p>";
    }
    echo "</div>";
}

// Debug: Show sample data
function debug_sample_data(mysqli $conn, $tableName, $limit = 3) {
    echo "<div style='background: #f8f9fa; padding: 10px; margin: 10px; border-left: 4px solid #28a745;'>";
    echo "<h4>Debug: Sample data from $tableName</h4>";
    
    $result = $conn->query("SELECT * FROM $tableName LIMIT $limit");
    if ($result && $result->num_rows > 0) {
        echo "<table border='1' cellpadding='5' style='border-collapse: collapse;'>";
        
        // Get column names
        $fields = $result->fetch_fields();
        echo "<tr>";
        foreach ($fields as $field) {
            echo "<th>" . $field->name . "</th>";
        }
        echo "</tr>";
        
        // Show data
        $result->data_seek(0);
        while ($row = $result->fetch_assoc()) {
            echo "<tr>";
            foreach ($row as $value) {
                echo "<td>" . htmlspecialchars(substr($value, 0, 50)) . (strlen($value) > 50 ? '...' : '') . "</td>";
            }
            echo "</tr>";
        }
        echo "</table>";
        $result->free();
    } else {
        echo "<p>No data found or error: " . $conn->error . "</p>";
    }
    echo "</div>";
}

// Enhanced date column detection
function detect_date_column(mysqli $conn): ?string {
    $possibleColumns = ['created_at', 'bid_date', 'timestamp', 'date_created'];
    
    foreach ($possibleColumns as $col) {
        $result = $conn->query("SHOW COLUMNS FROM user_bids_dp LIKE '$col'");
        if ($result && $result->num_rows > 0) {
            $result->free();
            return $col;
        }
        if ($result) $result->free();
    }
    return null;
}

$dateCol = detect_date_column($conn);

// Format currency
$fmt = function($v) {
    if (!is_numeric($v)) return $v;
    return (fmod((float)$v, 1.0) === 0.0) ? number_format((int)$v) : number_format((float)$v, 2);
};

// Show debug info if requested
$showDebug = isset($_GET['debug']) && $_GET['debug'] == '1';
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Double Paana Report</title>
  <style>
    * { box-sizing: border-box; margin: 0; padding: 0; }
    body { font-family: Arial, sans-serif; background: #f0f2f5; color: #333; }
    .navbar {
      background: #007bff;
      color: #fff;
      text-align: center;
      position: relative;
      padding: 12px 8px;
    }
    .navbar h2 { font-size: 1.2rem; }
    .back-button {
      position: absolute;
      left: 8px;
      top: 50%;
      transform: translateY(-50%);
      background: #28a745;
      color: #fff;
      padding: 6px 10px;
      text-decoration: none;
      border-radius: 4px;
      font-size: 0.85rem;
    }
    .back-button:hover { background: #218838; }

    .filter-bar {
      background:#fff; border-bottom:1px solid #e5e5e5; padding:10px 12px;
      display:flex; align-items:center; justify-content:center; gap:8px; flex-wrap:wrap;
    }
    .filter-bar form { display:flex; align-items:center; gap:8px; flex-wrap:wrap; }
    .filter-bar label { font-weight:bold; }
    .filter-bar input[type="date"] {
      padding:6px 8px; border:1px solid #ccc; border-radius:4px;
    }
    .btn {
      padding:6px 10px; border:none; border-radius:4px; cursor:pointer;
      font-size:0.9rem; text-decoration:none; display:inline-block;
    }
    .btn-primary { background:#007BFF; color:#fff; }
    .btn-secondary { background:#6c757d; color:#fff; }
    .btn-debug { background:#17a2b8; color:#fff; }

    .meta {
      text-align:center; font-size:0.9rem; color:#333; margin:10px 0;
    }

    .content {
      max-width: 1000px;
      margin: 8px auto 16px auto;
      padding: 0 12px;
      display: grid;
      grid-gap: 16px;
      grid-template-columns: repeat(auto-fit, minmax(260px, 1fr));
    }

    .game-card {
      background: #fff;
      border-radius: 10px;
      overflow: hidden;
      box-shadow: 0 2px 8px rgba(0,0,0,0.1);
      display: flex;
      flex-direction: column;
      transition: transform .2s, box-shadow .2s;
    }
    .game-card:hover {
      transform: translateY(-3px);
      box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    }

    .game-card h3 {
      margin: 0;
      padding: 14px;
      background: linear-gradient(135deg, #007bff, #0056b3);
      color: #fff;
      font-size: 1rem;
      text-align: center;
    }

    .body {
      padding: 12px;
      flex: 1;
      font-size: 0.9rem;
      line-height: 1.4;
    }
    .body p.status-message {
      color: #d9534f;
      font-weight: bold;
    }

    .minmax {
      margin-top: 10px;
      font-size: 0.9rem;
    }
    .minmax .highest {
      background: #d4edda;
      padding: 6px;
      margin-bottom: 5px;
      border-radius: 4px;
      display: flex;
      justify-content: space-between;
    }
    .minmax .lowest {
      background: #f8d7da;
      padding: 6px;
      border-radius: 4px;
      display: flex;
      justify-content: space-between;
    }
    
    .debug-info {
      background: #fff3cd;
      border: 1px solid #ffeaa7;
      padding: 15px;
      margin: 10px;
      border-radius: 5px;
    }
  </style>
</head>
<body>

<div class="navbar">
  <a href="dashboard.php" class="back-button">Back</a>
  <h2>Double Paana - Highest & Lowest Bids</h2>
</div>

<div class="filter-bar">
  <form method="get" action="">
    <label for="d">Select Date:</label>
    <input type="date" id="d" name="d" value="<?= htmlspecialchars($targetDate); ?>" required>
    <button type="submit" class="btn btn-primary">Apply</button>
    <a class="btn btn-secondary" href="?d=<?= htmlspecialchars($today); ?>">Today</a>
    <a class="btn btn-debug" href="?d=<?= htmlspecialchars($targetDate); ?>&debug=1">Debug</a>
  </form>
</div>

<div class="meta">
  Showing data for: <strong><?= htmlspecialchars($targetDate); ?></strong>
  <?php if (!$dateCol): ?>
    <div style="color:#a94442;margin-top:6px;">
      ⚠️ <strong>Warning:</strong> No date column (created_at/bid_date) found in user_bids_dp table.
      Totals will include ALL bids for each game regardless of date.
    </div>
  <?php else: ?>
    <div style="color:#28a745;margin-top:6px;">
      ✅ Using date column: <strong><?= $dateCol ?></strong>
    </div>
  <?php endif; ?>
</div>

<?php if ($showDebug): ?>
  <div class="debug-info">
    <h3>🔍 Debug Information</h3>
    <p><strong>Target Date:</strong> <?= $targetDate ?></p>
    <p><strong>Date Column Found:</strong> <?= $dateCol ?: 'None' ?></p>
    <p><strong>Reference Time:</strong> <?= date('Y-m-d H:i:s', $refNow) ?></p>
    
    <?php 
    debug_table_structure($conn, 'user_bids_dp');
    debug_sample_data($conn, 'user_bids_dp', 3);
    debug_table_structure($conn, 'game56');
    debug_sample_data($conn, 'game56', 3);
    ?>
  </div>
<?php endif; ?>

<div class="content">
<?php 
// Fetch games
$sql = "SELECT * FROM game56";
$result = $conn->query($sql);

if ($result && $result->num_rows > 0): ?>
  <?php while ($row = $result->fetch_assoc()): ?>
    <?php
      // Calculate game status
      $open_ts = strtotime($targetDate . ' ' . $row['open_time']);
      $open_bid_ts = strtotime($targetDate . ' ' . $row['open_bid_last_time']);
      $close_ts = strtotime($targetDate . ' ' . $row['close_time']);
      $close_bid_ts = strtotime($targetDate . ' ' . $row['close_bid_last_time']);

      if ($targetDate > $today) {
          $status_text = "Game has not started";
      } elseif ($targetDate < $today) {
          $status_text = "Game has Closed";
      } else {
          if ($refNow < $open_bid_ts) {
              $status_text = "Game is open for bid";
          } elseif ($refNow < $open_ts) {
              $status_text = "Open bidding is closed";
          } elseif ($refNow < $close_bid_ts) {
              $status_text = "Game is Running";
          } elseif ($refNow < $close_ts) {
              $status_text = "Close bidding is closed";
          } else {
              $status_text = "Game has Closed";
          }
      }

      // Update game status in database
      if ($targetDate === $today) {
          if ($upd = $conn->prepare("UPDATE game56 SET status = ? WHERE id = ?")) {
              $upd->bind_param('si', $status_text, $row['id']);
              $upd->execute();
              $upd->close();
          }
      }

      // Initialize totals array - using associative array to handle any number format
      $totals = [];
      $totalBidsFound = 0;

      // Build query based on available date column
      if ($dateCol) {
          $stmt = $conn->prepare("SELECT bid_details FROM user_bids_dp WHERE game_name = ? AND DATE($dateCol) = ?");
          if ($stmt) {
              $stmt->bind_param('ss', $row['game_name'], $targetDate);
          }
      } else {
          // Fallback: get all bids for this game (no date filter)
          $stmt = $conn->prepare("SELECT bid_details FROM user_bids_dp WHERE game_name = ?");
          if ($stmt) {
              $stmt->bind_param('s', $row['game_name']);
          }
      }

      if ($stmt && $stmt->execute()) {
          $bids = $stmt->get_result();
          while ($b = $bids->fetch_assoc()) {
              $totalBidsFound++;
              $data = json_decode($b['bid_details'], true);
              
              if (!$data || !is_array($data)) {
                  continue;
              }
              
              foreach ($data as $num => $amt) {
                  if (is_numeric($amt) && $amt > 0) {
                      $numStr = (string)$num;
                      if (!isset($totals[$numStr])) {
                          $totals[$numStr] = 0.0;
                      }
                      $totals[$numStr] += floatval($amt);
                  }
              }
          }
          $bids->free();
      }
      if ($stmt) $stmt->close();

      // Calculate highest and lowest
      if (empty($totals)) {
          $maxVal = 0;
          $maxIndex = '-';
          $minVal = 0;
          $minIndex = '-';
      } else {
          $maxVal = max($totals);
          $maxIndex = array_search($maxVal, $totals);
          $minVal = min($totals);
          $minIndex = array_search($minVal, $totals);
      }
    ?>
    <div class="game-card">
      <h3><?= htmlspecialchars($row['game_name']) ?> (<?= htmlspecialchars($row['game_number'] ?? '') ?>)</h3>
      <div class="body">
        <p><strong>Open Result:</strong> <?= date("g:i A", $open_ts) ?> IST</p>
        <p><strong>Close Result:</strong> <?= date("g:i A", $close_ts) ?> IST</p>
        <p class="status-message"><?= $status_text ?></p>
        
        <?php if ($showDebug): ?>
          <div style="background: #e7f3ff; padding: 8px; margin: 8px 0; border-radius: 4px; font-size: 0.8rem;">
            <strong>Debug Info:</strong><br>
            • Total bids found: <?= $totalBidsFound ?><br>
            • Date column: <?= $dateCol ?: 'None' ?><br>
            • Numbers with bids: <?= empty($totals) ? 'None' : implode(', ', array_keys($totals)) ?><br>
            • Bid amounts: <?= empty($totals) ? 'None' : '[' . implode(', ', array_map($fmt, $totals)) . ']' ?>
          </div>
        <?php endif; ?>
        
        <div class="minmax">
          <div class="highest">
            <span>Highest bidding number:</span>
            <span><?= htmlspecialchars((string)$maxIndex) ?> (₹<?= $fmt($maxVal) ?>)</span>
          </div>
          <div class="lowest">
            <span>Least bidding number:</span>
            <span><?= htmlspecialchars((string)$minIndex) ?> (₹<?= $fmt($minVal) ?>)</span>
          </div>
        </div>
        
        <?php if ($totalBidsFound == 0): ?>
          <div style="background: #fff3cd; padding: 8px; margin: 8px 0; border-radius: 4px; font-size: 0.85rem;">
            ⚠️ No bids found for this game on <?= $targetDate ?>
          </div>
        <?php endif; ?>
      </div>
    </div>
  <?php endwhile; ?>
<?php else: ?>
  <div class="debug-info">
    <h3>❌ No Games Found</h3>
    <p>No Double Paana games found in the game56 table.</p>
    <p><strong>SQL Error:</strong> <?= $conn->error ?: 'None' ?></p>
  </div>
<?php endif; ?>
</div>

</body>
</html>