<?php
// fs_api.php — accept multiple Full Sangam bids in one call

header('Content-Type: application/json; charset=utf-8');
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

require_once __DIR__ . '/db.php';
$conn->set_charset('utf8mb4');

// 1) parse JSON
$in = json_decode(file_get_contents('php://input'), true);
if (!$in || json_last_error() !== JSON_ERROR_NONE) {
  http_response_code(400);
  exit(json_encode(['status'=>false, 'message'=>'Invalid JSON']));
}

$mobile      = trim($in['mobile']       ?? '');
$game_id = intval($in['game_id'] ?? 0);
$game_name   = trim($in['game_name']    ?? '');
$bidsArr     = $in['bids']              ?? [];

if ($mobile === '' || $game_id <= 0 || $game_name === '' || !is_array($bidsArr)) {
  http_response_code(400);
  exit(json_encode(['status'=>false, 'message'=>'Required: mobile, game_id, game_name & bids(array).']));
}

// 2) Clean & sum bids
$minBid = 10;
$clean  = [];
$total  = 0.0;
foreach ($bidsArr as $b) {
  if (!is_array($b)) continue;
  $open  = trim($b['open_panna']  ?? '');
  $close = trim($b['close_panna'] ?? '');
  $amt   = floatval($b['amount'] ?? 0);
  if ($open === '' || $close === '' || $amt < $minBid) continue;
  $clean[] = ['open'=>$open,'close'=>$close,'amount'=>$amt];
  $total  += $amt;
}

if (empty($clean)) {
  http_response_code(400);
  exit(json_encode(['status'=>false,'message'=>"Min bid ₹{$minBid} & both picks required."]));
}

// 3) Lookup user
$stmt = $conn->prepare("SELECT user_id FROM users_data WHERE mobile = ? LIMIT 1");
$stmt->bind_param('s', $mobile);
$stmt->execute();
$stmt->bind_result($user_id);
if (!$stmt->fetch()) {
  http_response_code(404);
  exit(json_encode(['status'=>false,'message'=>'User not found.']));
}
$stmt->close();

// 4) Lookup game + all four timestamps
$game_id = intval($in['game_id'] ?? 0);
if ($game_id <= 0) {
  http_response_code(400);
  exit(json_encode(['status'=>false,'message'=>'`game_id` is required.']));
}

$stmt = $conn->prepare("
  SELECT 
    id,
    open_time,
    close_time,
    open_bid_last_time,
    close_bid_last_time
  FROM game56
  WHERE id = ?
    AND LOWER(game_name) = LOWER(?)
  LIMIT 1
");
$stmt->bind_param('is', $game_id, $game_name);
$stmt->execute();
$stmt->store_result();
if ($stmt->num_rows !== 1) {
  http_response_code(404);
  exit(json_encode(['status'=>false,'message'=>'Game not found.']));
}
$stmt->bind_result(
  $game_id,
  $open_time,
  $close_time,
  $open_bid_last,
  $close_bid_last
);
$stmt->fetch();
$stmt->close();

// 5) Compute the 5-state status_text
date_default_timezone_set('Asia/Kolkata');
$nowTs      = time();
$tsOpenBid  = strtotime($open_bid_last);
$tsOpenRes  = strtotime($open_time);
$tsCloseBid = strtotime($close_bid_last);
$tsCloseRes = strtotime($close_time);

if ($nowTs < $tsOpenBid) {
    $status_text = 'Game is open for bid';
}
elseif ($nowTs < $tsOpenRes) {
    $status_text = 'Open bidding is closed';
}
elseif ($nowTs < $tsCloseBid) {
    $status_text = 'Game is Running';
}
elseif ($nowTs < $tsCloseRes) {
    $status_text = 'Close bidding is closed';
}
else {
    $status_text = 'Game has Closed';
}

// 6) Reject until open-for-bid


// 7) Check wallet balance
$stmt = $conn->prepare("SELECT balance FROM wallet WHERE user_id = ?");
$stmt->bind_param('i', $user_id);
$stmt->execute();
$stmt->bind_result($wallet_balance);
$stmt->fetch();
$stmt->close();

if ($total > $wallet_balance) {
  http_response_code(400);
  exit(json_encode([
    'status'=>false,
    'message'=>'Insufficient balance.',
    'data'=>['wallet_balance'=>$wallet_balance]
  ]));
}

// 8) Insert bids & deduct balance

$conn->begin_transaction();
try {
  $ins = $conn->prepare("
    INSERT INTO user_bids_fs
      (user_id, game_id, bid_open_fs, bid_close_fs, total_bid_amount, bid_session)
    VALUES (?,?,?,?,?,?)
  ");
  foreach ($clean as $b) {
    $ins->bind_param(
      'iissds',
      $user_id,
      $game_id,
      $b['open'],
      $b['close'],
      $b['amount'],
      $status_text  // or use a simpler 'open'/'close' if you prefer
    );
    $ins->execute();
  }
  $ins->close();

  $upd = $conn->prepare("UPDATE wallet SET balance = balance - ? WHERE user_id = ?");
  $upd->bind_param('di', $total, $user_id);
  $upd->execute();
  $upd->close();

  $newBal = $wallet_balance - $total;
  $conn->commit();

  echo json_encode([
    'status'=>true,
    'message'=>"Placed ₹{$total} in “{$status_text}” session.",
    'data'    => [
    'wallet_balance'      => $newBal,
    'open_time'           => $open_time,
    'close_time'          => $close_time,
    'open_bid_last_time'  => $open_bid_last,
    'close_bid_last_time' => $close_bid_last,
    'session'             => $status_text,
  ]
  ]);
} catch (Exception $e) {
  $conn->rollback();
  http_response_code(500);
  echo json_encode(['status'=>false,'message'=>'Server error: '.$e->getMessage()]);
}

$conn->close();
