<?php
// games_api.php
// 1) Enable error reporting (disable in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// 2) CORS & JSON response headers
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Headers: Content-Type");
header("Content-Type: application/json; charset=UTF-8");

// 3) Include DB connection (defines $conn = new mysqli(...))
require_once __DIR__ . '/db.php';

// 4) Decode incoming JSON POST body
$input = json_decode(file_get_contents('php://input'), true);
$mobile = trim($input['mobile'] ?? '');

if ($mobile === '') {
    http_response_code(400);
    exit(json_encode([
        'success' => false,
        'message' => 'Please provide a valid mobile number.'
    ]));
}

// 5) Lookup the user_id by mobile
if ($stmt = $conn->prepare("SELECT user_id FROM users_data WHERE mobile = ? LIMIT 1")) {
    $stmt->bind_param('s', $mobile);
    $stmt->execute();
    $stmt->store_result();
    
    if ($stmt->num_rows !== 1) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'message' => 'User not found.'
        ]);
        exit;
    }
    
    $stmt->bind_result($user_id);
    $stmt->fetch();
    $stmt->close();
} else {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Database error during user lookup.'
    ]);
    exit;
}

// 6) Fetch all games (including bid-deadlines)
date_default_timezone_set('Asia/Kolkata');

// ===== NEW: WEEKEND CLOSURE LOGIC =====
$currentDay = date('N'); // 1=Monday, 6=Saturday, 7=Sunday

// Define games that close on Saturday and Sunday
$saturdaySundayClosedGames = [
    'KALYAN NIGHT',
    'RAJDHANI NIGHT', 
    'MAIN BAZAR'
];

// Define games that close only on Sunday
$sundayClosedGames = [
    'TIME BAZAR',
    'RAJDHANI DAY',
    'MILAN DAY',
    'KALYAN',
    'MILAN NIGHT'
];

// Function to check if a game should be closed today
function isGameClosedToday($gameName, $currentDay, $satSunClosed, $sunOnlyClosed) {
    $gameNameUpper = strtoupper(trim($gameName));
    
    // Check Saturday and Sunday closures
    if (($currentDay == 6 || $currentDay == 7)) { // Saturday or Sunday
        foreach ($satSunClosed as $closedGame) {
            if (strpos($gameNameUpper, strtoupper($closedGame)) !== false) {
                return true;
            }
        }
    }
    
    // Check Sunday only closures
    if ($currentDay == 7) { // Sunday only
        foreach ($sunOnlyClosed as $closedGame) {
            if (strpos($gameNameUpper, strtoupper($closedGame)) !== false) {
                return true;
            }
        }
    }
    
    return false;
}
// ===== END WEEKEND CLOSURE LOGIC =====

$sql = "
  SELECT 
    id,
    game_name,
    game_number,
    open_time,
    close_time,
    open_bid_last_time,
    close_bid_last_time
  FROM game56
";

$result = $conn->query($sql);
if (!$result) {
    http_response_code(500);
    exit(json_encode([
        'success' => false,
        'message' => 'Database error fetching games.'
    ]));
}

$games = [];
$now = time();

while ($row = $result->fetch_assoc()) {
    $openBidLast  = strtotime($row['open_bid_last_time']);
    $openResult   = strtotime($row['open_time']);
    $closeBidLast = strtotime($row['close_bid_last_time']);
    $closeResult  = strtotime($row['close_time']);
    
    // ===== NEW: CHECK WEEKEND CLOSURE FIRST =====
    $isClosedToday = isGameClosedToday(
        $row['game_name'], 
        $currentDay, 
        $saturdaySundayClosedGames, 
        $sundayClosedGames
    );
    
    if ($isClosedToday) {
        $status_text = 'Game is closed for today';
        $disabled = true;
    } else {
        // ===== ORIGINAL TIME-BASED LOGIC =====
        if ($now < $openBidLast) {
            $status_text = 'Game is open for bid';
            $disabled = false;
        }
        elseif ($now < $openResult) {
            $status_text = 'Open bidding is closed';
            $disabled = true;
        }
        elseif ($now < $closeBidLast) {
            $status_text = 'Game is Running';
            $disabled = false;
        }
        elseif ($now < $closeResult) {
            $status_text = 'Close bidding is closed';
            $disabled = true;
        }
        else {
            $status_text = 'Game has Closed';
            $disabled = true;
        }
    }
    
    // optionally persist status back to DB
    if ($upd = $conn->prepare("UPDATE game56 SET status = ? WHERE id = ?")) {
        $upd->bind_param('si', $status_text, $row['id']);
        $upd->execute();
        $upd->close();
    }
    
    $games[] = [
        'game_id'        => (int)$row['id'],
        'game_name'      => $row['game_name'],
        'game_number'    => $row['game_number'],
        'open'           => date("g:i A", $openResult) . " IST",
        'close'          => date("g:i A", $closeResult) . " IST",
        'open_bid_last'  => date("g:i A", $openBidLast) . " IST",
        'close_bid_last' => date("g:i A", $closeBidLast) . " IST",
        'status_text'    => $status_text,
        'disabled'       => $disabled,
    ];
}

echo json_encode([
    'success' => true,
    'data' => $games
]);

$conn->close();
?>