<?php
session_start();
ini_set('display_errors', 1);
error_reporting(E_ALL);

include 'db.php';

// Redirect if not logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

date_default_timezone_set('Asia/Kolkata');

/**
 * Resolve target date (YYYY-MM-DD). Defaults to today. Sanitizes bad input.
 */
$today = date('Y-m-d');
$targetDate = isset($_GET['d']) ? trim($_GET['d']) : $today;
$dt = DateTime::createFromFormat('Y-m-d', $targetDate);
if (!$dt || $dt->format('Y-m-d') !== $targetDate) {
    $targetDate = $today; // fallback if invalid
}

/**
 * Determine reference "now" for status logic:
 * - viewing today  -> real now
 * - viewing past   -> 23:59:59 of that date (so it's closed)
 * - viewing future -> 00:00:00 of that date (so it's not started)
 */
if ($targetDate === $today) {
    $refNow = time();
} elseif ($targetDate < $today) {
    $refNow = strtotime($targetDate . ' 23:59:59');
} else {
    $refNow = strtotime($targetDate . ' 00:00:00');
}

/**
 * Detect which date column to use for filtering user_bids_hs.
 * Prefers created_at, then bid_date; fallback = no date filter if neither exists.
 */
function detect_date_column(mysqli $conn): ?string {
    $col = null;
    if ($res = $conn->query("SHOW COLUMNS FROM user_bids_hs LIKE 'created_at'")) {
        if ($res->num_rows > 0) { $col = 'created_at'; }
        $res->free();
    }
    if (!$col && ($res2 = $conn->query("SHOW COLUMNS FROM user_bids_hs LIKE 'bid_date'"))) {
        if ($res2->num_rows > 0) { $col = 'bid_date'; }
        $res2->free();
    }
    return $col;
}
$dateCol = detect_date_column($conn);

/**
 * Currency formatting helper (no decimals if integer).
 */
$fmt = function($v) {
    if (!is_numeric($v)) return $v;
    return (fmod((float)$v, 1.0) === 0.0) ? number_format((int)$v) : number_format((float)$v, 2);
};

// Fetch games from game56 (sorted by open time)
$sql = "SELECT * FROM game56 ORDER BY open_time ASC";
$result = $conn->query($sql);

// Half Sangam number groups
$singleDigits = ['0','1','2','3','4','5','6','7','8','9'];
$triplePanna  = ['000','111','222','333','444','555','666','777','888','999'];

// Single & Double Panna arrays (from your Half Sangam master list)
$singlePanna = [127,136,145,190,235,280,370,389,460,479,569,578,128,137,146,236,245,290,380,470,489,560,579,678,129,138,147,156,237,246,345,390,480,570,589,679,120,139,148,157,238,247,256,346,490,580,670,689,130,149,158,167,239,248,257,347,356,590,680,789,140,159,168,230,249,258,267,348,357,456,690,780,123,150,169,178,240,259,268,349,358,367,457,790,124,160,178,179,250,269,340,359,368,458,467,890,125,134,170,189,260,279,350,369,468,378,459,567,126,135,180,234,270,289,360,379,450,469,478,568];

$doublePanna = [118,226,244,299,334,488,550,668,677,100,119,155,227,335,344,399,588,669,110,200,228,255,336,499,660,688,778,166,229,300,337,355,445,599,779,788,112,220,266,338,400,446,455,699,770,113,122,177,339,366,447,500,799,889,114,224,277,330,448,466,556,880,899,115,133,158,223,377,449,557,566,700,116,224,233,288,440,477,558,800,990,117,144,199,225,388,559,577,667,900];

// Merge all for checking example no-bid combinations
$allNumbers = array_merge($singleDigits, $singlePanna, $doublePanna, $triplePanna);
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Half Sangam Report - Highest & Lowest Bids</title>
<style>
body { background:#f8f9fa; font-family:Arial,sans-serif; margin:0; }
.navbar { background:#007BFF; color:#fff; text-align:center; padding:12px; position:relative; }
.navbar h2 { margin:0; font-size:1.1rem; }
.back-button {
  position:absolute; left:10px; top:50%; transform:translateY(-50%);
  background:#28a745; color:#fff; padding:6px 12px; text-decoration:none; border-radius:4px;
}
.filter-bar {
  background:#fff; border-bottom:1px solid #e5e5e5; padding:10px 12px;
  display:flex; align-items:center; justify-content:center; gap:8px; flex-wrap:wrap;
}
.filter-bar form { display:flex; align-items:center; gap:8px; flex-wrap:wrap; }
.filter-bar label { font-weight:bold; }
.filter-bar input[type="date"] { padding:6px 8px; border:1px solid #ccc; border-radius:4px; }
.btn { padding:6px 10px; border:none; border-radius:4px; cursor:pointer; font-size:0.9rem; text-decoration:none; display:inline-block; }
.btn-primary { background:#007BFF; color:#fff; }
.btn-secondary { background:#6c757d; color:#fff; }
.meta { text-align:center; font-size:0.9rem; color:#333; margin:10px 0; }
.content {
  display:grid; grid-template-columns:repeat(auto-fit,minmax(260px,1fr));
  gap:16px; max-width:1000px; margin:20px auto; padding:0 12px;
}
.game-card {
  background:#fff; border-radius:8px; border:2px solid #007BFF;
  box-shadow:0 2px 6px rgba(0,0,0,0.1); overflow:hidden;
}
.game-card h3 { background:#007BFF; color:#fff; padding:10px; margin:0; text-align:center; }
.body { padding:12px; text-align:center; }
.status-message { color:#d9534f; font-weight:bold; margin-bottom:10px; }
.minmax { display:flex; flex-direction:column; gap:8px; }
.highest, .lowest, .nobid {
  padding:8px; border-radius:4px; display:flex; justify-content:space-between; align-items:center;
}
.highest { background:#d4edda; }
.lowest  { background:#f8d7da; }
.nobid   { background:#fff3cd; }
@media (max-width:640px){ .back-button { font-size:0.85rem; padding:5px 9px; } }
</style>
</head>
<body>

<div class="navbar">
  <a href="dashboard.php" class="back-button">Back</a>
  <h2>Half Sangam - Highest & Lowest Bids</h2>
</div>

<div class="filter-bar">
  <form method="get" action="">
    <label for="d">Select Date:</label>
    <input type="date" id="d" name="d" value="<?php echo htmlspecialchars($targetDate); ?>" required>
    <button type="submit" class="btn btn-primary">Apply</button>
    <a class="btn btn-secondary" href="?d=<?php echo htmlspecialchars($today); ?>">Today</a>
  </form>
</div>

<div class="meta">
  Showing data for: <strong><?php echo htmlspecialchars($targetDate); ?></strong>
  <?php if (!$dateCol): ?>
    <div style="color:#a94442;margin-top:6px;">
      Note: No date column (created_at/bid_date) found in <code>user_bids_hs</code>; totals include all bids for each game.
    </div>
  <?php endif; ?>
</div>

<div class="content">
<?php if ($result && $result->num_rows > 0): ?>
  <?php while ($row = $result->fetch_assoc()): ?>
    <?php
      $gameName = $row['game_name'];

      // Build timestamps for the selected date for status display
      $open_ts  = strtotime($targetDate . ' ' . $row['open_time']);
      $close_ts = strtotime($targetDate . ' ' . $row['close_time']);

      // Status relative to selected date
      if ($targetDate > $today) {
          $status_text = "Game has not started";
      } elseif ($targetDate < $today) {
          $status_text = "Game has Closed";
      } else {
          $status_text = ($refNow >= $open_ts && $refNow <= $close_ts) ? "Game is Open" : "Game has Closed";
      }

      // Fetch bids for this game (filtered by date if we detected a date column)
      if ($dateCol) {
          $stmt = $conn->prepare("SELECT bid_open_hs, bid_close_hs, total_bid_amount FROM user_bids_hs WHERE game_name = ? AND DATE($dateCol) = ?");
          $stmt->bind_param('ss', $gameName, $targetDate);
      } else {
          $stmt = $conn->prepare("SELECT bid_open_hs, bid_close_hs, total_bid_amount FROM user_bids_hs WHERE game_name = ?");
          $stmt->bind_param('s', $gameName);
      }

      $totals = [];
      if ($stmt && $stmt->execute()) {
          $resBids = $stmt->get_result();
          while ($b = $resBids->fetch_assoc()) {
              $openNum  = $b['bid_open_hs'];
              $closeNum = $b['bid_close_hs'];
              $amt      = (float)$b['total_bid_amount'];

              // Build combination key "OPEN-CLOSE"
              $combination = $openNum . '-' . $closeNum;
              if (!isset($totals[$combination])) { $totals[$combination] = 0.0; }
              $totals[$combination] += $amt;
          }
          $resBids->free();
      }
      if ($stmt) { $stmt->close(); }

      // Highest & Lowest (non-zero minimum)
      if (!empty($totals)) {
          $maxVal   = max($totals);
          $maxIndex = array_search($maxVal, $totals, true);

          $nonZero  = array_filter($totals, fn($v) => $v > 0);
          $minVal   = $nonZero ? min($nonZero) : 0;
          $minIndex = $nonZero ? array_search($minVal, $totals, true) : '-';
      } else {
          $maxIndex = '-'; $maxVal = 0;
          $minIndex = '-'; $minVal = 0;
      }

      // Example no-bid combination (first OPEN-CLOSE pair not in totals)
      $bidKeys = array_keys($totals);
      $exampleNoBid = '-';
      foreach ($allNumbers as $numA) {
          foreach ($allNumbers as $numB) {
              $combo = $numA . '-' . $numB;
              if (!in_array($combo, $bidKeys, true)) {
                  $exampleNoBid = $combo;
                  break 2;
              }
          }
      }
    ?>
    <div class="game-card">
      <h3><?= htmlspecialchars($row['game_name']) ?> (<?= htmlspecialchars($row['game_number']) ?>)</h3>
      <div class="body">
        <p class="status-message"><?= htmlspecialchars($status_text) ?></p>
        <p>
          <strong>Open Time:</strong> <?= date('g:i A', $open_ts) ?> IST
          &nbsp;|&nbsp;
          <strong>Close Time:</strong> <?= date('g:i A', $close_ts) ?> IST
        </p>
        <div class="minmax">
          <div class="highest">
            <span>Highest bidding pair:</span>
            <span><?= htmlspecialchars((string)$maxIndex) ?> (₹<?= $fmt($maxVal) ?>)</span>
          </div>
          <div class="lowest">
            <span>Lowest bidding pair:</span>
            <span><?= htmlspecialchars((string)$minIndex) ?> (₹<?= $fmt($minVal) ?>)</span>
          </div>
          <div class="nobid">
            <span>Example no-bid pair:</span>
            <span><?= htmlspecialchars((string)$exampleNoBid) ?></span>
          </div>
        </div>
      </div>
    </div>
  <?php endwhile; ?>
<?php else: ?>
  <p style="text-align:center;margin-top:32px;">No games found.</p>
<?php endif; ?>
</div>

</body>
</html>
