<?php
// halfsangam.php
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

session_start();
date_default_timezone_set('Asia/Kolkata');

// 0) Auth
if (empty($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}
$user_id = (int)$_SESSION['user_id'];

require_once __DIR__ . '/db.php';
$conn->set_charset('utf8mb4');

// 1) Fetch game_id & info
if (empty($_GET['game_id']) || !ctype_digit($_GET['game_id'])) {
    die("No game selected.");
}
$game_id = (int)$_GET['game_id'];

$stmt = $conn->prepare("
  SELECT game_name, open_time, close_time
    FROM game56
   WHERE id = ?
");
$stmt->bind_param("i", $game_id);
$stmt->execute();
$stmt->bind_result($game_name, $open_time, $close_time);
if (!$stmt->fetch()) {
    die("Invalid game.");
}
$stmt->close();

// 2) Determine session (open/close)
$now = date('H:i:s');
if ($open_time < $close_time) {
    $session = ($now >= $open_time && $now <= $close_time) ? 'close' : 'open';
} else {
    $session = ($now >= $open_time || $now <= $close_time) ? 'close' : 'open';
}

// 3) Wallet balance
$balance = 0;
$stmt = $conn->prepare("SELECT balance FROM wallet WHERE user_id=?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$stmt->bind_result($balance);
$stmt->fetch();
$stmt->close();

// 4) Build all the number-sets

// 4.a Single Pana (3-digit combos)
$singlePana = [
  '0'=>['127','136','145','190','235','280','370','389','460','479','569','578'],
  '1'=>['128','137','146','236','245','290','380','470','489','560','579','678'],
  '2'=>['129','138','147','156','237','246','345','390','480','570','589','679'],
  '3'=>['120','139','148','157','238','247','256','346','490','580','670','689'],
  '4'=>['130','149','158','167','239','248','257','347','356','590','680','789'],
  '5'=>['140','159','168','230','249','258','267','348','357','456','690','780'],
  '6'=>['123','150','169','178','240','259','268','349','358','367','457','790'],
  '7'=>['124','160','178','179','250','269','340','359','368','458','467','890'],
  '8'=>['125','134','170','189','260','279','350','369','468','378','459','567'],
  '9'=>['126','135','180','234','270','289','360','379','450','469','478','568'],
];

// 4.b Double Pana
$doublePana = [
  '0'=>['118','226','244','299','334','488','550','668','677'],
  '1'=>['100','119','155','227','335','344','399','588','669'],
  '2'=>['110','200','228','255','336','499','660','688','778'],
  '3'=>['166','229','300','337','355','445','599','779','788'],
  '4'=>['112','220','266','338','400','446','455','699','770'],
  '5'=>['113','122','177','339','366','447','500','799','889'],
  '6'=>['114','224','277','330','448','466','556','880','899'],
  '7'=>['115','133','188','223','377','449','557','566','700'],
  '8'=>['116','224','233','288','440','477','558','800','990'],
  '9'=>['117','144','199','225','388','559','577','667','900'],
];

// 4.c Triple Pana
$triplePana = ['000','111','222','333','444','555','666','777','888','999'];

// 4.d Single Digit (0–9)
$singleDigit = array_map('strval', range(0,9));

// 4.e Assemble
$allSets = [
  'Single Digit' => $singleDigit,
  'Single Pana'  => array_reduce($singlePana,'array_merge',[]),
  'Double Pana'  => array_reduce($doublePana,'array_merge',[]),
  'Triple Pana'  => $triplePana,
];

// 5) Handle bid submission
$message = '';
if ($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['place_bid'])) {
    $open_sel  = trim($_POST['open_panna']  ?? '');
    $close_sel = trim($_POST['close_panna'] ?? '');
    $bid_amt   = floatval($_POST['bid_amount'] ?? 0);

    // === STRICT EMPTY CHECK ===
    if (
      ($open_sel === '' && $close_sel === '') ||
      ($open_sel !== '' && $close_sel !== '')
    ) {
        $message = "<div style='color:red;'>Please select exactly one: Open OR Close.</div>";
    }
    elseif (
      !in_array($open_sel,  array_merge(...array_values($allSets))) &&
      !in_array($close_sel, array_merge(...array_values($allSets)))
    ) {
        $message = "<div style='color:red;'>Invalid selection.</div>";
    }
    elseif ($bid_amt < 10) {
        $message = "<div style='color:red;'>Minimum bid is ₹10.</div>";
    }
    elseif ($bid_amt > $balance) {
        $message = "<div style='color:red;'>Insufficient balance.</div>";
    }
    else {
        $conn->begin_transaction();
        try {
            // === EMPTY-STRING FALLBACK ===
            $bid_open  = $open_sel  !== '' ? $open_sel  : '';
            $bid_close = $close_sel !== '' ? $close_sel : '';

            $ins = $conn->prepare("
              INSERT INTO user_bids_hs
                (user_id,game_id,bid_open_hs,bid_close_hs,total_bid_amount,bid_session)
              VALUES (?,?,?,?,?,?)
            ");
            $ins->bind_param(
              "iissds",
              $user_id,
              $game_id,
              $bid_open,
              $bid_close,
              $bid_amt,
              $session
            );
            $ins->execute();
            $ins->close();

            $upd = $conn->prepare("UPDATE wallet SET balance = balance - ? WHERE user_id=?");
            $upd->bind_param("di",$bid_amt,$user_id);
            $upd->execute();
            $upd->close();

            $conn->commit();
            header("Location: halfsangam.php?game_id={$game_id}");
            exit;
        } catch (Exception $e) {
            $conn->rollback();
            $message = "<div style='color:red;'>Error placing bid. Try again.</div>";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Half Sangam Bidding</title>
  <style>
    body { background:#f8f9fa; margin:0; font-family:Arial,sans-serif; }
    .wrap { max-width:900px; margin:1rem auto; padding:0 1rem; }
    h1 { text-align:center; }
    .info { text-align:center; margin-bottom:1rem; }
    .grid { display:grid; grid-template-columns:repeat(auto-fill,minmax(140px,1fr)); gap:1rem; margin-bottom:2rem; }
    .card { background:#fff; border:1px solid #f33; padding:.5rem; border-radius:6px; }
    .card h3 { margin:0 0 .5rem; font-size:1rem; color:#07f; text-align:center; }
    .card p { margin:0; font-size:.85rem; line-height:1.2; text-align:center; }
    .form { background:#fff; padding:1rem; border-radius:6px; box-shadow:0 1px 4px rgba(0,0,0,.1); max-width:500px; margin:0 auto; }
    .form label { display:block; margin:.5rem 0 .25rem; }
    .form select, .form input { width:100%; padding:.5rem; font-size:1rem; border:1px solid #ccc; border-radius:4px; }
    .form button { width:100%; padding:.75rem; background:#07f; color:#fff; border:none; border-radius:4px; font-size:1rem; cursor:pointer; margin-top:1rem; }
    .form button:hover { background:#0055aa; }
  </style>
</head>
<body>
  <div class="wrap">
    <h1>Half Sangam Bidding</h1>
    <div class="info">
      <div>Game: <strong><?= htmlspecialchars($game_name) ?></strong></div>
      <div>Session: <strong><?= ucfirst($session) ?></strong></div>
      <div>Wallet: <strong>₹<?= number_format($balance,2) ?></strong></div>
    </div>
    <div class="grid">
      <?php foreach ($allSets as $group => $nums): ?>
        <div class="card">
          <h3><?= $group ?></h3>
          <p><?= implode(', ', $nums) ?></p>
        </div>
      <?php endforeach ?>
    </div>
    <div class="form">
      <?= $message ?>
      <form method="POST">
        <label for="open_panna">Open Pick (choose ONE)</label>
        <select name="open_panna" id="open_panna">
          <option value="">-- none --</option>
          <?php foreach ($allSets as $grp => $nums): ?>
            <optgroup label="<?= $grp ?>">
              <?php foreach ($nums as $n): ?>
                <option><?= htmlspecialchars($n) ?></option>
              <?php endforeach ?>
            </optgroup>
          <?php endforeach ?>
        </select>
        <label for="close_panna">Close Pick (choose ONE)</label>
        <select name="close_panna" id="close_panna">
          <option value="">-- none --</option>
          <?php foreach ($allSets as $grp => $nums): ?>
            <optgroup label="<?= $grp ?>">
              <?php foreach ($nums as $n): ?>
                <option><?= htmlspecialchars($n) ?></option>
              <?php endforeach ?>
            </optgroup>
          <?php endforeach ?>
        </select>
        <label for="bid_amount">Bid Amount (₹)</label>
        <input type="number" name="bid_amount" id="bid_amount" min="10" step="0.01" required>
        <button type="submit" name="place_bid">Place Bid</button>
      </form>
    </div>
  </div>
</body>
</html>
