<?php
// hs_api.php — accept multiple Half Sangam bids in one call

header('Content-Type: application/json; charset=utf-8');

// 1) parse JSON
$raw = file_get_contents('php://input');
$in  = json_decode($raw, true);
if (!$in || json_last_error() !== JSON_ERROR_NONE) {
    http_response_code(400);
    exit(json_encode([
        'status'  => false,
        'message' => 'Invalid JSON payload.'
    ]));
}

// 2) extract & validate
$mobile      = trim($in['mobile']        ?? '');
$game_id     = intval($in['game_id']     ?? 0);
$game_name   = trim($in['game_name']     ?? '');
$bidsArr     = $in['bids']               ?? [];

if ($mobile === '' ||  $game_id <= 0 || $game_name === '' || !is_array($bidsArr)) {
    http_response_code(400);
    exit(json_encode([
        'status'  => false,
        'message' => 'Required fields: mobile, game_id, game_name & bids (array).'
    ]));
}

// 3) clean & sum bids
$minBid = 10;
$total  = 0.0;
$clean  = [];

foreach ($bidsArr as $b) {
    if (!is_array($b)) {
        continue;
    }
    $open  = trim($b['open_pick']  ?? '');
    $close = trim($b['close_pick'] ?? '');
    $amt   = floatval($b['amount'] ?? 0);

    // require both open & close
    if ($open === '' || $close === '') {
        continue;
    }
    // enforce minimum amount
    if ($amt < $minBid) {
        continue;
    }

    $clean[] = [
        'open'   => $open,
        'close'  => $close,
        'amount' => $amt
    ];
    $total += $amt;
}

if (empty($clean)) {
    http_response_code(400);
    exit(json_encode([
        'status'  => false,
        'message' => "No valid bids. Minimum bid is ₹{$minBid}."
    ]));
}

// 4) bootstrap database
require __DIR__ . '/db.php';
$conn->set_charset('utf8mb4');

// 5) lookup user_id
$stmt = $conn->prepare("SELECT user_id FROM users_data WHERE mobile = ? LIMIT 1");
$stmt->bind_param('s', $mobile);
$stmt->execute();
$stmt->bind_result($uid);
if (!$stmt->fetch()) {
    http_response_code(404);
    exit(json_encode([
        'status'  => false,
        'message' => 'User not found.'
    ]));
}
$stmt->close();

// 6) verify game exists
$stmt = $conn->prepare("
    SELECT id, open_time, close_time
      FROM game56
     WHERE id  = ?
       AND LOWER(game_name) = LOWER(?)
    LIMIT 1
");
$stmt->bind_param('is', $game_id, $game_name);
$stmt->execute();
$stmt->bind_result($game_id, $open_time, $close_time);
if (!$stmt->fetch()) {
    http_response_code(404);
    exit(json_encode([
        'status'  => false,
        'message' => 'Game not found.'
    ]));
}
$stmt->close();

// 7) fetch wallet balance
$stmt = $conn->prepare("SELECT balance FROM wallet WHERE user_id = ? LIMIT 1");
$stmt->bind_param('i', $uid);
$stmt->execute();
$stmt->bind_result($wallet_balance);
$stmt->fetch();
$stmt->close();

if ($total > $wallet_balance) {
    http_response_code(400);
    exit(json_encode([
        'status'  => false,
        'message' => 'Insufficient balance.',
        'data'    => ['wallet_balance' => $wallet_balance]
    ]));
}

// 8) determine session (open/close) based on server time
date_default_timezone_set('Asia/Kolkata');
$now = date('H:i:s');
if ($open_time < $close_time) {
    $session = ($now >= $open_time && $now <= $close_time) ? 'close' : 'open';
} else {
    $session = ($now >= $open_time || $now <= $close_time) ? 'close' : 'open';
}

// 9) insert bids + deduct wallet in one transaction
$conn->begin_transaction();

try {
    $ins = $conn->prepare("
        INSERT INTO user_bids_hs
          (user_id, game_id, bid_open_hs, bid_close_hs,
           total_bid_amount, bid_session, game_name)
        VALUES (?, ?, ?, ?, ?, ?, ?)
    ");

    foreach ($clean as $b) {
        $ins->bind_param(
            'iissdss',
            $uid,
            $game_id,
            $b['open'],
            $b['close'],
            $b['amount'],
            $session,
            $game_name
        );
        $ins->execute();
    }
    $ins->close();

    $upd = $conn->prepare("
        UPDATE wallet
           SET balance = balance - ?
         WHERE user_id = ?
    ");
    $upd->bind_param('di', $total, $uid);
    $upd->execute();
    $upd->close();

    $newBal = $wallet_balance - $total;
    $conn->commit();

    echo json_encode([
        'status'  => true,
        'message' => "Placed ₹{$total} in {$session} session.",
        'data'    => [
            
            'wallet_balance' => $newBal,
    'session'        => $session,
   'open_time'      => $open_time,
   'close_time'     => $close_time,
            ]
    ]);
    exit;

} catch (Exception $e) {
    $conn->rollback();
    http_response_code(500);
    echo json_encode([
        'status'  => false,
        'message' => 'Server error: ' . $e->getMessage()
    ]);
    exit;
}

$conn->close();
