<?php
// jd_api.php — accept multiple Jodi Digit bids in one call

// 1) Disable HTML/PHP error output for safety
ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(0);

// 2) JSON response header
header('Content-Type: application/json; charset=utf-8');

// 3) Include your DB connection
require_once __DIR__ . '/db.php';
$conn->set_charset('utf8mb4');

// 4) Parse incoming JSON
$raw  = file_get_contents('php://input');
$in   = json_decode($raw, true);
if (!$in || json_last_error() !== JSON_ERROR_NONE) {
    http_response_code(400);
    exit(json_encode([
        'status'  => false,
        'message' => 'Invalid JSON payload.'
    ]));
}

// 5) Validate required fields
$mobile      = trim($in['mobile']       ?? '');
$game_id = intval($in['game_id']?? 0);
$game_name   = trim($in['game_name']    ?? '');
$bids        = $in['bids']              ?? [];

if ($mobile === '' || $game_id <= 0    || $game_name === '' || !is_array($bids)) {
    http_response_code(400);
    exit(json_encode([
        'status'  => false,
        'message' => 'Required: mobile, game_id, game_name & bids(array).'
    ]));
}

// 6) Clean & sum bids (only 2-digit keys, ≥ ₹10 each)
$minBid    = 10;
$total     = 0.0;
$cleanBids = [];
foreach ($bids as $item) {
    if (!is_array($item) || !isset($item['number'], $item['amount'])) {
        continue;
    }
    $num = trim($item['number']);
    $amt = floatval($item['amount']);
    if (preg_match('/^[0-9]{2}$/', $num) && $amt >= $minBid) {
        // overwrite same number if repeated
        $cleanBids[$num] = $amt;
        $total += $amt;
    }
}

if (empty($cleanBids)) {
    http_response_code(400);
    exit(json_encode([
        'status'  => false,
        'message' => "Minimum bid is ₹{$minBid} on valid two-digit numbers."
    ]));
}

// 7) Lookup user_id by mobile
$stmt = $conn->prepare("SELECT user_id FROM users_data WHERE mobile = ? LIMIT 1");
$stmt->bind_param('s', $mobile);
$stmt->execute();
$stmt->bind_result($user_id);
if (!$stmt->fetch()) {
    http_response_code(404);
    exit(json_encode([
        'status'  => false,
        'message' => 'User not found.'
    ]));
}
$stmt->close();

// 8) Lookup game by game_number & name, fetch all four timestamps
$stmt = $conn->prepare("
    SELECT 
      id,
      open_time,
      close_time,
      open_bid_last_time,
      close_bid_last_time
    FROM game56
    WHERE id = ?
      AND LOWER(game_name) = LOWER(?)
    LIMIT 1
");
$stmt->bind_param('is', $game_id, $game_name);
$stmt->execute();
$stmt->store_result();
if ($stmt->num_rows !== 1) {
    http_response_code(404);
    exit(json_encode([
        'status'  => false,
        'message' => 'Game not found or name mismatch.'
    ]));
}
$stmt->bind_result(
    $game_id,
    $open_time,
    $close_time,
    $open_bid_last,
    $close_bid_last
);
$stmt->fetch();
$stmt->close();

// 9) Compute the five-state status_text
date_default_timezone_set('Asia/Kolkata');
$nowTs      = time();
$tsOpenBid  = strtotime($open_bid_last);
$tsOpenRes  = strtotime($open_time);
$tsCloseBid = strtotime($close_bid_last);
$tsCloseRes = strtotime($close_time);

if ($nowTs < $tsOpenBid) {
    $status_text = 'Game is open for bid';
}
elseif ($nowTs < $tsOpenRes) {
    $status_text = 'Open bidding is closed';
}
elseif ($nowTs < $tsCloseBid) {
    $status_text = 'Game is Running';
}
elseif ($nowTs < $tsCloseRes) {
    $status_text = 'Close bidding is closed';
}
else {
    $status_text = 'Game has Closed';
}

// 10) Reject if not in the “open for bid” phase


// 11) Check wallet balance
$stmt = $conn->prepare("SELECT balance FROM wallet WHERE user_id = ?");
$stmt->bind_param('i', $user_id);
$stmt->execute();
$stmt->bind_result($wallet_balance);
$stmt->fetch();
$stmt->close();

if ($total > $wallet_balance) {
    http_response_code(400);
    exit(json_encode([
        'status'  => false,
        'message' => 'Insufficient wallet balance.',
        'data'    => ['wallet_balance' => $wallet_balance]
    ]));
}

// 12) Insert bids & deduct balance in a transaction
$conn->begin_transaction();
try {
    // store the bids as a JSON object: { "12":100, "45":50, ... }
    $detailsJson = json_encode($cleanBids, JSON_NUMERIC_CHECK);

    $ins = $conn->prepare("
        INSERT INTO user_bids_jd
          (user_id, game_id, bid_details, total_bid_amount, bid_session, game_name)
        VALUES (?,?,?,?,?,?)
    ");
    $ins->bind_param(
        'iissss',
        $user_id,
        $game_id,
        $detailsJson,
        $total,
        $status_text,
        $game_name
    );
    $ins->execute();
    $ins->close();

    // deduct from wallet
    $upd = $conn->prepare("UPDATE wallet SET balance = balance - ? WHERE user_id = ?");
    $upd->bind_param('di', $total, $user_id);
    $upd->execute();
    $upd->close();

    $newBalance = $wallet_balance - $total;
    $conn->commit();

    echo json_encode([
        'status'  => true,
        'message' => "Placed total ₹{$total}.",
        'data'    => [
            'wallet_balance'      => $newBalance,
            'status_text'         => $status_text,
            'open_time'           => $open_time,
            'close_time'          => $close_time,
            'open_bid_last_time'  => $open_bid_last,
            'close_bid_last_time' => $close_bid_last,
        ]
    ]);
} catch (Exception $e) {
    $conn->rollback();
    http_response_code(500);
    echo json_encode([
        'status'  => false,
        'message' => 'Server error: '.$e->getMessage()
    ]);
}

$conn->close();
