<?php
// jodi_digit.php

// 1) DEV: show errors (remove in prod)
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

session_start();
include 'db.php'; // $conn

// 2) Auth
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}
$user_id = (int)$_SESSION['user_id'];

// 3) game_id
if (empty($_GET['game_id'])) {
    die("No game specified.");
}
$game_id = (int)$_GET['game_id'];

// 4) mark sub_game
$stmt = $conn->prepare("UPDATE game56 SET sub_games = 'jodi_digit' WHERE id = ?");
$stmt->bind_param("i",$game_id);
$stmt->execute();
$stmt->close();

// 5) fetch username
$stmt = $conn->prepare("SELECT username FROM users_data WHERE user_id = ?");
$stmt->bind_param("i",$user_id);
$stmt->execute();
$stmt->bind_result($username);
$stmt->fetch();
$stmt->close();

// 6) fetch game_name, open_time, close_time
$stmt = $conn->prepare("
  SELECT game_name, open_time, close_time
    FROM game56
   WHERE id = ?
");
$stmt->bind_param("i",$game_id);
$stmt->execute();
$stmt->bind_result($game_name,$open_time,$close_time);
if (!$stmt->fetch()) {
    die("Game not found.");
}
$stmt->close();

// 7) times & status logic
date_default_timezone_set('Asia/Kolkata');
$now = new DateTime();

// build today's open/close DateTime
$openDT  = DateTime::createFromFormat('H:i:s',$open_time);
$closeDT = DateTime::createFromFormat('H:i:s',$close_time);
if ($closeDT <= $openDT) {
    // overnight close
    $closeDT->modify('+1 day');
}

// thresholds
$runStart = (clone $openDT)->modify('-10 minutes');
$runEnd   = (clone $closeDT)->modify('-10 minutes');

// 7a) Game status
if ($now < $runStart) {
    $gameStatus = 'Open';
} elseif ($now >= $runStart && $now <= $runEnd) {
    $gameStatus = 'Running';
} else {
    $gameStatus = 'Closed';
}

if ($gameStatus === 'Closed') {
    echo "
      <script>
        alert('Bid time is closed for the day.');
        // Redirect back to the “Play Game” menu (adjust URL if needed)
        window.location = 'play_game.php?game_id={$game_id}';
      </script>
    ";
    exit;
}

// 7b) toggle classes (raw open/close)
$openBtnClass  = ($now < $openDT)                     ? 'btn-blue':'btn-default';
$closeBtnClass = ($now >= $openDT && $now <= $closeDT)? 'btn-blue':'btn-default';

// 7c) bid_session (raw)
$bid_session = ($now >= $openDT && $now <= $closeDT) ? 'open' : 'close';

// 8) fetch wallet
$stmt = $conn->prepare("SELECT balance FROM wallet WHERE user_id = ?");
$stmt->bind_param("i",$user_id);
$stmt->execute();
$stmt->bind_result($wallet_balance);
$stmt->fetch();
$stmt->close();

// 9) build 00–99 list
$jodiDigits = [];
for ($i=0;$i<100;$i++) {
    $jodiDigits[] = str_pad($i,2,'0',STR_PAD_LEFT);
}

// 10) handle POST
// 10) handle POST
$viewLink = '';
if ($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['place_bid'])) {
    // First, do not allow bids during "close" session:
    if ($bid_session === 'close') {
        $error = "Bidding is closed for this session.";
    } else {
        $selected = $_POST['selected_number'] ?? '';
        $bid      = floatval($_POST['bid_amount'] ?? 0);

        if ($selected === '') {
            $error = "Please select a number.";
        } elseif ($bid < 10) {
            $error = "Minimum bid is ₹10.";
        } elseif ($bid > $wallet_balance) {
            $error = "Insufficient wallet balance.";
        } else {
            $conn->begin_transaction();
            try {
                // insert
                $ins = $conn->prepare("
                  INSERT INTO user_bids_jd
                    (user_id, game_id, bid_details, total_bid_amount, bid_session, game_name)
                  VALUES (?,?,?,?,?,?)
                ");
                $ins->bind_param(
                  "isdsss",
                  $user_id,
                  $game_id,
                  $selected,
                  $bid,
                  $bid_session,
                  $game_name
                );
                $ins->execute();
                $ins->close();

                // deduct wallet
                $upd = $conn->prepare("
                  UPDATE wallet
                     SET balance = balance - ?
                   WHERE user_id = ?
                ");
                $upd->bind_param("di",$bid,$user_id);
                $upd->execute();
                $upd->close();

                $conn->commit();
                $wallet_balance -= $bid;
                // only show View Results link
                $viewLink = "<p class='view'><a href='random.php?game_id={$game_id}'>View Results</a></p>";
            } catch (Exception $e) {
                $conn->rollback();
                $error = "Error placing bid: ".$e->getMessage();
            }
        }
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Jodi Digit Bidding</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <style>
    body { background:#f8f9fa; margin:0; font-family:Arial,sans-serif; color:#333 }
    .back { display:inline-block; margin:20px; color:#1E90FF; text-decoration:none }
    .back:hover { text-decoration:underline }
    .container {
      max-width:600px; margin:0 auto; background:#fff; padding:20px;
      border-radius:8px; box-shadow:0 2px 5px rgba(0,0,0,0.1);
    }
    h1 { text-align:center; color:#007BFF }
    .wallet { text-align:center; margin-bottom:15px; font-size:1.1rem; }
    .status {
      text-align:center; font-weight:bold; margin:10px 0;
      color:<?= $gameStatus==='Running'?'#28a745':($gameStatus==='Closed'?'#c00':'#555') ?>;
    }
    .toggle {
      text-align:center; margin-bottom:20px;
    }
    .btn {
      padding:8px 16px; margin:0 5px; border:none; border-radius:4px;
      cursor:pointer; background:#ccc; color:#333;
    }
    .btn-blue { background:#007BFF; color:#fff; }
    .btn-default { background:#ccc; }
    .grid {
      display:grid; grid-template-columns:repeat(10,1fr);
      gap:6px; margin-bottom:20px;
    }
    .grid div {
      background:#fff; border:1px solid #ccc; border-radius:4px;
      padding:8px 0; text-align:center; font-weight:bold;
    }
    .form-container {
      background:#fff; padding:20px; border-radius:8px;
      box-shadow:0 2px 5px rgba(0,0,0,0.1);
    }
    .form-container h3 { margin-top:0; color:#007BFF; text-align:center }
    .alert { padding:0.75rem; margin-bottom:1rem; border-radius:4px; }
    .alert-danger { background:#f8d7da; color:#721c24; }
    .view { text-align:center; margin:15px 0; }
    .view a { color:#007BFF; text-decoration:underline; }
    label { display:block; margin:0.5rem 0; font-weight:bold; }
    select,input[type=number] {
      width:100%; padding:0.5rem; font-size:1rem;
      border:1px solid #ccc; border-radius:4px; margin-bottom:1rem;
    }
    button[type=submit] {
      width:100%; padding:0.75rem; font-size:1rem;
      background:#007BFF; color:#fff; border:none; border-radius:4px;
      cursor:pointer;
    }
    button[type=submit]:hover { background:#0056b3 }
  </style>
</head>
<body>

  <a href="dashboard.php" class="back">← Dashboard</a>
  <div class="container">
    <h1>Jodi Digit Bidding</h1>
    <div class="wallet">Wallet Balance: ₹<?= number_format($wallet_balance,2) ?></div>

    <div class="status">Game is <?= $gameStatus ?></div>

    <div class="toggle">
      <button type="button" class="btn <?= $openBtnClass ?>">Open</button>
      <button type="button" class="btn <?= $closeBtnClass ?>">Close</button>
    </div>

    <!-- Reference grid -->
    <div class="grid">
      <?php foreach($jodiDigits as $d): ?>
        <div><?= $d ?></div>
      <?php endforeach; ?>
    </div>

    <div class="form-container">
  <h3>Place Your Bid</h3>
  <?php if (!empty($error)): ?>
    <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
  <?php endif; ?>

  <form method="POST">
    <label for="selected_number">Select Number:</label>
    <select name="selected_number" id="selected_number" required>
      <option value="">-- Choose Jodi --</option>
      <?php foreach($jodiDigits as $d): ?>
        <option value="<?= $d ?>"><?= $d ?></option>
      <?php endforeach; ?>
    </select>

    <label for="bid_amount">Bid Amount (₹):</label>
    <input type="number" name="bid_amount" id="bid_amount"
           step="0.01" min="10" required>

    <button type="submit" name="place_bid">Place Bid</button>
  </form>

  <?= $viewLink ?>
</div>

  </div>

</body>
</html>
