<?php
// /Sai/lib/wallet_helper.php
require_once __DIR__ . '/../db.php';

function wallet__ensure_exists(int $user_id) {
  global $conn;
  $stmt = $conn->prepare("INSERT INTO wallet (user_id, balance)
                          VALUES (?, 0.00)
                          ON DUPLICATE KEY UPDATE user_id = user_id");
  $stmt->bind_param('i', $user_id);
  $stmt->execute();
  $stmt->close();
}

function wallet_credit(int $user_id, float $amount, string $reason,
                       ?string $game = null, ?string $ref_id = null): bool {
  global $conn;
  if ($amount <= 0) return false;

  $conn->begin_transaction();
  try {
    wallet__ensure_exists($user_id);

    // 1) add ledger row once
    $q = "INSERT INTO wallet_ledger (user_id, direction, amount, reason, game, ref_id)
      VALUES (?,?,?,?,?,?)
      ON DUPLICATE KEY UPDATE id = id";
$st = $conn->prepare($q);
$dir = 'credit'; // or 'debit'
$st->bind_param('isdsss', $user_id, $dir, $amount, $reason, $game, $ref_id);
    $st->execute();
    $inserted = $st->affected_rows > 0;  // false means duplicate -> already credited
    $st->close();

    if ($inserted) {
      // 2) update wallet balance
      $st2 = $conn->prepare("UPDATE wallet SET balance = balance + ? WHERE user_id = ?");
      $st2->bind_param('di', $amount, $user_id);
      $st2->execute();
      $st2->close();
    }

    $conn->commit();
    return true;
  } catch (Throwable $e) {
    $conn->rollback();
    return false;
  }
}

function wallet_debit(int $user_id, float $amount, string $reason,
                      ?string $game = null, ?string $ref_id = null): bool {
  global $conn;
  if ($amount <= 0) return false;

  $conn->begin_transaction();
  try {
    wallet__ensure_exists($user_id);

    $q = "INSERT INTO wallet_ledger (user_id, direction, amount, reason, game, ref_id)
          VALUES (?,?,?,?,?,?)
          ON DUPLICATE KEY UPDATE id = id";
    $st = $conn->prepare($q);
    $dir = 'debit';
    $st->bind_param('isdsss', $user_id, $dir, $amount, $reason, $game, $ref_id);
    $st->execute();
    $inserted = $st->affected_rows > 0;
    $st->close();

    if ($inserted) {
      $st2 = $conn->prepare("UPDATE wallet SET balance = balance - ? WHERE user_id = ?");
      $st2->bind_param('di', $amount, $user_id);
      $st2->execute();
      $st2->close();
    }

    $conn->commit();
    return true;
  } catch (Throwable $e) {
    $conn->rollback();
    return false;
  }
}


function wallet_win_credit(int $user_id, float $amount, string $type, int $bid_id, ?string $subkey, string $session): bool {
  // WIN-SD-12345-0-OPEN  |  WIN-JD-9876-12-CLOSE  |  WIN-HS-321--OPEN
  $ref = sprintf('WIN-%s-%d-%s-%s', strtoupper($type), $bid_id, ($subkey ?? ''), strtoupper($session));
  return wallet_credit($user_id, $amount, 'win-'.$type, $type, $ref);
}
