<?php
// 1) DEV: show all errors (remove in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// 2) JSON response header
header("Content-Type: application/json; charset=utf-8");

// 3) Database connection
include __DIR__ . '/db.php';

// 4) Merge raw JSON body into $_REQUEST so both JSON & form-data work
$raw = file_get_contents('php://input');
if ($json = @json_decode($raw, true)) {
    $_REQUEST = array_merge($_REQUEST, $json);
}

// 5) Read & validate inputs
$mobile  = isset($_REQUEST['mobile'])  ? trim($_REQUEST['mobile'])  : '';
$game_id = isset($_REQUEST['game_id']) ? (int)$_REQUEST['game_id'] : 0;

if ($mobile === '' || $game_id <= 0) {
    http_response_code(400);
    exit(json_encode([
        'status'  => false,
        'message' => 'Both mobile number and game_id are required.',
        'data'    => null
    ]));
}

try {
    // 6) Lookup user by mobile (selecting username, not `name`)
    $stmt = $conn->prepare("SELECT id, username AS name FROM users WHERE mobile = ?");
    if (!$stmt) {
        throw new Exception($conn->error);
    }
    $stmt->bind_param('s', $mobile);
    $stmt->execute();
    $userRes = $stmt->get_result();
    $stmt->close();

    if ($userRes->num_rows === 0) {
        http_response_code(404);
        exit(json_encode([
            'status'  => false,
            'message' => 'User not found.',
            'data'    => null
        ]));
    }
    $user = $userRes->fetch_assoc();

    // 7) Lookup game
    $stmt = $conn->prepare("SELECT game_name, open_time, close_time FROM game56 WHERE id = ?");
    if (!$stmt) {
        throw new Exception($conn->error);
    }
    $stmt->bind_param('i', $game_id);
    $stmt->execute();
    $gameRes = $stmt->get_result();
    $stmt->close();

    if ($gameRes->num_rows === 0) {
        http_response_code(404);
        exit(json_encode([
            'status'  => false,
            'message' => 'Game not found.',
            'data'    => null
        ]));
    }
    $game = $gameRes->fetch_assoc();

    // 8) Format times
    $game['formatted_open_time']  = date("g:i A", strtotime($game['open_time']))  . " IST";
    $game['formatted_close_time'] = date("g:i A", strtotime($game['close_time'])) . " IST";

    // 9) Build options list based on game_name
    $optionsMap = [
        'Sai night'    => ['single_digit.php','jodi_digit.php','singlepanna.php','triplepanna.php'],
        'Sai midnight' => ['single_digit.php','double_panna.php','jodi_digit.php','singlepanna.php','triplepanna.php'],
    ];
    $paths = $optionsMap[$game['game_name']] 
           ?? ['single_digit.php','double_panna.php','jodi_digit.php','singlepanna.php','triplepanna.php'];

    $baseUrl = "https://smartdigisolution.com/Sai/";
    $options = array_map(function($p) use ($game_id, $baseUrl) {
        $url = preg_match('#^https?://#i', $p)
             ? $p
             : $baseUrl . $p;
        $sep = (strpos($url, '?') === false ? '?' : '&');
        return "{$url}{$sep}game_id={$game_id}";
    }, $paths);

    // 10) Send JSON response
    echo json_encode([
        'status'  => true,
        'message' => 'Game data fetched successfully.',
        'data'    => [
            'user'    => [
                'id'     => $user['id'],
                'name'   => $user['name'],
                'mobile' => $mobile
            ],
            'game'    => $game,
            'options' => $options
        ]
    ]);

} catch (Exception $e) {
    // 11) Unexpected error
    http_response_code(500);
    echo json_encode([
        'status'  => false,
        'message' => 'Server error: ' . $e->getMessage(),
        'data'    => null
    ]);
}

// 12) Close connection
$conn->close();
