<?php
// play_games_api.php

// 1) Error reporting (disable in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// 2) CORS & JSON response headers
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");

// 3) Include DB connection
require_once __DIR__ . '/db.php';

// 4) Read `game_number` from GET or JSON body
$rawInput     = file_get_contents('php://input');
$jsonPayload  = json_decode($rawInput, true);

$game_id_in   = intval($jsonPayload['game_id'] ?? 0);
$game_number  = $jsonPayload['game_number'] ?? '';


if (!is_string($game_number) || $game_number === '') {
    http_response_code(400);
    exit(json_encode([
        'success' => false,
        'message' => 'Please provide a valid game_number.'
    ]));
}

// 5) Lookup the game row (including all times)
if ($game_id_in > 0) {
    // 5a) lookup by primary key id
    $stmt = $conn->prepare(
      "SELECT 
         id,
         game_name,
         open_time,
         close_time,
         open_bid_last_time,
         close_bid_last_time,
         status
       FROM game56
      WHERE id = ?
      LIMIT 1"
    );
    $stmt->bind_param("i", $game_id_in);
} else {
    // 5b) fallback to lookup by game_number
    $stmt = $conn->prepare(
      "SELECT 
         id,
         game_name,
         open_time,
         close_time,
         open_bid_last_time,
         close_bid_last_time,
         status
       FROM game56
      WHERE game_number = ?
      LIMIT 1"
    );
    $stmt->bind_param("s", $game_number);
}
$stmt->execute();
$stmt->store_result();

if ($stmt->num_rows !== 1) {
    http_response_code(404);
    exit(json_encode([
        'success' => false,
        'message' => 'No game found with that game_number.'
    ]));
}

$stmt->bind_result(
    $game_id,
    $game_name,
    $open_time,
    $close_time,
    $open_bid_last,
    $close_bid_last,
    $dbStatus
);
$stmt->fetch();
$stmt->close();
// sanitize the status column
$dbStatus     = trim($dbStatus);
// flag whether we’re in the “open for bid” phase
$isOpenForBid = (strcasecmp($dbStatus, 'Game is open for bid') === 0);


// 6) Compute the five-state status_text (for DB sync)
date_default_timezone_set('Asia/Kolkata');
$nowTs         = time();
$tsOpenBid     = strtotime($open_bid_last);
$tsOpenResult  = strtotime($open_time);
$tsCloseBid    = strtotime($close_bid_last);
$tsCloseResult = strtotime($close_time);

if ($nowTs < $tsOpenBid) {
    $status_text = 'Game is open for bid';
}
elseif ($nowTs < $tsOpenResult) {
    $status_text = 'Open bidding is closed';
}
elseif ($nowTs < $tsCloseBid) {
    $status_text = 'Game is Running';
}
elseif ($nowTs < $tsCloseResult) {
    $status_text = 'Close bidding is closed';
}
else {
    $status_text = 'Game has Closed';
}

// 7) Persist status back into the table
$upd = $conn->prepare("UPDATE game56 SET status = ? WHERE id = ?");
$upd->bind_param('si', $status_text, $game_id);
$upd->execute();
$upd->close();

// 8) Compute the three-state gameSession exactly as in play_game.php
//    — build DateTime objects for today’s open/close
// $openDT  = DateTime::createFromFormat('H:i:s', $open_time);
// $closeDT = DateTime::createFromFormat('H:i:s', $close_time);

// // if close is earlier than or equal to open, it wraps past midnight
// if ($closeDT <= $openDT) {
//     $closeDT->modify('+1 day');
// }

// // bidding must finish 10 minutes before the results open
// $runStart = (clone $openDT)->modify('-10 minutes');
// $runEnd   = (clone $closeDT)->modify('-10 minutes');

// $nowDT = new DateTime('now', new DateTimeZone('Asia/Kolkata'));

// if ($nowDT < $runStart) {
//     $gameSession = 'Open';
// }
// elseif ($nowDT >= $runStart && $nowDT <= $runEnd) {
//     $gameSession = 'Running';
// }
// else {
//     $gameSession = 'Closed';
// }

// 9) Define sub-game options
$allOptions = [
    "Single Digit"  => "assets/dice.png",
    "Jodi Digit"    => "assets/dice (1).png",
    "Single Panna"  => "assets/diamond.png",
    "Double Panna"  => "assets/poker-cards.png",
    "Triple Panna"  => "assets/cards.png",
    "Half Sangam"   => "assets/cards (1).png",
    "Full Sangam"   => "assets/cards (1).png",
];

// 10) Build options array, disabling only Jodi & Full Sangam when session ≠ “Open”
$options = [];
foreach ($allOptions as $label => $img) {
    $enabled = true;
    if (
        ($label === 'Jodi Digit' || $label === 'Full Sangam')
        && ! $isOpenForBid
    ) {
        $enabled = false;
    }
    $options[] = [
        'label'     => $label,
        'imagePath' => $img,
        'enabled'   => $enabled,
    ];
}

// 11) Return full JSON payload
echo json_encode([
    'success'          => true,
    'game_number'      => $game_number,
    'game_name'        => $game_name,
    'status_text'      => $status_text,
    // 'game_session'     => $gameSession,
    'open'             => date("g:i A", $tsOpenResult) . " IST",
    'close'            => date("g:i A", $tsCloseResult) . " IST",
    'open_bid_last'    => date("g:i A", $tsOpenBid)     . " IST",
    'close_bid_last'   => date("g:i A", $tsCloseBid)    . " IST",
    'options'          => $options,
]);

$conn->close();
