<?php
session_start();

// Set timezone to India time (Asia/Kolkata)
date_default_timezone_set('Asia/Kolkata');

// Include your database connection file
include('db.php');

// Check if the user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

// --------------------------------------------------
// 1. Prepare Filter for the Last 5 Days
// --------------------------------------------------
$last5days = [];
for ($i = 0; $i < 5; $i++) {
    $dateValue = date('Y-m-d', strtotime("-$i days"));
    $dateLabel = date('d/m/Y', strtotime("-$i days"));
    $last5days[$dateValue] = $dateLabel;
}

// Get the selected filter date from GET parameter, default to today if not set
$filter_date = isset($_GET['filter_date']) ? $_GET['filter_date'] : date('Y-m-d');

// --------------------------------------------------
// 2. Fetch Game Details for the Filter Date
// --------------------------------------------------
$sql_game_data = "
    SELECT id, game_name, open_time, close_time, status, winning_number, winning_time
    FROM game56
    WHERE DATE(open_time) = ?
";
$stmt = $conn->prepare($sql_game_data);
$stmt->bind_param("s", $filter_date);
$stmt->execute();
$stmt->bind_result($game_id, $game_name, $open_time, $close_time, $status, $winning_number, $winning_time);
$games = [];
while ($stmt->fetch()) {
    $games[] = [
        'game_id'        => $game_id,
        'game_name'      => $game_name,
        'open_time'      => $open_time,
        'close_time'     => $close_time,
        'status'         => $status,
        'winning_number' => $winning_number,
        'winning_time'   => $winning_time
    ];
}
$stmt->close();

// --------------------------------------------------
// 3. (Optional) Reset Winning Data for Past Dates
// --------------------------------------------------
$todayDate = date('Y-m-d');
if ($filter_date !== $todayDate) {
    foreach ($games as &$game) {
        $game_id = $game['game_id'];
        $reset_sql = "UPDATE game56 SET winning_number = NULL, winning_time = NULL WHERE id = ?";
        $stmt_reset = $conn->prepare($reset_sql);
        $stmt_reset->bind_param("i", $game_id);
        $stmt_reset->execute();
        $stmt_reset->close();

        $game['winning_number'] = null;
        $game['winning_time']   = null;
    }
}

// Current date/time for display in 12-hour format
$currentDateTime = date("d/m/Y g:i A");
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width,initial-scale=1.0">
  <title>Game Winning Status</title>
  <style>
      body {
          font-family: Arial, sans-serif;
          background-color: #f4f4f9;
          margin: 0;
          padding: 0;
          color: #333;
      }
      .container {
          max-width: 800px;
          margin: 50px auto;
          background-color: #fff;
          padding: 20px;
          border-radius: 8px;
          box-shadow: 0 0 10px rgba(0,0,0,0.1);
      }
      .filter-form {
          text-align: center;
          margin-bottom: 20px;
      }
      .game-box {
          padding: 15px;
          margin-top: 20px;
          text-align: center;
          border: 1px solid #ddd;
          border-radius: 5px;
      }
      .game-box h3 {
          margin: 10px 0;
          color: #333;
      }
      .game-box p {
          color: #555;
      }
      .message {
          padding: 15px;
          margin-top: 20px;
          text-align: center;
          border-radius: 5px;
      }
      .message.success {
          background-color: #d4edda;
          color: #155724;
      }
      .message.error {
          background-color: #f8d7da;
          color: #721c24;
      }
      .current-datetime {
          text-align: center;
          margin-bottom: 20px;
          font-size: 1.1em;
          color: #007BFF;
      }
  </style>
</head>
<body>
  <div class="container">
      <h1>Game Winning Status</h1>
      
      <!-- Current Date & Time -->
      <div class="current-datetime">
          Current Date & Time (IST): <?php echo $currentDateTime; ?>
      </div>
      
      <!-- Filter Form for Last 5 Days -->
      <div class="filter-form">
          <form method="GET" action="">
              <label for="filter_date">Select Date: </label>
              <select name="filter_date" id="filter_date">
                  <?php foreach ($last5days as $dateValue => $dateLabel): ?>
                      <option value="<?php echo $dateValue; ?>" <?php echo ($filter_date == $dateValue) ? "selected" : ""; ?>>
                          <?php echo $dateLabel; ?>
                      </option>
                  <?php endforeach; ?>
              </select>
              <button type="submit">Filter</button>
          </form>
      </div>
      
      <?php
      // --------------------------------------------------
      // 4. Loop Through Each Game and Display Details
      // --------------------------------------------------
      $current_time = time(); // current Unix timestamp

      foreach ($games as $game) {
          $game_id        = $game['game_id'];
          $game_name      = $game['game_name'];
          $open_time      = $game['open_time'];   // e.g., "2025-04-05 04:20:00"
          $close_time     = $game['close_time'];  // e.g., "2025-04-05 06:00:00"
          $status         = $game['status'];      // this field can be used if needed
          $winning_number = $game['winning_number'];
          $winning_time   = $game['winning_time'];

          // Convert open and close times to timestamps
          $open_ts  = strtotime($open_time);
          $close_ts = strtotime($close_time);

          // Handle next-day close: if close time appears earlier than open, add 24 hours.
          if ($close_ts < $open_ts) {
              $close_ts += 86400;
          }

          echo "<div class='game-box'>";
          echo "<h3>Game Name: {$game_name}</h3>";
          echo "<p>Open Time: " . date("d/m/Y g:i A", $open_ts) . " IST</p>";
          echo "<p>Close Time: " . date("d/m/Y g:i A", $close_ts) . " IST</p>";

          // --------------------------------------------------
          // 5. Determine Game Status Based on Current Time
          // --------------------------------------------------
          if ($current_time < $open_ts) {
              // Before open time: Yet to Open
              echo "<div class='message'>Status: Yet to Open</div>";
          }
          else if ($current_time >= $open_ts && $current_time < $close_ts) {
              // Between open and close: Running (show countdown)
              $remaining = $close_ts - $current_time;
              $hours   = floor($remaining / 3600);
              $minutes = floor(($remaining % 3600) / 60);
              $seconds = $remaining % 60;
              echo "<div class='message'>
                      Status: Running<br>
                      Time left until game closes: {$hours} hours, {$minutes} minutes, {$seconds} seconds.
                    </div>";
              echo "<script>setTimeout(function() { location.reload(); }, 30000);</script>";
          }
          else {
              // After close time: Closed.
              // Use the lowest bid logic to determine winning number and time if not already set.
              if (!$winning_number && $filter_date === $todayDate) {
                  // Query the bids table for the lowest bid for this game.
                  $bid_sql = "SELECT bid_amount, bid_time FROM bids WHERE game_id = ? ORDER BY bid_amount ASC LIMIT 1";
                  $stmt_bid = $conn->prepare($bid_sql);
                  $stmt_bid->bind_param("i", $game_id);
                  $stmt_bid->execute();
                  $stmt_bid->bind_result($lowest_bid, $bid_time);
                  $bid_found = $stmt_bid->fetch();
                  $stmt_bid->close();

                  if ($bid_found) {
                      // Use the lowest bid as the winning number.
                      $winning_number = $lowest_bid;
                      // You can choose to use bid_time or the current time as winning_time.
                      // Here, we use bid_time if available.
                      $winning_time = $bid_time;
                      
                      // Update the game record with the winning number and winning time.
                      $update_sql = "UPDATE game56 SET winning_number = ?, winning_time = ? WHERE id = ?";
                      $stmt_update = $conn->prepare($update_sql);
                      $stmt_update->bind_param("isi", $winning_number, $winning_time, $game_id);
                      $stmt_update->execute();
                      if ($stmt_update->error) {
                          echo "<div class='message error'>
                                  Error updating game: " . htmlspecialchars($stmt_update->error) . "
                                </div>";
                      } else {
                          $displayDateTime = date("d/m/Y g:i A", strtotime($winning_time));
                          echo "<div class='message success'>
                                  Status: Closed<br>
                                  Winning Number (Lowest Bid): {$winning_number}<br>
                                  Winning Date/Time: {$displayDateTime} (IST)
                                </div>";
                      }
                      $stmt_update->close();
                  } else {
                      // If no bids are found, show a message.
                      echo "<div class='message'>
                              Status: Closed<br>
                              No bids found for this game.
                            </div>";
                  }
              } else {
                  // If winning number already exists, display it.
                  $formattedTime = $winning_time ? date("d/m/Y g:i A", strtotime($winning_time)) : "N/A";
                  echo "<div class='message success'>
                          Status: Closed<br>
                          Winning Number (Lowest Bid): {$winning_number}<br>
                          Winning Date/Time: {$formattedTime} (IST)
                        </div>";
              }
          }

          echo "</div>"; // End of game-box
      }
      ?>
  </div>
</body>
</html>

<?php
$conn->close();
?>
