<?php
// register_api.php

declare(strict_types=1);

// — Show errors during development (disable in production) —
ini_set('display_errors', '1');
ini_set('display_startup_errors', '1');
error_reporting(E_ALL);

// — CORS & JSON headers —
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Content-Type: application/json; charset=UTF-8');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

// — Autoload & Firebase setup —
require __DIR__ . '/vendor/autoload.php';

use Kreait\Firebase\Factory;
use Kreait\Firebase\Exception\Auth\InvalidToken;
use Kreait\Firebase\Auth;

// Initialize Firebase
$factory = (new Factory)
    ->withServiceAccount(__DIR__ . '/firebase-service-account.json')
    ->withDatabaseUri('https://<YOUR_PROJECT_ID>.firebaseio.com'); // optional
$auth = $factory->createAuth();

// — Parse Bearer ID token from Authorization header —
$authHeader = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
if (!preg_match('/^Bearer\s+(.+)$/', $authHeader, $matches)) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Missing or invalid Authorization header.']);
    exit;
}
$idTokenString = $matches[1];

// — Verify ID token with Firebase —
try {
    $verifiedIdToken = $auth->verifyIdToken($idTokenString);
} catch (InvalidToken $e) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Invalid ID token: '.$e->getMessage()]);
    exit;
}

$uid        = $verifiedIdToken->getClaim('sub');         // Firebase UID
$phone      = $verifiedIdToken->getClaim('phone_number'); // e.g. "+919876543210"

// — Read & validate request body —
$input = json_decode(file_get_contents('php://input'), true);
$username = trim($input['username']  ?? '');
$email    = trim($input['email']     ?? '');
$password =           $input['password'] ?? '';

$errors = [];
if ($username === '') {
    $errors[] = 'Username is required.';
}
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $errors[] = 'A valid email is required.';
}
if (strlen($password) < 6) {
    $errors[] = 'Password must be at least 6 characters.';
}
if ($errors) {
    http_response_code(422);
    echo json_encode(['success' => false, 'message' => implode(' ', $errors)]);
    exit;
}

// — Connect your MySQL (or other) DB —
require_once __DIR__ . '/db.php'; // should set up `$conn` as mysqli

try {
    // 1) Check for existing email or phone
    $stmt = $conn->prepare("
        SELECT
          SUM(email  = ?) AS email_exists,
          SUM(mobile = ?) AS mobile_exists
        FROM users_data
    ");
    $stmt->bind_param('ss', $email, $phone);
    $stmt->execute();
    $stmt->bind_result($emailExists, $mobileExists);
    $stmt->fetch();
    $stmt->close();

    if ($emailExists) {
        throw new Exception('That email is already registered.', 409);
    }
    if ($mobileExists) {
        throw new Exception('That mobile number is already registered.', 409);
    }

    // 2) Hash & insert
    $hash = password_hash($password, PASSWORD_DEFAULT);
    $ins = $conn->prepare("
      INSERT INTO users_data
        (firebase_uid, username, email, mobile, password, created_at)
      VALUES (?, ?, ?, ?, ?, NOW())
    ");
    $ins->bind_param('sssss', $uid, $username, $email, $phone, $hash);
    if (! $ins->execute()) {
        throw new Exception('Database error: ' . $ins->error, 500);
    }
    $newId = $ins->insert_id;
    $ins->close();

    // 3) Success
    http_response_code(201);
    echo json_encode([
        'success' => true,
        'message' => 'Registration successful.',
        'data'    => [
            'user_id'     => $newId,
            'firebase_uid'=> $uid,
            'username'    => $username,
            'email'       => $email,
            'mobile'      => $phone,
        ],
    ]);
} catch (Exception $e) {
    $code = $e->getCode();
    if ($code < 400 || $code >= 600) {
        $code = 500;
    }
    http_response_code($code);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
