<?php
// register_api.php

// — Show errors during development (disable in production) —
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// — CORS & JSON headers —
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Content-Type: application/json; charset=UTF-8');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

require_once __DIR__ . '/db.php';

$input = json_decode(file_get_contents('php://input'), true);
if ($_SERVER['REQUEST_METHOD'] !== 'POST' || !is_array($input)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Invalid request or malformed JSON.'
    ]);
    exit;
}

// — Trim & extract —
$username = trim($input['username'] ?? '');
$mobile   = trim($input['mobile']   ?? '');
$mpin     = $input['mpin']     ?? '';

// — Validate —
$errors = [];
if ($username === '') {
    $errors[] = 'Username is required.';
}
if (!preg_match('/^[0-9]{10,15}$/', $mobile)) {
    $errors[] = 'Mobile must be 10–15 digits.';
}
if (!preg_match('/^\d{4}$/', $mpin)) {
     $errors[] = 'MPIN must be exactly 4 digits.';
 }
if ($errors) {
    http_response_code(422);
    echo json_encode([
        'success' => false,
        'message' => implode(' ', $errors)
    ]);
    exit;
}

try {
    // — Check mobile uniqueness —
    $stmt = $conn->prepare("
        SELECT SUM(mobile = ?) AS mobile_exists
          FROM users_data
    ");
    $stmt->bind_param('s', $mobile);
    $stmt->execute();
    $stmt->bind_result($mobileExists);
    $stmt->fetch();
    $stmt->close();

    if ($mobileExists) {
        throw new Exception('That mobile number is already registered.', 409);
    }

    // — Hash & insert —
    $hash = $mpin;
    $ins = $conn->prepare("
        INSERT INTO users_data
          (username, mobile, password, created_at)
        VALUES (?, ?, ?, NOW())
    ");
    $ins->bind_param('sss', $username, $mobile, $hash);
    if (! $ins->execute()) {
        throw new Exception('Database error: ' . $ins->error, 500);
    }
    $newId = $ins->insert_id;
    $ins->close();

    http_response_code(201);
    echo json_encode([
        'success' => true,
        'message' => 'Registration successful.',
        'data'    => [
            'user_id'  => $newId,
            'username' => $username,
            'mobile'   => $mobile,
        ],
    ]);
    exit;

} catch (Exception $e) {
    $code = $e->getCode();
    if ($code < 400 || $code >= 600) {
        $code = 500;
    }
    http_response_code($code);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
    exit;
}
