<?php
// show all errors (remove in production)
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

session_start();
include __DIR__ . '/db.php';  // your mysqli connection

// 1) Require a game_id in the URL
if (empty($_GET['game_id']) || !ctype_digit($_GET['game_id'])) {
    die("No game specified. Provide ?game_id=ID in the URL.");
}
$game_id = (int)$_GET['game_id'];

// 2) Fetch game metadata
$stmt = $conn->prepare("
  SELECT game_name, sub_games, winning_number
    FROM game56
   WHERE id = ?
");
$stmt->bind_param("i", $game_id);
$stmt->execute();
$stmt->bind_result($game_name, $subname, $winning_number);
if (!$stmt->fetch()) {
    die("Game not found.");
}
$stmt->close();

// 3) Pull every bid for this game, with username and current wallet
$q = $conn->prepare("
  SELECT 
    ub.user_id,
    ub.bid_details,
    u.username,
    w.balance
  FROM user_bids AS ub
  JOIN users_data AS u ON ub.user_id = u.user_id
  JOIN wallet     AS w ON ub.user_id = w.user_id
  WHERE ub.game_id = ?
");
$q->bind_param("i", $game_id);
$q->execute();
$q->store_result();
$q->bind_result($user_id, $bid_json, $username, $current_balance);

$rows = [];
while ($q->fetch()) {
    $bids = json_decode($bid_json, true) ?: [];
    // did they bet on the winning number?
    if (isset($bids[$winning_number]) && $bids[$winning_number] >= 10) {
        $stake  = (float)$bids[$winning_number];
        $payout = $stake * 10;  // 10× payout

        // credit their wallet
        $u = $conn->prepare("
          UPDATE wallet 
             SET balance = balance + ? 
           WHERE user_id = ?
        ");
        $u->bind_param("di", $payout, $user_id);
        $u->execute();
        $u->close();

        $new_balance = $current_balance + $payout;
        $result_text = "Won ₹" . number_format($payout,2);
    } else {
        $new_balance = $current_balance;
        $result_text = "Lost";
    }

    $rows[] = [
      'username'      => $username,
      'bid_details'   => $bids,
      'result'        => $result_text,
      'new_balance'   => $new_balance
    ];
}
$q->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Game Result</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <style>
    body { font-family:Arial,sans-serif; background:#f4f4f9; color:#333; margin:0; padding:20px }
    .container { max-width:900px; margin:0 auto; background:#fff; padding:20px; 
                 border-radius:8px; box-shadow:0 2px 5px rgba(0,0,0,0.1) }
    h1 { text-align:center; color:#1E90FF; margin-bottom:10px }
    .meta { margin-bottom:20px; }
    .meta div { margin:4px 0; }
    table { width:100%; border-collapse:collapse; margin-top:10px }
    th, td { border:1px solid #ddd; padding:8px; text-align:center; vertical-align:top }
    th { background:#007BFF; color:#fff; }
    .bid-list { text-align:left; font-size:0.9rem; }
    .bid-list div { margin:2px 0; }
    .won { color:green; font-weight:bold; }
    .lost { color:red; }
  </style>
</head>
<body>
  <div class="container">
    <h1>Result for “<?= htmlspecialchars($game_name) ?>”</h1>

    <div class="meta">
      <div><strong>Sub‑game:</strong> <?= htmlspecialchars($subname) ?></div>
      <div><strong>Winning Number:</strong> <?= $winning_number ?></div>
    </div>

    <table>
      <tr>
        <th>User Name</th>
        <th>Bid Details</th>
        <th>Result</th>
        <th>New Wallet Balance (₹)</th>
      </tr>
      <?php if (empty($rows)): ?>
        <tr><td colspan="4">No bids placed for this game.</td></tr>
      <?php else: ?>
        <?php foreach ($rows as $r): ?>
          <tr>
            <td><?= htmlspecialchars($r['username']) ?></td>
            <td class="bid-list">
              <?php if (empty($r['bid_details'])): ?>
                <em>No bids</em>
              <?php else: ?>
                <?php foreach ($r['bid_details'] as $digit => $amt): ?>
                  <div><?= $digit ?> &rarr; ₹<?= number_format($amt,2) ?></div>
                <?php endforeach; ?>
              <?php endif; ?>
            </td>
            <td class="<?= strpos($r['result'],'Won')===0 ? 'won' : 'lost' ?>">
              <?= htmlspecialchars($r['result']) ?>
            </td>
            <td><?= number_format($r['new_balance'],2) ?></td>
          </tr>
        <?php endforeach; ?>
      <?php endif; ?>
    </table>
  </div>
</body>
</html>
