<?php
// DEBUG: show all errors
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

session_start();
date_default_timezone_set('Asia/Kolkata');

// 1) Auth
if (empty($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// 2) DB
require_once __DIR__ . '/db.php';
if ($conn->connect_error) {
    die("DB Connect Error: " . $conn->connect_error);
}
$conn->set_charset('utf8mb4');

// 3) Date filter (clamp future dates to today)
$today       = date('Y-m-d');
$filter_date = $_GET['filter_date'] ?? $today;
if ($filter_date > $today) {
    $filter_date = $today;
}

// 4) Seed dp_result (only if dp_rewards exists for this date)
$seedSQL = "
  INSERT IGNORE INTO dp_result
    (reward_id,user_id,username,bid_number,bid_amount,bid_session,
     win_open_dp,win_close_dp,res_open,res_close)
  SELECT
    r.id,
    b.user_id,
    u.username,
    b.bid_details,
    b.total_bid_amount,
    b.bid_session,
    r.winning_open_time_dp,
    r.winning_close_time_dp,
    IF(
      b.bid_session='open'
      AND b.bid_details = r.winning_open_time_dp,
      CONCAT('🏆 ₹',FORMAT(b.total_bid_amount*300,2)),
      '❌'
    ),
    IF(
      b.bid_session='close'
      AND b.bid_details = r.winning_close_time_dp,
      CONCAT('🏆 ₹',FORMAT(b.total_bid_amount*300,2)),
      '❌'
    )
  FROM dp_rewards    AS r
  JOIN user_bids_dp  AS b
    ON b.game_id = r.game_id
   AND DATE(b.created_at) = r.reward_date
  JOIN users_data    AS u
    ON u.user_id = b.user_id
  WHERE r.reward_date = ?
";
$ins = $conn->prepare($seedSQL);
$ins->bind_param('s', $filter_date);
$ins->execute();
$ins->close();

// 5) Credit winners once
foreach (['open'=>'paid_open','close'=>'paid_close'] as $side=>$flag) {
    $credit = "
      UPDATE wallet w
      JOIN dp_result d ON d.user_id = w.user_id
      JOIN dp_rewards r ON r.id = d.reward_id
      SET 
        w.balance = w.balance + (d.bid_amount * 300),
        d.$flag    = 1
      WHERE r.reward_date = ?
        AND d.bid_session = ?
        AND d.bid_number = IF(d.bid_session='open',d.win_open_dp,d.win_close_dp)
        AND d.$flag = 0
    ";
    $u = $conn->prepare($credit);
    $u->bind_param('ss', $filter_date, $side);
    $u->execute();
    $u->close();
}

// 6) Fetch & display — show all bids for the date, reward columns empty until dp_rewards exists
$rows = [];
$sel = "
  SELECT
    'Double Panna' AS game_name,
    DATE_FORMAT(r.open_time,'%l:%i %p')  AS open_fmt,
    DATE_FORMAT(r.close_time,'%l:%i %p') AS close_fmt,
    u.username,
    b.bid_session,
    b.bid_details    AS bid_number,
    b.total_bid_amount AS bid_amount,
    r.winning_open_time_dp  AS win_open_dp,
    r.winning_close_time_dp AS win_close_dp
  FROM user_bids_dp  AS b
  JOIN users_data    AS u
    ON u.user_id = b.user_id
  LEFT JOIN dp_rewards AS r
    ON r.game_id    = b.game_id
   AND r.reward_date = ?
  WHERE DATE(b.created_at) = ?
  ORDER BY b.created_at
";
$st = $conn->prepare($sel);
$st->bind_param('ss', $filter_date, $filter_date);
$st->execute();
$st->bind_result(
  $gname, $open_fmt, $close_fmt,
  $username, $bid_session,
  $bid_number, $bid_amount,
  $win_open_dp, $win_close_dp
);
while ($st->fetch()) {
    $rows[] = [
      'game_name'  => $gname,
      'open_fmt'   => $open_fmt,
      'close_fmt'  => $close_fmt,
      'username'   => $username,
      'bid_session'=> $bid_session,
      'bid_number' => $bid_number,
      'bid_amount' => $bid_amount,
      'win_open'   => $win_open_dp ?? '',
      'win_close'  => $win_close_dp ?? '',
      'res_open'   => ($bid_session==='open' && $win_open_dp !== null && $bid_number===$win_open_dp)
                      ? '🏆 ₹'.number_format($bid_amount*300,2)
                      : ($win_open_dp !== null ? '❌' : ''),
      'res_close'  => ($bid_session==='close' && $win_close_dp !== null && $bid_number===$win_close_dp)
                      ? '🏆 ₹'.number_format($bid_amount*300,2)
                      : ($win_close_dp !== null ? '❌' : ''),
    ];
}
$st->close();
$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Double Panna : Player Results</title>
  <meta http-equiv="refresh" content="10">
  <style>
    body { background:#f5f6f7; font-family:Arial,sans-serif; margin:0; padding:2rem; }
    .container { max-width:1000px; margin:0 auto; background:#fff;
                 padding:1.5rem; border-radius:8px;
                 box-shadow:0 2px 8px rgba(0,0,0,.1); }
    h1 { margin-top:0; }
    .controls { margin-bottom:1rem; text-align:center; }
    .controls input, .controls select { padding:.5rem; font-size:1rem; margin:0 .5rem; }
    table { width:100%; border-collapse:collapse; margin-top:1rem; }
    th, td { padding:8px; border:1px solid #eee; text-align:center; }
    th { background:#007bff; color:#fff; }
    td.left { text-align:left; }
  </style>
</head>
<body>
  <div class="container">
    <h1>Double Panna : Player Results</h1>

    <div class="controls">
      <form style="display:inline" action="" method="GET">
        <label for="filter_date">Date:</label>
        <input
          type="date" id="filter_date" name="filter_date"
          value="<?= htmlspecialchars($filter_date,ENT_QUOTES)?>"
          max="<?= $today ?>"
          onchange="this.form.submit()"
        >
      </form>

      <?php $current = basename($_SERVER['PHP_SELF']); ?>
      <form style="display:inline">
        <label for="view_select">View Results:</label>
        <select
          id="view_select"
          onchange="location.href=this.value + '?filter_date=<?= htmlspecialchars($filter_date)?>'"
        >
          <option value="result_sd.php">Single Digit</option>
          <option value="result_db2.php" selected>Double Panna</option>
          <option value="result_tp.php">Triple Panna</option>
          <option value="result_jd.php">Jodi Digit</option>
          <option value="result_sp.php">Single Panna</option>
        </select>
      </form>
    </div>

    <table>
      <thead>
        <tr>
          <th>Game</th><th>Open</th><th>Close</th>
          <th>User</th><th>Session</th><th>Bid #</th><th>Amt</th>
          <th>Win # (O)</th><th>Win # (C)</th>
          <th>Result O</th><th>Result C</th>
        </tr>
      </thead>
      <tbody>
        <?php if (empty($rows)): ?>
          <tr>
            <?php for ($i = 0; $i < 11; $i++): ?>
              <td>&nbsp;</td>
            <?php endfor; ?>
          </tr>
        <?php else: ?>
          <?php foreach($rows as $r): ?>
            <tr>
              <td><?= htmlspecialchars($r['game_name']) ?></td>
              <td><?= htmlspecialchars($r['open_fmt']) ?></td>
              <td><?= htmlspecialchars($r['close_fmt']) ?></td>
              <td class="left"><?= htmlspecialchars($r['username']) ?></td>
              <td><?= ucfirst(htmlspecialchars($r['bid_session'])) ?></td>
              <td><?= htmlspecialchars($r['bid_number']) ?></td>
              <td>₹<?= number_format($r['bid_amount'],2) ?></td>
              <td><?= htmlspecialchars($r['win_open']) ?></td>
              <td><?= htmlspecialchars($r['win_close']) ?></td>
              <td><?= htmlspecialchars($r['res_open']) ?></td>
              <td><?= htmlspecialchars($r['res_close']) ?></td>
            </tr>
          <?php endforeach; ?>
        <?php endif; ?>
      </tbody>
    </table>
  </div>
</body>
</html>
