<?php
/****************************************************************
 * result_fsp.php — Full-Sangam results & wallet updater + display (live-reloading)
 ****************************************************************/

// hide notices/warnings, but still log errors
ini_set('display_errors', 1);
ini_set('log_errors',     1);
ini_set('error_log',      __DIR__.'/result_fsp_error.log');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);

session_start();
date_default_timezone_set('Asia/Kolkata');

require_once __DIR__.'/db.php';
$conn->set_charset('utf8mb4');

// 1) Auth guard
if (empty($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// 2) Check if this is an AJAX request for table data only
$isAjaxRequest = isset($_GET['ajax']) && $_GET['ajax'] === '1';

// 3) Date filter
$filter_date = (isset($_GET['filter_date']) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $_GET['filter_date']))
    ? $_GET['filter_date']
    : date('Y-m-d');

// 4) Payout multiplier
$FS_MULT = 100000;

// 5) Explode ticket function for Full Sangam bids
function explode_fs_ticket($open_raw, $close_raw, $stake) {
    $open_vals = [];
    $close_vals = [];
    
    // Process open values
    if (!empty($open_raw)) {
        $j = json_decode($open_raw, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($j)) {
            foreach ($j as $val => $amt) {
                $open_vals[] = [$val, (float)$amt];
            }
        } else {
            // Fallback parsing
            foreach (preg_split('/[,\s]+/', $open_raw) as $val) {
                if ($val !== '' && is_numeric($val)) {
                    $open_vals[] = [$val, $stake];
                }
            }
        }
    }
    
    // Process close values  
    if (!empty($close_raw)) {
        $j = json_decode($close_raw, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($j)) {
            foreach ($j as $val => $amt) {
                $close_vals[] = [$val, (float)$amt];
            }
        } else {
            // Fallback parsing
            foreach (preg_split('/[,\s]+/', $close_raw) as $val) {
                if ($val !== '' && is_numeric($val)) {
                    $close_vals[] = [$val, $stake];
                }
            }
        }
    }
    
    return [$open_vals, $close_vals];
}

// 6) Load today's rewards
$rewards = [];
$stmt = $conn->prepare("
    SELECT r.id, r.game_id, r.winning_open_op_fs, r.winning_open_cp_fs,
           r.winning_close_op_fs, r.winning_close_cp_fs,
           g.game_name, g.open_time, g.close_time
    FROM fs_rewards r
    JOIN game56 g ON g.id = r.game_id
    WHERE r.reward_date = ?
");
$stmt->bind_param('s', $filter_date);
$stmt->execute();
$result = $stmt->get_result();

while ($row = $result->fetch_assoc()) {
    $rewards[$row['game_id']] = [
        'rid'   => $row['id'],
        'open_op'  => $row['winning_open_op_fs'],
        'open_cp'  => $row['winning_open_cp_fs'],
        'close_op' => $row['winning_close_op_fs'],
        'close_cp' => $row['winning_close_cp_fs'],
        'game_name' => $row['game_name'],
        'open_time' => $row['open_time'],
        'close_time' => $row['close_time']
    ];
}
$stmt->close();

if (empty($rewards)) {
    if ($isAjaxRequest) {
        echo '<tr><td colspan="17" style="text-align:center;color:#666;">No rewards set for ' . htmlspecialchars($filter_date) . '</td></tr>';
        exit;
    }
    
    echo '<!DOCTYPE html>
<html><head><meta charset="utf-8"><title>No Rewards</title>
<style>body{font-family:Arial,sans-serif;text-align:center;padding:2rem}</style>
</head><body>
  <h1>Full-Sangam Results</h1>
  <p>No rewards set for <strong>' . htmlspecialchars($filter_date) . '</strong>.</p>
</body></html>';
    exit;
}

// 7) Clear existing results for today first to prevent duplicates
$gameList = implode(',', array_keys($rewards));
$clear_stmt = $conn->prepare("
    DELETE fs FROM fs_result fs 
    JOIN fs_rewards r ON r.id = fs.reward_id 
    WHERE r.reward_date = ? AND r.game_id IN ({$gameList})
");
$clear_stmt->bind_param('s', $filter_date);
$clear_stmt->execute();
$clear_stmt->close();

// 8) Get all bids for today and process them fresh
$sql = "
    SELECT b.user_id, u.username, b.game_id, b.bid_session, b.bid_open_fs, b.bid_close_fs, b.total_bid_amount, b.created_at
    FROM user_bids_fs b
    JOIN users_data u ON u.user_id = b.user_id
    WHERE DATE(b.created_at) = ?
    AND b.game_id IN ({$gameList})
    ORDER BY b.created_at DESC
";

$bst = $conn->prepare($sql);
$bst->bind_param('s', $filter_date);
$bst->execute();
$bidResult = $bst->get_result();

// 9) Process each bid and insert into fs_result
$ist = $conn->prepare("
    INSERT INTO fs_result
    (reward_id, user_id, username, bid_open_fs, bid_close_fs, bid_amount, bid_session, 
     win_open_op_fs, win_open_cp_fs, win_close_op_fs, win_close_cp_fs,
     res_open_op, res_open_cp, res_close_op, res_close_cp)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
");

$current_time = time();
while ($bid = $bidResult->fetch_assoc()) {
    $gID = $bid['game_id'];
    if (!isset($rewards[$gID])) continue;
    
    $cfg = $rewards[$gID];
    
    // Calculate results - For Full Sangam, all results go in Open columns only
    $resOpenOP = '–';
    $resOpenCP = '–';
    $resCloseOP = '–';
    $resCloseCP = '–';
    
    // For Full Sangam, always calculate for open session results only
    $open_timestamp = strtotime($cfg['open_time']);
    
    // Open OP result
    if (!is_null($cfg['open_op'])) {
        if ($current_time >= $open_timestamp) {
            if ($bid['bid_open_fs'] == $cfg['open_op']) {
                $resOpenOP = "🏆 ₹" . number_format($bid['total_bid_amount'] * $FS_MULT, 2);
            } else {
                $resOpenOP = "❌ Better luck";
            }
        } else {
            $resOpenOP = 'Pending';
        }
    } else {
        $resOpenOP = 'Pending';
    }
    
    // Open CP result
    if (!is_null($cfg['open_cp'])) {
        if ($current_time >= $open_timestamp) {
            if ($bid['bid_close_fs'] == $cfg['open_cp']) {
                $resOpenCP = "🏆 ₹" . number_format($bid['total_bid_amount'] * $FS_MULT, 2);
            } else {
                $resOpenCP = "❌ Better luck";
            }
        } else {
            $resOpenCP = 'Pending';
        }
    } else {
        $resOpenCP = 'Pending';
    }
    
    // For Full Sangam, Close result columns always show "–" since only open session is used
    
    $session = 'open'; // Force session to open for Full Sangam
    $ist->bind_param(
        'iisssdsssssssss',
        $cfg['rid'],
        $bid['user_id'],
        $bid['username'],
        $bid['bid_open_fs'],
        $bid['bid_close_fs'],
        $bid['total_bid_amount'],
        $session, // Use variable instead of $bid['bid_session']
        $cfg['open_op'],
        $cfg['open_cp'],
        $cfg['close_op'],
        $cfg['close_cp'],
        $resOpenOP,
        $resOpenCP,
        $resCloseOP,
        $resCloseCP
    );
    $ist->execute();
}

$bst->close();
$ist->close();

// 10) Credit winners (only for completed games and not already paid)
foreach ([
  'res_open_op'  => 'paid_open_op',
  'res_open_cp'  => 'paid_open_cp',
  'res_close_op' => 'paid_close_op',
  'res_close_cp' => 'paid_close_cp'
] as $resCol => $paidCol) {
    $up = $conn->prepare("
        UPDATE wallet w
        JOIN fs_result d ON d.user_id = w.user_id
        JOIN fs_rewards r ON r.id = d.reward_id
        SET w.balance = w.balance + (d.bid_amount * ?),
            d.{$paidCol} = 1
        WHERE r.reward_date = ? 
        AND d.{$paidCol} = 0
        AND d.{$resCol} LIKE '🏆%'
    ");
    $up->bind_param('ds', $FS_MULT, $filter_date);
    $up->execute();
    $up->close();
}

// 11) Fetch final results
$rows = [];
$rep = $conn->prepare("
    SELECT
        r.game_name,
        DATE_FORMAT(g.open_time, '%l:%i %p') AS fmt_open,
        DATE_FORMAT(g.close_time, '%l:%i %p') AS fmt_close,
        d.username, d.bid_open_fs, d.bid_close_fs,
        d.bid_amount, d.bid_session,
        d.win_open_op_fs, d.win_open_cp_fs,
        d.win_close_op_fs, d.win_close_cp_fs,
        d.res_open_op, d.res_open_cp,
        d.res_close_op, d.res_close_cp
    FROM fs_result d
    JOIN fs_rewards r ON r.id = d.reward_id
    JOIN game56 g ON g.id = r.game_id
    WHERE r.reward_date = ?
    ORDER BY d.id DESC, r.open_time, d.bid_session, d.username
");
$rep->bind_param('s', $filter_date);
$rep->execute();
$result = $rep->get_result();

while ($row = $result->fetch_assoc()) {
    $rows[] = [
        'gName' => $row['game_name'],
        'fmtO'  => $row['fmt_open'],
        'fmtC'  => $row['fmt_close'],
        'uN'    => $row['username'],
        'bidOP' => $row['bid_open_fs'],
        'bidCP' => $row['bid_close_fs'],
        'amt'   => $row['bid_amount'],
        'ss'    => ucfirst($row['bid_session']),
        'winOpOpen'  => $row['win_open_op_fs'],
        'winCpOpen'  => $row['win_open_cp_fs'],
        'winOpClose' => $row['win_close_op_fs'],
        'winCpClose' => $row['win_close_cp_fs'],
        'resOpOpen'  => $row['res_open_op'],
        'resCpOpen'  => $row['res_open_cp'],
        'resOpClose' => $row['res_close_op'],
        'resCpClose' => $row['res_close_cp']
    ];
}
$rep->close();
$conn->close();

// 12) If AJAX request, return only table rows
if ($isAjaxRequest) {
    if (empty($rows)) {
        echo '<tr><td colspan="17" style="text-align:center;color:#666;">No bids found for ' . htmlspecialchars($filter_date) . '</td></tr>';
    } else {
        foreach($rows as $r) {
            echo '<tr>';
            echo '<td>' . htmlspecialchars($r['gName']) . '</td>';
            echo '<td>' . htmlspecialchars($r['fmtO']) . '</td>';
            echo '<td>' . htmlspecialchars($r['fmtC']) . '</td>';
            echo '<td class="left">' . htmlspecialchars($r['uN']) . '</td>';
            echo '<td>' . htmlspecialchars($r['ss']) . '</td>';
            echo '<td>' . ($r['bidOP'] !== null ? htmlspecialchars($r['bidOP']) : '–') . '</td>';
            echo '<td>' . ($r['bidCP'] !== null ? htmlspecialchars($r['bidCP']) : '–') . '</td>';
            echo '<td>₹' . number_format($r['amt'], 2) . '</td>';
            echo '<td>' . ($r['winOpOpen'] !== null ? htmlspecialchars($r['winOpOpen']) : '–') . '</td>';
            echo '<td>' . ($r['winCpOpen'] !== null ? htmlspecialchars($r['winCpOpen']) : '–') . '</td>';
            echo '<td>' . ($r['winOpClose'] !== null ? htmlspecialchars($r['winOpClose']) : '–') . '</td>';
            echo '<td>' . ($r['winCpClose'] !== null ? htmlspecialchars($r['winCpClose']) : '–') . '</td>';
            echo '<td>' . htmlspecialchars($r['resOpOpen']) . '</td>';
            echo '<td>' . htmlspecialchars($r['resCpOpen']) . '</td>';
            echo '<td>' . htmlspecialchars($r['resOpClose']) . '</td>';
            echo '<td>' . htmlspecialchars($r['resCpClose']) . '</td>';
            echo '</tr>';
        }
    }
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Full-Sangam : Player Results</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <style>
    body { font-family: Arial, sans-serif; background: #f5f6f7; margin: 0; padding: 2rem; }
    .container { max-width: 1400px; margin: 0 auto; background: #fff; padding: 1.5rem; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,.1); }
    table { width: 100%; border-collapse: collapse; margin-top: 1rem; }
    th, td { padding: 6px; border: 1px solid #eee; text-align: center; font-size: 0.75rem; }
    th { background: #28a745; color: #fff; font-weight: bold; }
    td.left { text-align: left; }
    .controls { margin-bottom: 1rem; text-align: center; display: flex; justify-content: center; align-items: center; gap: 1rem; flex-wrap: wrap; }
    .controls input, .controls select { padding: .5rem; font-size: 1rem; border: 1px solid #ccc; border-radius: 4px; }
    .controls label { font-weight: bold; }
    .status-indicator { 
        position: fixed; 
        top: 10px; 
        right: 10px; 
        background: #28a745; 
        color: white; 
        padding: 5px 10px; 
        border-radius: 4px; 
        font-size: 0.8rem;
        z-index: 1000;
    }
    .status-indicator.updating { background: #ffc107; color: #000; }
    .status-indicator.error { background: #dc3545; }
    .highlight-new { 
        background-color: #e8f5e8 !important; 
        animation: fadeHighlight 3s ease-out;
    }
    @keyframes fadeHighlight {
        0% { background-color: #90EE90 !important; }
        100% { background-color: transparent !important; }
    }
    @media (max-width: 768px) {
        .container { padding: 1rem; }
        th, td { padding: 4px; font-size: 0.7rem; }
        .controls { flex-direction: column; gap: 0.5rem; }
    }
  </style>
</head>
<body class="bg-light">
  <div class="status-indicator" id="status">Live</div>
  
  <div class="container">
    <div class="card shadow-sm">
      <div class="card-body">

        <div class="d-flex justify-content-between align-items-center mb-3">
          <h1 class="h4 mb-0">Full-Sangam : Player Results</h1>
          <div class="controls">
            <form style="display: inline">
                <label>Date:</label>
                <input type="date" name="filter_date" value="<?=htmlspecialchars($filter_date)?>" onchange="this.form.submit()">
            </form>
            <form style="display: inline">
                <label>View Results:</label>
                <select onchange="location.href=this.value + '?filter_date=' + encodeURIComponent('<?=htmlspecialchars($filter_date)?>')">
                    <option value="resultsd.php">Single Digit Results</option>
                    <option value="result_dp.php">Double Panna</option>
                    <option value="result_tp.php">Triple Panna</option>
                    <option value="result_jd.php">Jodi Digit</option>
                    <option value="result_sp.php">Single Panna</option>
                    <option value="resulths.php">Half Sangam</option>
                    <option value="result_fsp.php" selected>Full Sangam</option>
                </select>
            </form>
          </div>
        </div>

        <?php if (empty($rows)): ?>
          <div class="alert alert-info">
            No bids found for <strong><?=htmlspecialchars($filter_date)?></strong>.
          </div>
        <?php else: ?>
          <div class="table-responsive">
            <table class="table table-bordered text-center align-middle mb-0">
              <thead>
                <tr>
                  <th>Game</th>
                  <th>Open</th>
                  <th>Close</th>
                  <th>User</th>
                  <th>Session</th>
                  <th>Bid OP</th>
                  <th>Bid CP</th>
                  <th>Amount</th>
                  <th>Win OP (O)</th><th>Win CP (O)</th>
                  <th>Win OP (C)</th><th>Win CP (C)</th>
                  <th>Res OP (O)</th>
                  <th>Res CP (O)</th>
                  <th>Res OP (C)</th>
                  <th>Res CP (C)</th>
                </tr>
              </thead>
              <tbody id="bids-tbody">
                <?php foreach ($rows as $r): ?>
                  <tr>
                    <td><?=htmlspecialchars($r['gName'])?></td>
                    <td><?=htmlspecialchars($r['fmtO'])?></td>
                    <td><?=htmlspecialchars($r['fmtC'])?></td>
                    <td class="left"><?=htmlspecialchars($r['uN'])?></td>
                    <td><?=htmlspecialchars($r['ss'])?></td>
                    <td><?= $r['bidOP'] !== null ? htmlspecialchars($r['bidOP']) : '–' ?></td>
                    <td><?= $r['bidCP'] !== null ? htmlspecialchars($r['bidCP']) : '–' ?></td>
                    <td>₹<?=number_format($r['amt'],2)?></td>
                    <td><?= $r['winOpOpen'] !== null ? htmlspecialchars($r['winOpOpen']) : '–' ?></td>
                    <td><?= $r['winCpOpen'] !== null ? htmlspecialchars($r['winCpOpen']) : '–' ?></td>
                    <td><?= $r['winOpClose'] !== null ? htmlspecialchars($r['winOpClose']) : '–' ?></td>
                    <td><?= $r['winCpClose'] !== null ? htmlspecialchars($r['winCpClose']) : '–' ?></td>
                    <td><?=htmlspecialchars($r['resOpOpen'])?></td>
                    <td><?=htmlspecialchars($r['resCpOpen'])?></td>
                    <td><?=htmlspecialchars($r['resOpClose'])?></td>
                    <td><?=htmlspecialchars($r['resCpClose'])?></td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>

      </div>
    </div>
  </div>
  
  <script>
    let lastRowCount = <?= count($rows) ?>;
    
    function updateResults() {
        const status = document.getElementById('status');
        const tbody = document.getElementById('bids-tbody');
        
        status.textContent = 'Updating...';
        status.className = 'status-indicator updating';
        
        fetch('result_fsp.php?ajax=1&filter_date=' + encodeURIComponent('<?= htmlspecialchars($filter_date) ?>'))
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.text();
            })
            .then(data => {
                if (data.trim()) {
                    const tempDiv = document.createElement('div');
                    tempDiv.innerHTML = '<table><tbody>' + data + '</tbody></table>';
                    const newRowCount = tempDiv.querySelectorAll('tr').length;
                    
                    // Only update if there are changes
                    if (tbody.innerHTML !== data) {
                        tbody.innerHTML = data;
                        
                        // Highlight if new rows were added
                        if (newRowCount > lastRowCount) {
                            const newRows = tbody.querySelectorAll('tr');
                            for (let i = 0; i < newRowCount - lastRowCount; i++) {
                                if (newRows[i]) {
                                    newRows[i].classList.add('highlight-new');
                                }
                            }
                        }
                        
                        lastRowCount = newRowCount;
                    }
                }
                
                status.textContent = 'Live';
                status.className = 'status-indicator';
            })
            .catch(error => {
                console.error('Error updating results:', error);
                status.textContent = 'Error';
                status.className = 'status-indicator error';
            });
    }
    
    // Update every 5 seconds
    setInterval(updateResults, 5000);
    
    // Also update when page becomes visible (user switches back to tab)
    document.addEventListener('visibilitychange', function() {
        if (!document.hidden) {
            setTimeout(updateResults, 500);
        }
    });
    
    // Remove highlight classes after animation
    document.addEventListener('animationend', function(e) {
        if (e.animationName === 'fadeHighlight') {
            e.target.classList.remove('highlight-new');
        }
    });
  </script>

</body>
</html>