<?php
// result_jd.php — Jodi-Digit Player Results (live-reloading with timing logic)

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();
date_default_timezone_set('Asia/Kolkata');

// 1) Auth
if (empty($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// 2) DB
require_once __DIR__ . '/db.php';
if ($conn->connect_error) {
    die("DB Connect Error: " . $conn->connect_error);
}
$conn->set_charset('utf8mb4');

// 3) Check if this is an AJAX request for table data only
$isAjaxRequest = isset($_GET['ajax']) && $_GET['ajax'] === '1';

// 4) Date filter
$filter_date = $_GET['filter_date'] ?? date('Y-m-d');

// 5) Payout multiplier
$JD_MULT = 100;

// 6) Explode ticket for JD format - handles JD number format
function explode_ticket_jd($raw, $fallback_stake = 0) {
    // Handle empty or null values
    if (empty($raw)) {
        return [];
    }
    
    // For JD, bid_details is typically just the 2-digit number as string
    // like "45", "78", "12", etc.
    if (is_numeric($raw) && strlen($raw) == 2) {
        return [[(string)$raw, (float)$fallback_stake]];
    }
    
    // Try JSON decode in case it's stored as JSON
    $j = json_decode($raw, true);
    if (json_last_error() === JSON_ERROR_NONE && is_array($j)) {
        $out = [];
        foreach ($j as $key => $value) {
            if (is_numeric($key)) {
                // Format: {"45": 100, "78": 200}
                $out[] = [(string)$key, (float)$value];
            } elseif (is_numeric($value)) {
                // Format: ["45", "78"]
                $out[] = [(string)$value, (float)$fallback_stake];
            }
        }
        return $out;
    }
    
    // Fallback: treat as string
    return [[(string)$raw, (float)$fallback_stake]];
}

// 7) Load today's rewards
$rewards = [];
$stmt = $conn->prepare("
    SELECT r.id, r.game_id, r.winning_open_time_jd, r.winning_close_time_jd,
           g.game_name, g.open_time, g.close_time
    FROM jd_rewards r
    JOIN game56 g ON g.id = r.game_id
    WHERE r.reward_date = ?
");
$stmt->bind_param('s', $filter_date);
$stmt->execute();
$result = $stmt->get_result();

while ($row = $result->fetch_assoc()) {
    $rewards[$row['game_id']] = [
        'rid'   => $row['id'],
        'open'  => is_null($row['winning_open_time_jd']) ? null : (string)$row['winning_open_time_jd'],
        'close' => is_null($row['winning_close_time_jd']) ? null : (string)$row['winning_close_time_jd'],
        'game_name' => $row['game_name'],
        'open_time' => $row['open_time'],
        'close_time' => $row['close_time']
    ];
}
$stmt->close();

if (empty($rewards)) {
    if ($isAjaxRequest) {
        echo '<tr><td colspan="11" style="text-align:center;color:#666;">No rewards set for ' . htmlspecialchars($filter_date) . '</td></tr>';
        exit;
    }
    
    echo <<<HTML
<!DOCTYPE html>
<html><head><meta charset="utf-8"><title>No Rewards</title>
<style>body{font-family:Arial,sans-serif;text-align:center;padding:2rem}</style>
</head><body>
  <h1>Jodi-Digit Results</h1>
  <p>No rewards set for <strong>{$filter_date}</strong>.</p>
</body></html>
HTML;
    exit;
}

// 8) Clear existing results for today first to prevent duplicates
$gameList = implode(',', array_keys($rewards));
if (!empty($gameList)) {
    $clear_stmt = $conn->prepare("
        DELETE jd FROM jd_result jd 
        JOIN jd_rewards r ON r.id = jd.reward_id 
        WHERE r.reward_date = ? AND r.game_id IN ({$gameList})
    ");
    $clear_stmt->bind_param('s', $filter_date);
    $clear_stmt->execute();
    $clear_stmt->close();
}

// 9) Get all bids for today and process them fresh
$sql = "
    SELECT b.id, b.user_id, u.username, b.game_id, b.bid_session, b.bid_details, b.total_bid_amount, b.created_at
    FROM user_bids_jd b
    JOIN users_data u ON u.user_id = b.user_id
    WHERE DATE(b.created_at) = ?
    AND b.game_id IN ({$gameList})
    ORDER BY b.created_at DESC
";

$bst = $conn->prepare($sql);
$bst->bind_param('s', $filter_date);
$bst->execute();
$bidResult = $bst->get_result();

// 10) Process each bid and insert into jd_result
$ist = $conn->prepare("
    INSERT IGNORE INTO jd_result
    (reward_id, user_id, username, bid_number, bid_amount, bid_session, win_open_jd, win_close_jd, res_open, res_close)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
");

$current_time = time();

while ($bid = $bidResult->fetch_assoc()) {
    $gID = $bid['game_id'];
    
    // Skip if game not in rewards
    if (!isset($rewards[$gID])) {
        continue;
    }
    
    $cfg = $rewards[$gID];
    
    // Parse bid details using the updated function
    $tickets = explode_ticket_jd($bid['bid_details'], (float)$bid['total_bid_amount']);
    
    if (empty($tickets)) {
        // If no tickets parsed, log and continue
        error_log("JD - Failed to parse bid details for bid ID: " . $bid['id'] . ", details: " . $bid['bid_details']);
        continue;
    }

    foreach ($tickets as $t) {
        list($number, $amt) = $t;
        
        // Skip if number or amount is invalid
        if (empty($number) || !is_numeric($amt)) {
            continue;
        }
        
        // Calculate results based on session and timing
       // Calculate results - For JD, all results go in Result O column
$resO = '–';
$resC = '–';

// For JD, always calculate for open result since all bids are open session
if (!is_null($cfg['open'])) {
    // Check if open time has passed
    $open_timestamp = strtotime($cfg['open_time']);
    if ($current_time >= $open_timestamp) {
        // Time has passed, show result in Result O column
        if ($number === $cfg['open']) {
            $resO = "🏆 ₹" . number_format($amt * $JD_MULT, 2);
        } else {
            $resO = "❌ Better luck";
        }
    } else {
        // Time hasn't passed yet
        $resO = 'Pending';
    }
} else {
    // No winning number set yet
    $resO = 'Pending';
}

// For JD, Result C column always shows "–" since only open session is used
        
$session = 'open'; // Create variable first
$ist->bind_param(
    'iissdsssss',
    $cfg['rid'],
    $bid['user_id'],
    $bid['username'],
    $number,
    $amt,
    $session, // Use variable instead of 'open' literal
    $cfg['open'],
    $cfg['close'],
    $resO,
    $resC
);
        // Execute and handle any errors
        if (!$ist->execute()) {
            error_log("Failed to insert JD result: " . $ist->error);
        }
    }
}

$bst->close();
$ist->close();

// 11) Credit winners (only for completed games and not already paid)
foreach (['open' => 'paid_open', 'close' => 'paid_close'] as $side => $flag) {
    $up = $conn->prepare("
        UPDATE wallet w
        JOIN jd_result d ON d.user_id = w.user_id
        JOIN jd_rewards r ON r.id = d.reward_id
        SET w.balance = w.balance + (d.bid_amount * ?),
            d.{$flag} = 1
        WHERE r.reward_date = ? 
        AND d.bid_session = ? 
        AND d.{$flag} = 0
        AND d.bid_number = IF(d.bid_session = 'open', d.win_open_jd, d.win_close_jd)
        AND (
            (d.bid_session = 'open' AND d.res_open LIKE '🏆%') OR
            (d.bid_session = 'close' AND d.res_close LIKE '🏆%')
        )
    ");
    $up->bind_param('iss', $JD_MULT, $filter_date, $side);
    $up->execute();
    $up->close();
}

// 12) Fetch final results
$rows = [];
$rep = $conn->prepare("
    SELECT
        r.game_name,
        DATE_FORMAT(g.open_time, '%l:%i %p') AS fmt_open,
        DATE_FORMAT(g.close_time, '%l:%i %p') AS fmt_close,
        d.username, d.bid_number,
        d.bid_amount, d.bid_session,
        d.win_open_jd, d.win_close_jd,
        d.res_open, d.res_close
    FROM jd_result d
    JOIN jd_rewards r ON r.id = d.reward_id
    JOIN game56 g ON g.id = r.game_id
    WHERE r.reward_date = ?
    ORDER BY d.id DESC
");
$rep->bind_param('s', $filter_date);
$rep->execute();
$result = $rep->get_result();

while ($row = $result->fetch_assoc()) {
    $rows[] = [
        'gName' => $row['game_name'],
        'fmtO'  => $row['fmt_open'],
        'fmtC'  => $row['fmt_close'],
        'uN'    => $row['username'],
        'dig'   => $row['bid_number'],
        'amt'   => $row['bid_amount'],
        'ss'    => ucfirst($row['bid_session']),
        'oN'    => $row['win_open_jd'],
        'cN'    => $row['win_close_jd'],
        'rO'    => $row['res_open'],
        'rC'    => $row['res_close']
    ];
}
$rep->close();
$conn->close();

// 13) If AJAX request, return only table rows
if ($isAjaxRequest) {
    if (empty($rows)) {
        echo '<tr><td colspan="11" style="text-align:center;color:#666;">No bids found for ' . htmlspecialchars($filter_date) . '</td></tr>';
    } else {
        foreach($rows as $r) {
            echo '<tr>';
            echo '<td>' . htmlspecialchars($r['gName']) . '</td>';
            echo '<td>' . htmlspecialchars($r['fmtO']) . '</td>';
            echo '<td>' . htmlspecialchars($r['fmtC']) . '</td>';
            echo '<td class="left">' . htmlspecialchars($r['uN']) . '</td>';
            echo '<td>' . htmlspecialchars($r['dig']) . '</td>';
            echo '<td>₹' . number_format($r['amt'], 2) . '</td>';
            echo '<td>' . htmlspecialchars($r['ss']) . '</td>';
            echo '<td>' . ($r['oN'] ? htmlspecialchars($r['oN']) : '–') . '</td>';
            echo '<td>' . ($r['cN'] ? htmlspecialchars($r['cN']) : '–') . '</td>';
            echo '<td>' . htmlspecialchars($r['rO']) . '</td>';
            echo '<td>' . htmlspecialchars($r['rC']) . '</td>';
            echo '</tr>';
        }
    }
    exit;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Jodi-Digit : Player Results</title>
    <style>
        body { font-family: Arial, sans-serif; background: #f5f6f7; margin: 0; padding: 2rem; }
        .container { max-width: 1200px; margin: 0 auto; background: #fff; padding: 1.5rem; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,.1); }
        table { width: 100%; border-collapse: collapse; margin-top: 1rem; }
        th, td { padding: 8px; border: 1px solid #eee; text-align: center; font-size: 0.9rem; }
        th { background: #007bff; color: #fff; font-weight: bold; }
        td.left { text-align: left; }
        .controls { margin-bottom: 1rem; text-align: center; display: flex; justify-content: center; align-items: center; gap: 1rem; flex-wrap: wrap; }
        .controls input, .controls select { padding: .5rem; font-size: 1rem; border: 1px solid #ccc; border-radius: 4px; }
        .controls label { font-weight: bold; }
        .status-indicator { 
            position: fixed; 
            top: 10px; 
            right: 10px; 
            background: #28a745; 
            color: white; 
            padding: 5px 10px; 
            border-radius: 4px; 
            font-size: 0.8rem;
            z-index: 1000;
        }
        .status-indicator.updating { background: #ffc107; color: #000; }
        .status-indicator.error { background: #dc3545; }
        .highlight-new { 
            background-color: #e8f5e8 !important; 
            animation: fadeHighlight 3s ease-out;
        }
        @keyframes fadeHighlight {
            0% { background-color: #90EE90 !important; }
            100% { background-color: transparent !important; }
        }
        @media (max-width: 768px) {
            .container { padding: 1rem; }
            th, td { padding: 6px; font-size: 0.8rem; }
            .controls { flex-direction: column; gap: 0.5rem; }
        }
        .debug-info { background: #f8f9fa; padding: 10px; margin: 10px 0; border-radius: 4px; font-family: monospace; font-size: 12px; }
    </style>
</head>
<body>
    <div class="status-indicator" id="status">Live</div>
    
    <div class="container">
        <h1>Jodi-Digit : Player Results</h1>
        <div class="controls">
            <form style="display: inline">
                <label>Date:</label>
                <input type="date" name="filter_date" value="<?=htmlspecialchars($filter_date)?>" onchange="this.form.submit()">
            </form>
            <form style="display: inline">
                <label>View Results:</label>
                <select onchange="location.href=this.value + '?filter_date=' + encodeURIComponent('<?=htmlspecialchars($filter_date)?>')">
                    <option value="resultsd.php">Single Digit Results</option>
                    <option value="result_dp.php">Double Panna</option>
                    <option value="result_tp.php">Triple Panna</option>
                    <option value="result_jd.php" selected>Jodi Digit</option>
                    <option value="result_sp.php">Single Panna</option>
                    <option value="resulths.php">Half Sangam</option>
                    <option value="result_fsp.php">Full Sangam</option>
                </select>
            </form>
        </div>

        <?php if (empty($rows)): ?>
            <div style="text-align: center; color: #666; padding: 1rem;">
                No bids found for <strong><?=htmlspecialchars($filter_date)?></strong>.
                <div class="debug-info">
                    Debug Info:<br>
                    - Date filter: <?=htmlspecialchars($filter_date)?><br>
                    - Games with rewards: <?=count($rewards)?><br>
                    <?php if (!empty($rewards)): ?>
                    - Game IDs: <?=implode(', ', array_keys($rewards))?><br>
                    <?php endif; ?>
                </div>
            </div>
        <?php else: ?>
            <!--<div class="debug-info">-->
            <!--    Found <?=count($rows)?> result entries for <?=htmlspecialchars($filter_date)?>-->
            <!--</div>-->
            <table>
                <thead>
                    <tr>
                        <th>Game</th><th>Open</th><th>Close</th>
                        <th>User</th><th>Number</th><th>Amt</th>
                        <th>Session</th><th>Win # (O)</th><th>Win # (C)</th>
                        <th>Result O</th><th>Result C</th>
                    </tr>
                </thead>
                <tbody id="bids-tbody">
                    <?php foreach($rows as $r): ?>
                        <tr>
                            <td><?= htmlspecialchars($r['gName']) ?></td>
                            <td><?= htmlspecialchars($r['fmtO']) ?></td>
                            <td><?= htmlspecialchars($r['fmtC']) ?></td>
                            <td class="left"><?= htmlspecialchars($r['uN']) ?></td>
                            <td><?= htmlspecialchars($r['dig']) ?></td>
                            <td>₹<?= number_format($r['amt'], 2) ?></td>
                            <td><?= htmlspecialchars($r['ss']) ?></td>
                            <td><?= $r['oN'] ? htmlspecialchars($r['oN']) : '–' ?></td>
                            <td><?= $r['cN'] ? htmlspecialchars($r['cN']) : '–' ?></td>
                            <td><?= htmlspecialchars($r['rO']) ?></td>
                            <td><?= htmlspecialchars($r['rC']) ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>

    <script>
        let lastRowCount = <?= count($rows) ?>;
        
        function updateResults() {
            const status = document.getElementById('status');
            const tbody = document.getElementById('bids-tbody');
            
            status.textContent = 'Updating...';
            status.className = 'status-indicator updating';
            
            fetch('result_jd.php?ajax=1&filter_date=' + encodeURIComponent('<?= htmlspecialchars($filter_date) ?>'))
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.text();
                })
                .then(data => {
                    if (data.trim()) {
                        const tempDiv = document.createElement('div');
                        tempDiv.innerHTML = '<table><tbody>' + data + '</tbody></table>';
                        const newRowCount = tempDiv.querySelectorAll('tr').length;
                        
                        // Only update if there are changes
                        if (tbody.innerHTML !== data) {
                            tbody.innerHTML = data;
                            
                            // Highlight if new rows were added
                            if (newRowCount > lastRowCount) {
                                const newRows = tbody.querySelectorAll('tr');
                                for (let i = 0; i < newRowCount - lastRowCount; i++) {
                                    if (newRows[i]) {
                                        newRows[i].classList.add('highlight-new');
                                    }
                                }
                            }
                            
                            lastRowCount = newRowCount;
                        }
                    }
                    
                    status.textContent = 'Live';
                    status.className = 'status-indicator';
                })
                .catch(error => {
                    console.error('Error updating results:', error);
                    status.textContent = 'Error';
                    status.className = 'status-indicator error';
                });
        }
        
        // Update every 5 seconds
        setInterval(updateResults, 5000);
        
        // Also update when page becomes visible (user switches back to tab)
        document.addEventListener('visibilitychange', function() {
            if (!document.hidden) {
                setTimeout(updateResults, 500); // Small delay to ensure page is fully visible
            }
        });
        
        // Remove highlight classes after animation
        document.addEventListener('animationend', function(e) {
            if (e.animationName === 'fadeHighlight') {
                e.target.classList.remove('highlight-new');
            }
        });
    </script>
</body>
</html>