<?php
/****************************************************************
 * result_hs.php — Half-Sangam results & wallet updater (live-reloading)
 ****************************************************************/

// hide notices/warnings, but still log errors
ini_set('display_errors', 1);
ini_set('log_errors',     1);
ini_set('error_log',      __DIR__.'/result_hs_error.log');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);

session_start();
date_default_timezone_set('Asia/Kolkata');

require_once __DIR__.'/db.php';
$conn->set_charset('utf8mb4');

// 1) Auth guard
if (empty($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// 2) Check if this is an AJAX request for table data only
$isAjaxRequest = isset($_GET['ajax']) && $_GET['ajax'] === '1';

// 3) Date filter
$filter_date = (isset($_GET['filter_date']) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $_GET['filter_date']))
             ? $_GET['filter_date']
             : date('Y-m-d');

// 4) Payout multiplier
$HS_MULT = 10000;

/*------------------------------------------------------------
 * 5) Clear & reseed hs_result
 *------------------------------------------------------------*/
$ridList = [];
if ($stm = $conn->prepare("SELECT id FROM hs_rewards WHERE reward_date = ?")) {
    $stm->bind_param('s', $filter_date);
    $stm->execute();
    $stm->bind_result($rid);
    while ($stm->fetch()) {
        $ridList[] = $rid;
    }
    $stm->close();
}
if (!empty($ridList)) {
    $in = implode(',', array_map('intval', $ridList));
    $conn->query("DELETE FROM hs_result WHERE reward_id IN ($in)");
}

/*------------------------------------------------------------
 * 6) Seed — your existing Pending/Win logic (EXACTLY as original)
 *------------------------------------------------------------*/
$seed = $conn->prepare("
  INSERT IGNORE INTO hs_result (
    reward_id, user_id, username,
    bid_od_hs, bid_cp_hs,
    bid_amount, bid_session,
    win_open_time_od_hs, win_open_time_cp_hs,
    win_close_time_od_hs, win_close_time_cp_hs,
    res_open_od, res_open_cp,
    res_close_od, res_close_cp
  )
  SELECT
    r.id, b.user_id, u.username,
    b.bid_open_hs, b.bid_close_hs,
    b.total_bid_amount, b.bid_session,
    r.winning_open_time_od_hs, r.winning_open_time_cp_hs,
    r.winning_close_time_od_hs, r.winning_close_time_cp_hs,

    /* Open-session, open-digit result */
    IF(
      r.winning_open_time_od_hs IS NULL, 'Pending',
      IF(
        LOWER(b.bid_session)='open'
        AND b.bid_open_hs = r.winning_open_time_od_hs,
        CONCAT('🏆 ₹', FORMAT(b.total_bid_amount * ?,2)),
        '❌ Better luck'
      )
    ),

    /* Open-session, close-digit result */
    IF(
      r.winning_open_time_cp_hs IS NULL, 'Pending',
      IF(
        LOWER(b.bid_session)='open'
        AND b.bid_close_hs = r.winning_open_time_cp_hs,
        CONCAT('🏆 ₹', FORMAT(b.total_bid_amount * ?,2)),
        '❌ Better luck'
      )
    ),

    /* Close-session, open-digit result */
    IF(
      r.winning_close_time_od_hs IS NULL, 'Pending',
      IF(
        LOWER(b.bid_session)='close'
        AND b.bid_open_hs = r.winning_close_time_od_hs,
        CONCAT('🏆 ₹', FORMAT(b.total_bid_amount * ?,2)),
        '❌ Better luck'
      )
    ),

    /* Close-session, close-digit result */
    IF(
      r.winning_close_time_cp_hs IS NULL, 'Pending',
      IF(
        LOWER(b.bid_session)='close'
        AND b.bid_close_hs = r.winning_close_time_cp_hs,
        CONCAT('🏆 ₹', FORMAT(b.total_bid_amount * ?,2)),
        '❌ Better luck'
      )
    )

  FROM hs_rewards   AS r
  JOIN user_bids_hs AS b
    ON b.game_id         = r.game_id
   AND DATE(b.created_at)= r.reward_date
  JOIN users_data    AS u
    ON u.user_id         = b.user_id
  WHERE r.reward_date    = ?
");

if ($seed) {
    $seed->bind_param(
      'dddds',
      $HS_MULT,  // for open-OD payout
      $HS_MULT,  // for open-CP payout
      $HS_MULT,  // for close-OD payout
      $HS_MULT,  // for close-CP payout
      $filter_date
    );
    $seed->execute();
    $seed->close();
}

// 7) Credit winners (unchanged)
foreach ([
  'res_open_od'  => 'paid_open_od',
  'res_open_cp'  => 'paid_open_cp',
  'res_close_od' => 'paid_close_od',
  'res_close_cp' => 'paid_close_cp',
] as $resCol => $paidCol) {
  $pay = $conn->prepare("
    UPDATE wallet w
    JOIN hs_result d
      ON d.user_id    = w.user_id
     AND d.{$paidCol} = 0
    SET
      w.balance    = w.balance + (d.bid_amount * ?),
      d.{$paidCol} = 1
    WHERE d.{$resCol} LIKE '🏆%'
  ");
  if ($pay) {
    $pay->bind_param('d', $HS_MULT);
    $pay->execute();
    $pay->close();
  }
}

// 8) Fetch & render — **directly from hs_result** so it always matches
$rows = [];
$sql = "
  SELECT
    hr.game_name                         AS game,
    DATE_FORMAT(hr.open_time,'%l:%i %p')  AS openFmt,
    DATE_FORMAT(hr.close_time,'%l:%i %p') AS closeFmt,
    r.username                          AS user,
    r.bid_session                       AS session,
    r.bid_od_hs                         AS bid_od,
    r.bid_cp_hs                         AS bid_cp,
    r.bid_amount                        AS amount,

    /* these four: */
    hr.winning_open_time_od_hs    AS win_od_open,
    hr.winning_open_time_cp_hs    AS win_cp_open,
    hr.winning_close_time_od_hs   AS win_od_close,
    hr.winning_close_time_cp_hs   AS win_cp_close,

    r.res_open_od                       AS res_od_open,
    r.res_open_cp                       AS res_cp_open,
    r.res_close_od                      AS res_od_close,
    r.res_close_cp                      AS res_cp_close
  FROM hs_result r
  JOIN hs_rewards hr
    ON r.reward_id   = hr.id
   AND hr.reward_date = ?
  ORDER BY r.id DESC
";

$stmt = $conn->prepare($sql);
if ($stmt) {
    $stmt->bind_param('s', $filter_date);
    $stmt->execute();
    $stmt->bind_result(
      $game, $openFmt, $closeFmt, $user,
      $session, $bid_od, $bid_cp, $amt,
      $winOdOpen, $winCpOpen, $winOdClose, $winCpClose,
      $resOO, $resOC, $resCO, $resCC
    );

    while ($stmt->fetch()) {
      $rows[] = [
      'game'         => $game,
      'openFmt'      => $openFmt,
      'closeFmt'     => $closeFmt,
      'user'         => $user,
      'session'      => ucfirst($session),
      'bid_od'       => $bid_od,
      'bid_cp'       => $bid_cp,
      'amount'       => $amt,
      'win_od_open'  => $winOdOpen,
      'win_cp_open'  => $winCpOpen,
      'win_od_close' => $winOdClose,
      'win_cp_close' => $winCpClose,
      'res_od_open'  => $resOO,
      'res_cp_open'  => $resOC,
      'res_od_close' => $resCO,
      'res_cp_close' => $resCC,
    ];
    }
    $stmt->close();
}
$conn->close();

// 9) If AJAX request, return only table rows
if ($isAjaxRequest) {
    if (empty($rows)) {
        echo '<tr><td colspan="16" style="text-align:center;color:#666;">No bids found for ' . htmlspecialchars($filter_date) . '</td></tr>';
    } else {
        foreach($rows as $r) {
            echo '<tr>';
            echo '<td>' . htmlspecialchars($r['game']) . '</td>';
            echo '<td>' . htmlspecialchars($r['openFmt']) . '</td>';
            echo '<td>' . htmlspecialchars($r['closeFmt']) . '</td>';
            echo '<td>' . htmlspecialchars($r['user']) . '</td>';
            echo '<td>' . htmlspecialchars($r['session']) . '</td>';
            echo '<td>' . ($r['bid_od'] !== null ? htmlspecialchars($r['bid_od']) : '–') . '</td>';
            echo '<td>' . ($r['bid_cp'] !== null ? htmlspecialchars($r['bid_cp']) : '–') . '</td>';
            echo '<td>₹' . number_format($r['amount'], 2) . '</td>';
            echo '<td>' . ($r['win_od_open'] !== null ? htmlspecialchars($r['win_od_open']) : '–') . '</td>';
            echo '<td>' . ($r['win_cp_open'] !== null ? htmlspecialchars($r['win_cp_open']) : '–') . '</td>';
            echo '<td>' . ($r['win_od_close'] !== null ? htmlspecialchars($r['win_od_close']) : '–') . '</td>';
            echo '<td>' . ($r['win_cp_close'] !== null ? htmlspecialchars($r['win_cp_close']) : '–') . '</td>';
            echo '<td>' . htmlspecialchars($r['res_od_open']) . '</td>';
            echo '<td>' . htmlspecialchars($r['res_cp_open']) . '</td>';
            echo '<td>' . htmlspecialchars($r['res_od_close']) . '</td>';
            echo '<td>' . htmlspecialchars($r['res_cp_close']) . '</td>';
            echo '</tr>';
        }
    }
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>Half-Sangam : Player Results</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    .table thead th {
      background-color: #007bff;
      color: #fff;
    }
    .status-indicator { 
        position: fixed; 
        top: 10px; 
        right: 10px; 
        background: #28a745; 
        color: white; 
        padding: 5px 10px; 
        border-radius: 4px; 
        font-size: 0.8rem;
        z-index: 1000;
    }
    .status-indicator.updating { background: #ffc107; color: #000; }
    .status-indicator.error { background: #dc3545; }
    .highlight-new { 
        background-color: #e8f5e8 !important; 
        animation: fadeHighlight 3s ease-out;
    }
    @keyframes fadeHighlight {
        0% { background-color: #90EE90 !important; }
        100% { background-color: transparent !important; }
    }
  </style>
</head>
<body class="bg-light">
  <div class="status-indicator" id="status">Live</div>
  
  <div class="container my-4">
    <div class="card shadow-sm">
      <div class="card-body">

        <div class="d-flex justify-content-between align-items-center mb-3">
          <h1 class="h4 mb-0">Half-Sangam : Player Results</h1>
          <div>
            <label for="filter_date" class="form-label mb-0 me-2">Date:</label>
            <input
              id="filter_date"
              type="date"
              class="form-control d-inline-block"
              style="width:auto;"
              value="<?=htmlspecialchars($filter_date)?>"
              onchange="location.search='?filter_date='+this.value"
            >
            <select class="form-control d-inline-block ms-2" style="width:auto;" onchange="location.href=this.value + '?filter_date=' + encodeURIComponent('<?=htmlspecialchars($filter_date)?>')">
                <option value="resultsd.php">Single Digit Results</option>
                <option value="result_dp.php">Double Panna</option>
                <option value="result_tp.php">Triple Panna</option>
                <option value="result_jd.php">Jodi Digit</option>
                <option value="result_sp.php">Single Panna</option>
                <option value="resulths.php" selected>Half Sangam</option>
                <option value="result_fsp.php">Full Sangam</option>
            </select>
          </div>
        </div>

        <?php if (empty($rows)): ?>
          <div class="alert alert-info">
            No bids found for <strong><?=htmlspecialchars($filter_date)?></strong>.
          </div>
        <?php else: ?>
          <div class="table-responsive">
            <table class="table table-bordered text-center align-middle mb-0">
              <thead>
                <tr>
                  <th>Game</th>
                  <th>Open</th>
                  <th>Close</th>
                  <th>User</th>
                  <th>Session</th>
                  <th>Bid OD</th>
                  <th>Bid CP</th>
                  <th>Amount</th>
                  <th>Win OD (O)</th><th>Win CP (O)</th>
                  <th>Win OD (C)</th><th>Win CP (C)</th>
                  <th>Res OD (O)</th>
                  <th>Res CP (O)</th>
                  <th>Res OD (C)</th>
                  <th>Res CP (C)</th>
                </tr>
              </thead>
              <tbody id="bids-tbody">
                <?php foreach ($rows as $r): ?>
                  <tr>
                    <td><?=htmlspecialchars($r['game'])?></td>
                    <td><?=htmlspecialchars($r['openFmt'])?></td>
                    <td><?=htmlspecialchars($r['closeFmt'])?></td>
                    <td><?=htmlspecialchars($r['user'])?></td>
                    <td><?=htmlspecialchars($r['session'])?></td>
                    <td><?= $r['bid_od'] !== null ? htmlspecialchars($r['bid_od']) : '–' ?></td>
                    <td><?= $r['bid_cp'] !== null ? htmlspecialchars($r['bid_cp']) : '–' ?></td>
                    <td>₹<?=number_format($r['amount'],2)?></td>
                    <td><?= $r['win_od_open'] !== null ? htmlspecialchars($r['win_od_open']) : '–' ?></td>
                    <td><?= $r['win_cp_open'] !== null ? htmlspecialchars($r['win_cp_open']) : '–' ?></td>
                    <td><?= $r['win_od_close'] !== null ? htmlspecialchars($r['win_od_close']) : '–' ?></td>
                    <td><?= $r['win_cp_close'] !== null ? htmlspecialchars($r['win_cp_close']) : '–' ?></td>
                    <td><?=htmlspecialchars($r['res_od_open'])?></td>
                    <td><?=htmlspecialchars($r['res_cp_open'])?></td>
                    <td><?=htmlspecialchars($r['res_od_close'])?></td>
                    <td><?=htmlspecialchars($r['res_cp_close'])?></td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>

      </div>
    </div>
  </div>
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
  
  <script>
    let lastRowCount = <?= count($rows) ?>;
    
    function updateResults() {
        const status = document.getElementById('status');
        const tbody = document.getElementById('bids-tbody');
        
        status.textContent = 'Updating...';
        status.className = 'status-indicator updating';
        
        fetch('result_hs.php?ajax=1&filter_date=' + encodeURIComponent('<?= htmlspecialchars($filter_date) ?>'))
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.text();
            })
            .then(data => {
                if (data.trim()) {
                    const tempDiv = document.createElement('div');
                    tempDiv.innerHTML = '<table><tbody>' + data + '</tbody></table>';
                    const newRowCount = tempDiv.querySelectorAll('tr').length;
                    
                    // Only update if there are changes
                    if (tbody.innerHTML !== data) {
                        tbody.innerHTML = data;
                        
                        // Highlight if new rows were added
                        if (newRowCount > lastRowCount) {
                            const newRows = tbody.querySelectorAll('tr');
                            for (let i = 0; i < newRowCount - lastRowCount; i++) {
                                if (newRows[i]) {
                                    newRows[i].classList.add('highlight-new');
                                }
                            }
                        }
                        
                        lastRowCount = newRowCount;
                    }
                }
                
                status.textContent = 'Live';
                status.className = 'status-indicator';
            })
            .catch(error => {
                console.error('Error updating results:', error);
                status.textContent = 'Error';
                status.className = 'status-indicator error';
            });
    }
    
    // Update every 5 seconds
    setInterval(updateResults, 5000);
    
    // Also update when page becomes visible (user switches back to tab)
    document.addEventListener('visibilitychange', function() {
        if (!document.hidden) {
            setTimeout(updateResults, 500);
        }
    });
    
    // Remove highlight classes after animation
    document.addEventListener('animationend', function(e) {
        if (e.animationName === 'fadeHighlight') {
            e.target.classList.remove('highlight-new');
        }
    });
  </script>

</body>
</html>