<?php
// resultsd.php — Single-Digit Player Results (with auto-refresh)

ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

session_start();
date_default_timezone_set('Asia/Kolkata');

// 1) Auth
if (empty($_SESSION['user_id'])) {
	header('Location: login.php');
	exit;
}

// 2) DB
require_once __DIR__ . '/db.php';
if ($conn->connect_error) {
	die("DB Connect Error: " . $conn->connect_error);
}
$conn->set_charset('utf8mb4');

// 3) Date filter
$filter_date = $_GET['filter_date'] ?? date('Y-m-d');

// 4) Auto-refresh settings
$refresh_interval = $_GET['refresh'] ?? 30; // Default 30 seconds
$auto_refresh_enabled = $_GET['auto_refresh'] ?? 'on'; // Default enabled

// 5) Function to parse bid details (kept from original)
function explode_ticket($raw, $stake) {
	// Handle empty or null values
	if (empty($raw)) {
		return [];
	}
	
	// Try JSON decode first
	$j = json_decode($raw, true);
	if (json_last_error() === JSON_ERROR_NONE && is_array($j)) {
		$out = [];
		foreach ($j as $d => $a) {
			$out[] = [(int)$d, (float)$a];
		}
		return $out;
	}
	
	// Handle square bracket format like [5] -> {"0":5}
	if (strlen($raw) > 2 && $raw[0] === '[' && substr($raw, -1) === ']' && substr_count($raw, ',') === 0) {
		$inner = trim(substr($raw, 1, -1));
		if (is_numeric($inner)) {
			return [[(int)$inner, $stake]];
		}
	}
	
	// Fallback: extract digits from string
	$out = [];
	foreach (preg_split('/\D+/', $raw) as $d) {
		if ($d !== '' && is_numeric($d)) {
			$out[] = [(int)$d, $stake];
		}
	}
	return $out;
}

// 6) Function to calculate result based on winning numbers and timing
function calculateResult($digit, $amount, $session, $winning_open, $winning_close, $open_time, $close_time) {
	$current_time = time();
	$resO = '–';
	$resC = '–';
	
	if ($session === 'open') {
		if (!is_null($winning_open)) {
			$open_timestamp = strtotime($open_time);
			if ($current_time >= $open_timestamp) {
				if ($digit == $winning_open) {
					$resO = "🏆 ₹" . number_format($amount * 10, 2);
				} else {
					$resO = "❌ Better luck";
				}
			} else {
				$resO = 'Pending';
			}
		} else {
			$resO = 'Pending';
		}
	} else {
		if (!is_null($winning_close)) {
			$close_timestamp = strtotime($close_time);
			if ($current_time >= $close_timestamp) {
				if ($digit == $winning_close) {
					$resC = "🏆 ₹" . number_format($amount * 10, 2);
				} else {
					$resC = "❌ Better luck";
				}
			} else {
				$resC = 'Pending';
			}
		} else {
			$resC = 'Pending';
		}
	}
	
	return [$resO, $resC];
}

try {
	// 7) Simple direct query like invested.php
	$rows = [];
	$sql = "
	  SELECT
		g.game_name,
		DATE_FORMAT(g.open_time, '%l:%i %p') AS fmt_open,
		DATE_FORMAT(g.close_time, '%l:%i %p') AS fmt_close,
		g.open_time,
		g.close_time,
		u.username,
		ub.bid_details,
		ub.total_bid_amount,
		ub.bid_session,
		r.winning_open_time_sd,
		r.winning_close_time_sd,
		ub.created_at
	  FROM user_bids ub
	  JOIN users_data u ON u.user_id = ub.user_id
	  JOIN game56 g ON g.id = ub.game_id
	  LEFT JOIN sd_rewards r ON r.game_id = g.id AND r.reward_date = ?
	  WHERE DATE(ub.created_at) = ?
	  ORDER BY ub.created_at DESC
	";

	$stmt = $conn->prepare($sql);
	if ($stmt) {
		$stmt->bind_param('ss', $filter_date, $filter_date);
		$stmt->execute();
		$result = $stmt->get_result();
		
		while ($row = $result->fetch_assoc()) {
			// Parse bid details to get individual digits
			$tickets = explode_ticket($row['bid_details'], (float)$row['total_bid_amount']);
			
			foreach ($tickets as $ticket) {
				list($digit, $amount) = $ticket;
				
				// Calculate results
				list($resO, $resC) = calculateResult(
					$digit, 
					$amount, 
					$row['bid_session'],
					$row['winning_open_time_sd'],
					$row['winning_close_time_sd'],
					$row['open_time'],
					$row['close_time']
				);
				
				$rows[] = [
					'gName' => $row['game_name'],
					'fmtO'  => $row['fmt_open'],
					'fmtC'  => $row['fmt_close'],
					'uN'    => $row['username'],
					'dig'   => $digit,
					'amt'   => $amount,
					'ss'    => ucfirst($row['bid_session']),
					'oN'    => $row['winning_open_time_sd'],
					'cN'    => $row['winning_close_time_sd'],
					'rO'    => $resO,
					'rC'    => $resC,
					'created_at' => $row['created_at']
				];
			}
		}
		$stmt->close();
	}

} catch (Exception $e) {
	error_log("Error in resultsd.php: " . $e->getMessage());
	echo <<<HTML
<!DOCTYPE html>
<html><head><meta charset="utf-8"><title>Error</title>
<style>body{font-family:Arial,sans-serif;text-align:center;padding:2rem;color:#dc3545;}</style>
</head><body>
  <h1>Error</h1>
  <p>Unable to load results. Please try again later.</p>
  <p><a href="rewards_sd.php">Go to Rewards Page</a></p>
</body></html>
HTML;
	exit;
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
	<meta charset="utf-8">
	<title>Single-Digit : Player Results</title>
	<?php if ($auto_refresh_enabled === 'on'): ?>
		<meta http-equiv="refresh" content="<?= intval($refresh_interval) ?>">
	<?php endif; ?>
	<style>
		body { 
			font-family: Arial, sans-serif; 
			background: #f5f6f7; 
			margin: 0; 
			padding: 2rem; 
		}
		.container { 
			max-width: 1200px; 
			margin: 0 auto; 
			background: #fff; 
			padding: 1.5rem; 
			border-radius: 8px; 
			box-shadow: 0 2px 8px rgba(0,0,0,.1); 
		}
		.header {
			display: flex;
			justify-content: space-between;
			align-items: center;
			margin-bottom: 1rem;
			flex-wrap: wrap;
		}
		.refresh-controls {
			display: flex;
			align-items: center;
			gap: 1rem;
			background: #f8f9fa;
			padding: 0.5rem 1rem;
			border-radius: 5px;
			border: 1px solid #dee2e6;
			flex-wrap: wrap;
		}
		.refresh-status {
			color: #28a745;
			font-weight: bold;
			font-size: 0.9rem;
		}
		.refresh-disabled {
			color: #dc3545;
		}
		table { 
			width: 100%; 
			border-collapse: collapse; 
			margin-top: 1rem; 
		}
		th, td { 
			padding: 8px; 
			border: 1px solid #eee; 
			text-align: center; 
			font-size: 0.9rem; 
		}
		th { 
			background: #007bff; 
			color: #fff; 
			font-weight: bold; 
		}
		td.left { 
			text-align: left; 
		}
		.controls { 
			margin-bottom: 1rem; 
			text-align: center; 
			display: flex; 
			justify-content: center; 
			align-items: center; 
			gap: 1rem; 
			flex-wrap: wrap; 
		}
		.controls input, .controls select, .controls button { 
			padding: .5rem; 
			font-size: 1rem; 
			border: 1px solid #ccc; 
			border-radius: 4px; 
		}
		.controls label { 
			font-weight: bold; 
		}
		.btn {
			background: #007bff;
			color: white;
			text-decoration: none;
			padding: 0.5rem 1rem;
			border-radius: 4px;
			border: none;
			cursor: pointer;
		}
		.btn:hover {
			background: #0056b3;
		}
		.btn-success { background: #28a745; }
		.btn-success:hover { background: #1e7e34; }
		.btn-danger { background: #dc3545; }
		.btn-danger:hover { background: #c82333; }
		.last-updated {
			font-size: 0.8rem;
			color: #666;
			text-align: center;
			margin-top: 1rem;
		}
		@media (max-width: 768px) {
			.container { 
				padding: 1rem; 
			}
			th, td { 
				padding: 6px; 
				font-size: 0.8rem; 
			}
			.controls, .header { 
				flex-direction: column; 
				gap: 0.5rem; 
			}
			.refresh-controls {
				flex-direction: column;
				gap: 0.5rem;
			}
		}
	</style>
</head>
<body>
	<div class="container">
		<div class="header">
			<h1>Single-Digit : Player Results</h1>
			<div class="refresh-controls">
				<span class="refresh-status <?= $auto_refresh_enabled === 'off' ? 'refresh-disabled' : '' ?>">
					<?php if ($auto_refresh_enabled === 'on'): ?>
						🔄 Auto-refresh: <?= $refresh_interval ?>s
					<?php else: ?>
						⏸️ Auto-refresh: OFF
					<?php endif; ?>
				</span>
				<?php if ($auto_refresh_enabled === 'on'): ?>
					<a href="?filter_date=<?= urlencode($filter_date) ?>&auto_refresh=off&refresh=<?= $refresh_interval ?>" class="btn btn-danger">Stop</a>
				<?php else: ?>
					<a href="?filter_date=<?= urlencode($filter_date) ?>&auto_refresh=on&refresh=<?= $refresh_interval ?>" class="btn btn-success">Start</a>
				<?php endif; ?>
				<button onclick="location.reload()" class="btn">Refresh Now</button>
			</div>
		</div>

		<div class="controls">
			<form style="display: inline">
				<label>Date:</label>
				<input type="date" name="filter_date" value="<?=htmlspecialchars($filter_date)?>" onchange="updateUrl()">
				<input type="hidden" name="auto_refresh" value="<?= htmlspecialchars($auto_refresh_enabled) ?>">
				<input type="hidden" name="refresh" value="<?= htmlspecialchars($refresh_interval) ?>">
			</form>
			
			<form style="display: inline">
				<label>Refresh Interval:</label>
				<select name="refresh_interval" onchange="updateRefreshInterval()">
					<option value="10" <?= $refresh_interval == 10 ? 'selected' : '' ?>>10 seconds</option>
					<option value="30" <?= $refresh_interval == 30 ? 'selected' : '' ?>>30 seconds</option>
					<option value="60" <?= $refresh_interval == 60 ? 'selected' : '' ?>>1 minute</option>
					<option value="120" <?= $refresh_interval == 120 ? 'selected' : '' ?>>2 minutes</option>
					<option value="300" <?= $refresh_interval == 300 ? 'selected' : '' ?>>5 minutes</option>
				</select>
				<input type="hidden" name="filter_date" value="<?= htmlspecialchars($filter_date) ?>">
				<input type="hidden" name="auto_refresh" value="<?= htmlspecialchars($auto_refresh_enabled) ?>">
			</form>
			
			<form style="display: inline">
				<label>View Results:</label>
				<select onchange="navigateToPage(this.value)">
					<option value="resultsd.php" selected>Single Digit Results</option>
					<option value="result_dp.php">Double Panna</option>
					<option value="result_tp.php">Triple Panna</option>
					<option value="result_jd.php">Jodi Digit</option>
					<option value="result_sp.php">Single Panna</option>
					<option value="resulths.php">Half Sangam</option>
					<option value="result_fsp.php">Full Sangam</option>
				</select>
			</form>
		</div>

		<?php if (empty($rows)): ?>
			<div style="text-align: center; color: #666; padding: 1rem;">
				No bids found for <strong><?=htmlspecialchars($filter_date)?></strong>.
			</div>
		<?php else: ?>
			<table>
				<thead>
					<tr>
						<th>Game</th><th>Open</th><th>Close</th>
						<th>User</th><th>Digit</th><th>Amt</th>
						<th>Session</th><th>Win # (O)</th><th>Win # (C)</th>
						<th>Result O</th><th>Result C</th>
					</tr>
				</thead>
				<tbody>
					<?php foreach($rows as $r): ?>
						<tr>
							<td><?= htmlspecialchars($r['gName']) ?></td>
							<td><?= htmlspecialchars($r['fmtO']) ?></td>
							<td><?= htmlspecialchars($r['fmtC']) ?></td>
							<td class="left"><?= htmlspecialchars($r['uN']) ?></td>
							<td><?= htmlspecialchars($r['dig']) ?></td>
							<td>₹<?= number_format($r['amt'], 2) ?></td>
							<td><?= htmlspecialchars($r['ss']) ?></td>
							<td><?= $r['oN'] ? htmlspecialchars($r['oN']) : '–' ?></td>
							<td><?= $r['cN'] ? htmlspecialchars($r['cN']) : '–' ?></td>
							<td><?= htmlspecialchars($r['rO']) ?></td>
							<td><?= htmlspecialchars($r['rC']) ?></td>
						</tr>
					<?php endforeach; ?>
				</tbody>
			</table>
		<?php endif; ?>
		
		<div class="last-updated">
			Last updated: <?= date('Y-m-d H:i:s') ?> (Asia/Kolkata)
		</div>
	</div>

	<script>
		function updateUrl() {
			const filterDate = document.querySelector('input[name="filter_date"]').value;
			const autoRefresh = document.querySelector('input[name="auto_refresh"]').value;
			const refresh = document.querySelector('input[name="refresh"]').value;
			
			window.location.href = `?filter_date=${encodeURIComponent(filterDate)}&auto_refresh=${autoRefresh}&refresh=${refresh}`;
		}
		
		function updateRefreshInterval() {
			const filterDate = '<?= htmlspecialchars($filter_date) ?>';
			const autoRefresh = '<?= htmlspecialchars($auto_refresh_enabled) ?>';
			const refreshInterval = document.querySelector('select[name="refresh_interval"]').value;
			
			window.location.href = `?filter_date=${encodeURIComponent(filterDate)}&auto_refresh=${autoRefresh}&refresh=${refreshInterval}`;
		}
		
		function navigateToPage(page) {
			const filterDate = '<?= htmlspecialchars($filter_date) ?>';
			const autoRefresh = '<?= htmlspecialchars($auto_refresh_enabled) ?>';
			const refresh = '<?= htmlspecialchars($refresh_interval) ?>';
			
			window.location.href = `${page}?filter_date=${encodeURIComponent(filterDate)}&auto_refresh=${autoRefresh}&refresh=${refresh}`;
		}

		// Countdown timer for next refresh (optional)
		<?php if ($auto_refresh_enabled === 'on'): ?>
		let countdown = <?= intval($refresh_interval) ?>;
		setInterval(function() {
			countdown--;
			if (countdown <= 0) {
				countdown = <?= intval($refresh_interval) ?>;
			}
		}, 1000);
		<?php endif; ?>
	</script>
</body>
</html>