<?php
/**********************************************************************
 *  rewards_sd.php  –  Single-digit reward generator / viewer
 *********************************************************************/

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();
date_default_timezone_set('Asia/Kolkata');

require 'db.php'; // provides $conn (mysqli)

/* ------------------------------------------------------------------ */
/* 1 – Auth guard                                                     */
/* ------------------------------------------------------------------ */
if (empty($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

/* ------------------------------------------------------------------ */
/* 2 – Date filter (default = today, or ?filter_date=YYYY-MM-DD)      */
/* ------------------------------------------------------------------ */
$filter_date = $_GET['filter_date'] ?? date('Y-m-d');
$today       = date('Y-m-d');
$current_ts  = time();

/* ------------------------------------------------------------------ */
/* 3 – Fetch all games whose open_time falls on that day              */
/* ------------------------------------------------------------------ */
$games = [];
$stmt = $conn->prepare("
    SELECT id, game_name, open_time, close_time
      FROM game56
     WHERE DATE(open_time) = ?
");
$stmt->bind_param('s', $filter_date);
$stmt->execute();
$stmt->bind_result($gid, $gname, $gopen, $gclose);
while ($stmt->fetch()) {
    $games[$gid] = [
        'id'                    => $gid,
        'game_name'             => $gname,
        'open_time'             => $gopen,
        'close_time'            => $gclose,
        'winning_open_time_sd'  => null,
        'winning_close_time_sd' => null,
    ];
}
$stmt->close();

/* ------------------------------------------------------------------ */
/* 4 – Ensure a rewards row exists (INSERT IGNORE)                   */
/* ------------------------------------------------------------------ */
$ins = $conn->prepare("
    INSERT IGNORE INTO sd_rewards
        (game_id, game_name, open_time, close_time, reward_date)
    VALUES (?, ?, ?, ?, ?)
");
foreach ($games as $g) {
    $ins->bind_param(
        'issss',
        $g['id'],
        $g['game_name'],
        $g['open_time'],
        $g['close_time'],
        $filter_date
    );
    $ins->execute();
}
$ins->close();

/* ------------------------------------------------------------------ */
/* 5 – Read back any already-generated digits                         */
/* ------------------------------------------------------------------ */
$rw = $conn->prepare("
    SELECT game_id, winning_open_time_sd, winning_close_time_sd
      FROM sd_rewards
     WHERE reward_date = ?
");
$rw->bind_param('s', $filter_date);
$rw->execute();
$rw->bind_result($rid, $winOpen, $winClose);
while ($rw->fetch()) {
    if (isset($games[$rid])) {
        $games[$rid]['winning_open_time_sd']  = $winOpen;
        $games[$rid]['winning_close_time_sd'] = $winClose;
    }
}
$rw->close();

/* ------------------------------------------------------------------ */
/* 6 – If “today,” generate digits once open/close times pass         */
/* ------------------------------------------------------------------ */
if ($filter_date === $today) {
    $upOpen  = $conn->prepare("
        UPDATE sd_rewards
           SET winning_open_time_sd = ?,
               created_at           = NOW()
         WHERE game_id = ? 
           AND reward_date = ?
    ");
    $upClose = $conn->prepare("
        UPDATE sd_rewards
           SET winning_close_time_sd = ?,
               created_at            = NOW()
         WHERE game_id = ?
           AND reward_date = ?
    ");

    foreach ($games as &$_g) {
        $open_ts  = strtotime($_g['open_time']);
        $close_ts = strtotime($_g['close_time']);

        // — Generate Open digit
        if ($current_ts >= $open_ts && $_g['winning_open_time_sd'] === null) {
            $d = random_int(0, 9);
            $upOpen->bind_param('iis', $d, $_g['id'], $filter_date);
            $upOpen->execute();
            $_g['winning_open_time_sd'] = $d;
            error_log("Open Winning Number for Game ID {$_g['id']}: $d");
        }

        // — Generate Close digit
        if ($current_ts >= $close_ts && $_g['winning_close_time_sd'] === null) {
            $d = random_int(0, 9);
            $upClose->bind_param('iis', $d, $_g['id'], $filter_date);
            $upClose->execute();
            $_g['winning_close_time_sd'] = $d;
            error_log("Close Winning Number for Game ID {$_g['id']}: $d");
        }
    }
    unset($_g);

    $upOpen->close();
    $upClose->close();
}

/* ------------------------------------------------------------------ */
/* 7 – Status pill logic                                              */
/* ------------------------------------------------------------------ */
function status_pill(int $now, int $open_ts, int $close_ts): string
{
    // >10 min before open
    if ($now < $open_ts - 600) {
        return '<span class="status yet-open">Game is open</span>';
    }
    // within last 10 min before open
    if ($now < $open_ts) {
        return '<span class="status bidding-closed">Open bidding closed</span>';
    }
    // from open until 10 min before close
    if ($now < $close_ts - 600) {
        return '<span class="status running">Running</span>';
    }
    // within 10 min before close or after
    return '<span class="status closed">Closed</span>';
}

/* ------------------------------------------------------------------ */
/* 8 – Render HTML                                                    */
/* ------------------------------------------------------------------ */
$current_display = date('d/m/Y g:i A');
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Single-Digit Rewards</title>
  <style>
    body { font-family:Arial,Helvetica,sans-serif; background:#f4f9f9; margin:20px; }
    h1 { text-align:center; }
    table { width:90%; margin:0 auto; border-collapse:collapse; }
    th, td { padding:12px; border:1px solid #ddd; text-align:center; }
    th { background:#007BFF; color:#fff; }
    tr:nth-child(even) { background:#f9f9f9; }
    .status { font-weight:bold; }
    .yet-open       { color:#555; }
    .bidding-closed { color:#e67e22; }
    .running        { color:#28a745; }
    .closed         { color:#dc3545; }
    .filter { text-align:center; margin-bottom:20px; color:#007BFF; }
  </style>
</head>
<body>

  <h1>Single-Digit Rewards</h1>

  <div class="filter">
    <form>
      <label>View Rewards:</label>
      <select onchange="location.href=this.value;">
        <option value="rewards_sd.php" selected>Single Digit Rewards</option>
        <option value="rewards_dp.php">Double Panna Rewards</option>
        <option value="rewards_tp.php">Triple Panna Rewards</option>
        <option value="rewards_jd.php">Jodi Digit Rewards</option>
        <option value="rewards_sp.php">Single Panna Rewards</option>
        <option value="rewards_hs.php">Half Sangam Rewards</option>
        <option value="rewards_fs.php">Full Sangam Rewards</option>
      </select>

      &nbsp;&nbsp;
      <label for="filter_date">Date:</label>
      <input
        type="date"
        id="filter_date"
        name="filter_date"
        value="<?= htmlspecialchars($filter_date) ?>"
        onchange="location.href='rewards_sd.php?filter_date='+this.value"
      />
    </form>
  </div>

  <div style="text-align:center; margin-bottom:20px;">
    Current IST: <strong><?= $current_display ?></strong>
  </div>

  <table>
    <thead>
      <tr>
        <th>Game</th>
        <th>Open</th>
        <th>Close</th>
        <th>Win # (Open)</th>
        <th>Win # (Close)</th>
        <th>Status</th>
      </tr>
    </thead>
    <tbody>
      <?php foreach ($games as $g): ?>
      <tr data-game-id="<?= $g['id'] ?>">
        <td><?= htmlspecialchars($g['game_name']) ?></td>
        <td><?= date('d/m/Y g:i A', strtotime($g['open_time'])) ?></td>
        <td><?= date('d/m/Y g:i A', strtotime($g['close_time'])) ?></td>
        <td class="open-digit"><?= $g['winning_open_time_sd']  ?? 'N/A' ?></td>
        <td class="close-digit"><?= $g['winning_close_time_sd'] ?? 'N/A' ?></td>
        <td>
          <?= status_pill(
                $current_ts,
                strtotime($g['open_time']),
                strtotime($g['close_time'])
              ) ?>
        </td>
      </tr>
      <?php endforeach; ?>
    </tbody>
  </table>

  <script>
    // Poll every 30 seconds for any newly-generated digits:
    function fetchLatestResults() {
      fetch('get_latest_sd_results.php?filter_date=<?= $filter_date ?>')
        .then(r => r.json())
        .then(data => {
          data.forEach(res => {
            const row = document.querySelector(`tr[data-game-id="${res.game_id}"]`);
            if (!row) return;
            row.querySelector('.open-digit').innerText  = res.open_digit  ?? 'N/A';
            row.querySelector('.close-digit').innerText = res.close_digit ?? 'N/A';
          });
        })
        .catch(console.error);
    }

    setInterval(fetchLatestResults, 30000);
  </script>

</body>
</html>
<?php
$conn->close();
?>
