<?php
// -----------------------------------------------------------------------------
// rewards2.php
// -----------------------------------------------------------------------------
// 1) Show all errors (so you never get a silent 503)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// 2) Start session & timezone
session_start();
date_default_timezone_set('Asia/Kolkata');

// 3) DB connection
include 'db.php'; // must define $conn = new mysqli(...);

// 4) Authentication
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

// 5) Date filter & timestamp
$filter_date = $_GET['filter_date'] ?? date('Y-m-d');
$now_ts      = time();

// 6) Seed dp_rewards with only double_panna games
$seed_sql = "
  INSERT IGNORE INTO dp_rewards
    (game_id, game_name, open_time, close_time, reward_date, created_at)
  SELECT
    id, game_name, open_time, close_time, ?, NOW()
  FROM game56
  WHERE sub_games = 'double_panna'
    AND DATE(open_time) = ?
";
$seed = $conn->prepare($seed_sql);
$seed->bind_param("ss", $filter_date, $filter_date);
$seed->execute();
$seed->close();

// 7) Helper: fetch all rows via bind_result()
function fetchGames($conn, $filter_date, $now_ts) {
    $sql = "
      SELECT
        r.id,
        r.game_name,
        r.open_time,
        r.close_time,
        r.winning_open_time_dp,
        r.winning_close_time_dp,
        u.username,
        b.bid_details     AS bid_number,
        b.total_bid_amount,
        IF(
          ? < UNIX_TIMESTAMP(r.close_time),
          'Open','Close'
        ) AS session
      FROM dp_rewards AS r
      LEFT JOIN user_bids_dp AS b
        ON b.game_id    = r.game_id
       AND b.bid_session = IF(
                             ? < UNIX_TIMESTAMP(r.close_time),
                             'Open','Close'
                           )
       AND DATE(b.created_at) = r.reward_date
      LEFT JOIN users_data AS u
        ON u.user_id = b.user_id
      WHERE r.reward_date = ?
      ORDER BY r.open_time
    ";
    $st = $conn->prepare($sql);
    $st->bind_param("iis", $now_ts, $now_ts, $filter_date);
    $st->execute();
    $st->bind_result(
      $id,
      $game_name,
      $open_time,
      $close_time,
      $win_open,
      $win_close,
      $username,
      $bid_number,
      $bid_amount,
      $session
    );
    $rows = [];
    while ($st->fetch()) {
      $rows[] = [
        'id'                    => $id,
        'game_name'             => $game_name,
        'open_time'             => $open_time,
        'close_time'            => $close_time,
        'winning_open_time_dp'  => $win_open,
        'winning_close_time_dp' => $win_close,
        'username'              => $username,
        'bid_number'            => $bid_number,
        'bid_amount'            => $bid_amount,
        'session'               => $session,
      ];
    }
    $st->close();
    return $rows;
}

// 8) Initial fetch
$games = fetchGames($conn, $filter_date, $now_ts);

// 9) If viewing today, fill any missing winners
if ($filter_date === date('Y-m-d')) {
  foreach ($games as $g) {
    $open_ts  = strtotime($g['open_time']);
    $close_ts = strtotime($g['close_time']);
    // OPEN result
    if ($now_ts >= $open_ts && $g['winning_open_time_dp'] === null) {
      $d = rand(0,9);
      $conn->query("UPDATE dp_rewards SET winning_open_time_dp=$d WHERE id={$g['id']}");
    }
    // CLOSE result
    if ($now_ts >= $close_ts && $g['winning_close_time_dp'] === null) {
      $d2 = rand(0,9);
      $conn->query("UPDATE dp_rewards SET winning_close_time_dp=$d2 WHERE id={$g['id']}");
    }
  }
  // re‑fetch with filled winners
  $games = fetchGames($conn, $filter_date, $now_ts);
}

// 10) Render HTML
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Double Panna Bid Results</title>
  <style>
    body { font-family:Arial,sans-serif; background:#f8f9fa; padding:2rem; }
    h2  { text-align:center; margin-bottom:0.2em; }
    p   { text-align:center; color:#007bff; margin-top:0; }
    table { width:100%; border-collapse:collapse; margin-top:1rem; }
    th, td { padding:8px; border:1px solid #ccc; text-align:center; }
    th { background:#007bff; color:#fff; }
    .win  { color:green; font-weight:bold; }
    .lose { color:red;   font-weight:bold; }
  </style>
</head>
<body>
  <h2>Double Panna Bid Results</h2>
  <p>Current IST: <?= date('d/m/Y g:i A') ?></p>

  <table>
    <tr>
      <th>Game</th>
      <th>Open</th>
      <th>Close</th>
      <th>User</th>
      <th>Bid #</th>
      <th>Amt</th>
      <th>Session</th>
      <th>Win #</th>
      <th>Result</th>
    </tr>
    <?php foreach ($games as $g):
      $winNum = ($g['session']==='Open')
                ? $g['winning_open_time_dp']
                : $g['winning_close_time_dp'];
      $isWin  = ($g['bid_number'] == $winNum);
      $amt    = $g['bid_amount'] ?? 0;
    ?>
    <tr>
      <td><?= htmlspecialchars($g['game_name']) ?></td>
      <td><?= date('g:i A', strtotime($g['open_time'])) ?></td>
      <td><?= date('g:i A', strtotime($g['close_time'])) ?></td>
      <td><?= htmlspecialchars($g['username'] ?? '–') ?></td>
      <td><?= $g['bid_number'] ?? '–' ?></td>
      <td>₹<?= number_format($amt,2) ?></td>
      <td><?= $g['session'] ?></td>
      <td><?= $winNum ?? 'N/A' ?></td>
      <td class="<?= $isWin ? 'win' : 'lose' ?>">
        <?php if ($isWin): ?>
          🏆 ₹<?= number_format($amt * 30, 2) ?>
        <?php else: ?>
          ❌
        <?php endif; ?>
      </td>
    </tr>
    <?php endforeach; ?>
  </table>
</body>
</html>
