<?php
session_start();
date_default_timezone_set('Asia/Kolkata');
include('db.php');

// 1) Ensure user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

// 2) Build last‑5‑days filter
$last5days = [];
for ($i = 0; $i < 5; $i++) {
    $dv = date('Y-m-d', strtotime("-$i days"));
    $dl = date('d/m/Y',   strtotime("-$i days"));
    $last5days[$dv] = $dl;
}
$filter_date = $_GET['filter_date'] ?? date('Y-m-d');

// 3) Fetch games for that date
$sql = "
  SELECT 
    id,
    game_name,
    open_time,
    close_time,
    winning_open_time
  FROM game56
  WHERE DATE(open_time) = ?
";
$stmt = $conn->prepare($sql);
$stmt->bind_param("s", $filter_date);
$stmt->execute();
$stmt->bind_result(
  $game_id,
  $game_name,
  $open_time,
  $close_time,
  $win_open_time
);
$games = [];
while ($stmt->fetch()) {
    $games[] = [
      'id'                => $game_id,
      'game_name'         => $game_name,
      'open_time'         => $open_time,
      'close_time'        => $close_time,
      'winning_open_time' => $win_open_time,
    ];
}
$stmt->close();

// 4) Reset past days’ winners
$today      = date('Y-m-d');
$current_ts = time();
if ($filter_date !== $today) {
    $conn->query("
      UPDATE game56
         SET winning_open_time = NULL
       WHERE DATE(open_time) = '{$filter_date}'
    ");
    foreach ($games as &$g) {
        $g['winning_open_time'] = null;
    }
    unset($g);
}

// 5) Prepare display timestamp
$current_display = date("d/m/Y g:i A");
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Game Open Bid Results</title>
  <style>
    body { font-family:Arial,sans-serif; background:#f4f4f9; margin:20px; color:#333 }
    h1 { text-align:center }
    .filter { text-align:center; margin-bottom:20px }
    table { width:90%; margin:0 auto; border-collapse:collapse }
    th, td { padding:12px; border:1px solid #ddd; text-align:center }
    th { background:#007BFF; color:#fff }
    tr:nth-child(even) { background:#f9f9f9 }
    .status { font-weight:bold; }
    .yet-open        { color:#555; }
    .bidding-closed  { color:#e67e22; }
    .running         { color:#28a745; }
    .closed          { color:#dc3545; }
  </style>
</head>
<body>
  <h1>Game Open Bid Results</h1>

  <div style="text-align:center; margin-bottom:20px; color:#007BFF;">
    Current IST: <?= $current_display ?>
  </div>

  <div class="filter">
    <form method="GET">
      <label>Select Date:</label>
      <select name="filter_date">
        <?php foreach ($last5days as $dv => $dl): ?>
          <option value="<?= $dv ?>"<?= $dv === $filter_date ? ' selected' : '' ?>>
            <?= $dl ?>
          </option>
        <?php endforeach; ?>
      </select>
      <button>Filter</button>
    </form>
  </div>

  <table>
    <thead>
      <tr>
        <th>Game Name</th>
        <th>Open Time</th>
        <th>Close Time</th>
        <th>Open Bid Result</th>
        <th>Status</th>
      </tr>
    </thead>
    <tbody>
      <?php foreach ($games as &$g):
        // Build full timestamps from filter_date + time
        $open_ts  = strtotime("{$filter_date} {$g['open_time']}");
        $close_ts = strtotime("{$filter_date} {$g['close_time']}");
        if ($close_ts < $open_ts) {
            // handle overnight close
            $close_ts += 86400;
        }

        // 1) Generate winning_open_time at or after open_time
        if (
            $filter_date === $today &&
            $current_ts   >= $open_ts &&
            $g['winning_open_time'] === null
        ) {
            $rand_open = rand(0,9);
            $nowf      = date('Y-m-d H:i:s');
            $upd = $conn->prepare("
              UPDATE game56
                 SET winning_open_time = ?, winning_time = ?
               WHERE id = ?
            ");
            $upd->bind_param("isi", $rand_open, $nowf, $g['id']);
            $upd->execute();
            $upd->close();
            $g['winning_open_time'] = $rand_open;
        }

        // 2) Determine status with 10‑min pre‑open window
        if ($current_ts < $open_ts - 600) {
            $status = '<span class="status yet-open">Yet to Open</span>';
        } elseif ($current_ts < $open_ts) {
            $status = '<span class="status bidding-closed">Open bidding is closed</span>';
        } elseif ($current_ts < $close_ts - 600) {
            $status = '<span class="status running">Game is Running</span>';
        } else {
            $status = '<span class="status closed">Game has Closed</span>';
        }

        // 3) Format for display
        $open_disp  = date("d/m/Y g:i A", $open_ts);
        $close_disp = date("d/m/Y g:i A", $close_ts);

        // 4) Show the digit directly (column now TINYINT)
        $open_win_disp = $g['winning_open_time'] !== null
                       ? $g['winning_open_time']
                       : 'N/A';
      ?>
      <tr>
        <td><?= htmlspecialchars($g['game_name']) ?></td>
        <td><?= $open_disp ?></td>
        <td><?= $close_disp ?></td>
        <td><?= $open_win_disp ?></td>
        <td><?= $status ?></td>
      </tr>
      <?php endforeach; unset($g); ?>
    </tbody>
  </table>
</body>
</html>
<?php
$conn->close();
?>
