<?php
// Show errors for debugging (remove in production)
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

session_start();
date_default_timezone_set('Asia/Kolkata');
include 'db.php'; // $conn = your mysqli connection

// ───────────────────────────────────────────────────────────────────
// Handle inline DP edits
if ($_SERVER['REQUEST_METHOD']==='POST' && ($_POST['action'] ?? '')==='update_dp_digit') {
    header('Content-Type: application/json');
    // already did session_start() above, so just guard
    if (empty($_SESSION['user_id'])) {
        echo json_encode(['success'=>false,'message'=>'Unauthorized']);
        exit;
    }

    $id    = intval($_POST['game_id']);
    $type  = $_POST['type'];    // "open" or "close"
    $digit = trim($_POST['digit']);

    // must be a 3-digit number
    if (!preg_match('/^[0-9]{3}$/', $digit)) {
        echo json_encode(['success'=>false,'message'=>'Invalid code']);
        exit;
    }

    // pick column
    $col = $type==='open'
         ? 'winning_open_time_dp'
         : 'winning_close_time_dp';

    $stmt = $conn->prepare("
      UPDATE dp_rewards
         SET {$col} = ?, created_at = NOW()
       WHERE id = ?
    ");
    $stmt->bind_param('si', $digit, $id);
    $ok = $stmt->execute();
    echo json_encode(['success'=>$ok]);
    exit;
}
// ───────────────────────────────────────────────────────────────────


// 1) Ensure user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

// 2) Build last‑5‑days filter
$last5days = [];
for ($i = 0; $i < 5; $i++) {
    $d = date('Y-m-d', strtotime("-{$i} days"));
    $l = date('d/m/Y',   strtotime("-{$i} days"));
    $last5days[$d] = $l;
}
$filter_date = $_GET['filter_date'] ?? date('Y-m-d');
$today       = date('Y-m-d');
$current_ts  = time();

// **Double Panna number sets**
$double_panna = [
  0 => ['118','226','244','299','334','488','550','668','677'],
  1 => ['100','119','155','227','335','344','399','588','669'],
  2 => ['110','200','228','255','336','499','660','688','778'],
  3 => ['166','229','300','337','355','445','599','779','788'],
  4 => ['112','220','266','338','400','446','455','699','770'],
  5 => ['113','122','177','339','366','447','500','799','889'],
  6 => ['600','114','277','330','448','466','556','880','899'],
  7 => ['115','133','188','223','377','449','557','566','700'],
  8 => ['116','224','233','288','440','477','558','800','990'],
  9 => ['117','144','199','225','388','559','577','667','900'],
];

// 3) Seed dp_rewards from game56 (one row per game)
$ins = $conn->prepare("
  INSERT IGNORE INTO dp_rewards
    (game_id, game_name, open_time, close_time, reward_date, created_at)
  SELECT id, game_name, open_time, close_time, ?, NOW()
    FROM game56
   WHERE DATE(open_time) = ?
");
$ins->bind_param("ss", $filter_date, $filter_date);
$ins->execute();
$ins->close();

// 4) Fetch dp_rewards for this date
$stmt = $conn->prepare("
  SELECT 
    id,
    game_name,
    open_time,
    close_time,
    winning_open_time_dp,
    winning_close_time_dp
  FROM dp_rewards
  WHERE reward_date = ?
  ORDER BY open_time
");
$stmt->bind_param("s", $filter_date);
$stmt->execute();
$stmt->bind_result(
  $rid,
  $gname,
  $open_dt,
  $close_dt,
  $win_open_dp,
  $win_close_dp
);
$games = [];
while ($stmt->fetch()) {
    $games[] = [
      'id'                    => $rid,
      'game_name'             => $gname,
      'open_time'             => $open_dt,
      'close_time'            => $close_dt,
      'winning_open_time_dp'  => $win_open_dp,
      'winning_close_time_dp' => $win_close_dp,
    ];
}
$stmt->close();

// 5) Clear past‑day winners in memory
foreach ($games as &$g) {
    if ($filter_date !== $today) {
        $g['winning_open_time_dp']  = null;
        $g['winning_close_time_dp'] = null;
    }
}
unset($g);

// 6) At open/close times, fill in missing winners
if ($filter_date === $today) {
    // prepare once, reuse
    $upOpen = $conn->prepare("
      UPDATE dp_rewards
         SET winning_open_time_dp = ?,
             created_at            = NOW()
       WHERE id = ?
    ");
    $upClose = $conn->prepare("
      UPDATE dp_rewards
         SET winning_close_time_dp = ?,
             created_at             = NOW()
       WHERE id = ?
    ");

    foreach ($games as &$g) {
      $open_ts  = strtotime($g['open_time']);
      $close_ts = strtotime($g['close_time']);

      // OPEN
      if ($current_ts >= $open_ts && $g['winning_open_time_dp'] === null) {
          $key   = array_rand($double_panna);
          $digit = $double_panna[$key][ array_rand($double_panna[$key]) ];

          $upOpen->bind_param("si", $digit, $g['id']);
          $upOpen->execute();
          $g['winning_open_time_dp'] = $digit;
      }

      // CLOSE
      if ($current_ts >= $close_ts && $g['winning_close_time_dp'] === null) {
          $key    = array_rand($double_panna);
          $digit2 = $double_panna[$key][ array_rand($double_panna[$key]) ];

          $upClose->bind_param("si", $digit2, $g['id']);
          $upClose->execute();
          $g['winning_close_time_dp'] = $digit2;
      }
    }
    unset($g);

    $upOpen->close();
    $upClose->close();
}


// 7) Prepare display timestamp
$current_display = date("d/m/Y g:i A");
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Double Panna Bid Results</title>
  <style>
    body { font-family:Arial,sans-serif; background:#f4f9f9; margin:20px; color:#333 }
    h1 { text-align:center }
    .filter { text-align:center; margin-bottom:20px }
    table { width:90%; margin:0 auto; border-collapse:collapse }
    th, td { padding:12px; border:1px solid #ddd; text-align:center }
    th { background:#007BFF; color:#fff }
    tr:nth-child(even) { background:#f9f9f9 }
    .status { font-weight:bold; }
    .yet-open       { color:#555; }
    .bidding-closed { color:#e67e22; }
    .running        { color:#28a745; }
    .closed         { color:#dc3545; }
  </style>
</head>
<body>
  <h1>Double Panna Bid Results</h1>

  <div style="text-align:center; margin-bottom:20px; color:#007BFF;">
    Current IST: <?= $current_display ?>
  </div>

  <div class="filter">
    <form>
      <label>View Rewards:</label>
      <select onchange="location.href=this.value;">
        <option value="rewards_sd.php">Single Digit Rewards</option>
        <option value="rewards_dp.php" selected>Double Panna Rewards</option>
        <option value="rewards_tp.php">Triple Panna Rewards</option>
        <option value="rewards_jd.php">Jodi Digit Rewards</option>
        <option value="rewards_sp.php">Single Panna Rewards</option>
      </select>
    </form>
  </div>

  <table>
    <thead>
      <tr>
        <th>Game Name</th>
        <th>Open Time</th>
        <th>Close Time</th>
        <th>Open Bid Result</th>
        <th>Close Bid Result</th>
        <th>Status</th>
      </tr>
    </thead>
    <tbody>
<?php foreach ($games as $g):
    // timestamps
    $open_ts   = strtotime($g['open_time']);
    $close_ts  = strtotime($g['close_time']);

    // status pill
    if ($current_ts < $open_ts - 600) {
        $status = '<span class="status yet-open">Game is open</span>';
    } elseif ($current_ts < $open_ts) {
        $status = '<span class="status bidding-closed">Open bidding closed</span>';
    } elseif ($current_ts < $close_ts - 600) {
        $status = '<span class="status running">Running</span>';
    } else {
        $status = '<span class="status closed">Closed</span>';
    }

    // display times
    $od = date("d/m/Y g:i A", $open_ts);
    $cd = date("d/m/Y g:i A", $close_ts);

    // has a DP result been stored?
    $openSet  = $g['winning_open_time_dp']  !== null;
    $closeSet = $g['winning_close_time_dp'] !== null;

    // should we reveal it now?
    $showOpen  = ($filter_date === $today && $current_ts >= $open_ts && $openSet);
    $showClose = ($filter_date === $today && $current_ts >= $close_ts && $closeSet);
?>
  <tr data-game-id="<?= $g['id'] ?>">
    <td><?= htmlspecialchars($g['game_name']) ?></td>
    <td><?= $od ?></td>
    <td><?= $cd ?></td>

    <!-- Open Bid Result -->
    <td 
      class="open-digit"
      contenteditable="true"
      data-type="open"
    >
      <?= $showOpen
           ? $g['winning_open_time_dp']
           : (
               $openSet
                 ? "({$g['winning_open_time_dp']}) N/A"
                 : 'N/A'
             )
      ?>
    </td>

    <!-- Close Bid Result -->
    <td 
      class="close-digit"
      contenteditable="true"
      data-type="close"
    >
      <?= $showClose
           ? $g['winning_close_time_dp']
           : (
               $closeSet
                 ? "({$g['winning_close_time_dp']}) N/A"
                 : 'N/A'
             )
      ?>
    </td>

    <td><?= $status ?></td>
  </tr>
<?php endforeach; ?>
</tbody>

  </table>
  <script>
  // reload page every 30 seconds
  setTimeout(() => window.location.reload(), 10000);
</script>
<script>
  document.querySelectorAll('td.open-digit, td.close-digit').forEach(cell => {
    cell.addEventListener('focus', () => {
      cell.dataset.orig = cell.innerText.trim();
    });
    cell.addEventListener('blur', () => {
      const orig  = cell.dataset.orig;
      const value = cell.innerText.trim();
      if (value === orig) return;

      // validate 3 digits
      if (!/^[0-9]{3}$/.test(value)) {
        alert("Please enter exactly 3 digits (000–999).");
        cell.innerText = orig;
        return;
      }

      const row    = cell.closest('tr');
      const gameId = row.dataset.gameId;
      const type   = cell.dataset.type; // "open" or "close"

      fetch('rewards_dp.php', {
        method: 'POST',
        headers: {'Content-Type':'application/x-www-form-urlencoded'},
        body: new URLSearchParams({
          action:      'update_dp_digit',
          game_id:     gameId,
          type:        type,
          digit:       value
        })
      })
      .then(r => r.json())
      .then(res => {
        if (!res.success) {
          alert(res.message || "Update failed");
          cell.innerText = orig;
        } else {
          // lock and show "(xxx) N/A" until time passes
          cell.innerText = `(${value}) N/A`;
          cell.contentEditable = "false";
        }
      })
      .catch(err => {
        console.error(err);
        cell.innerText = orig;
      });
    });
  });
</script>

</body>
</html>
<?php $conn->close(); ?>



