<?php
/****************************************************************
 * rewards_fs.php — Full-Sangam rewards & live picker/updater
 ****************************************************************/

ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

session_start();
date_default_timezone_set('Asia/Kolkata');
require_once __DIR__ . '/db.php';
$conn->set_charset('utf8mb4');

// ────────────────────────────────────────────────────────────────
// Inline-edit handler for Full Sangam
if ($_SERVER['REQUEST_METHOD']==='POST'
 && ($_POST['action'] ?? '')==='update_fs_digit')
{
    header('Content-Type: application/json');
    if (empty($_SESSION['user_id'])) {
        echo json_encode(['success'=>false,'message'=>'Unauthorized']);
        exit;
    }

    $id    = (int)$_POST['game_id'];
    $type  = $_POST['type'];    // open_op | open_cp | close_op | close_cp
    $digit = trim($_POST['digit']); // must be 2 or 3 digits

    // determine column & format check
    switch($type) {
      case 'open_op':  $col='winning_open_op_fs';  $re='/^[0-9]{2,3}$/'; break;
      case 'open_cp':  $col='winning_open_cp_fs';  $re='/^[0-9]{2,3}$/'; break;
      case 'close_op': $col='winning_close_op_fs'; $re='/^[0-9]{2,3}$/'; break;
      case 'close_cp': $col='winning_close_cp_fs'; $re='/^[0-9]{2,3}$/'; break;
      default:
        echo json_encode(['success'=>false,'message'=>'Bad type']); exit;
    }
    if (!preg_match($re, $digit)) {
        echo json_encode(['success'=>false,'message'=>'Invalid code']); exit;
    }

    $stmt = $conn->prepare("
      UPDATE fs_rewards
         SET {$col} = ?, created_at = NOW()
       WHERE id = ?
    ");
    $stmt->bind_param('si', $digit, $id);
    $ok = $stmt->execute();
    echo json_encode(['success'=>(bool)$ok]);
    exit;
}
// ────────────────────────────────────────────────────────────────

// 1) Auth guard
if (empty($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// 2) Date filter (YYYY-MM-DD)
$filter_date = (
    isset($_GET['filter_date'])
    && preg_match('/^\d{4}-\d{2}-\d{2}$/', $_GET['filter_date'])
)
? $_GET['filter_date']
: date('Y-m-d');

$today      = date('Y-m-d');
$current_ts = time();

// 3) Last-5-days for the dropdown
$last5 = [];
for ($i = 0; $i < 5; $i++) {
    $d = date('Y-m-d', strtotime("-{$i} days"));
    $last5[$d] = date('d/m/Y', strtotime($d));
}

// 4) Full-Sangam sets
// 4.a Jodi digits (00–99)
$jodi = [];
for ($i = 0; $i < 100; $i++) {
    $jodi[] = str_pad((string)$i, 2, '0', STR_PAD_LEFT);
}
// 4.b Single-Pana
$singlePana = [
  '0'=>['127','136','145','190','235','280','370','389','460','479','569','578'],
  '1'=>['128','137','146','236','245','290','380','470','489','560','579','678'],
  '2'=>['129','138','147','156','237','246','345','390','480','570','589','679'],
  '3'=>['120','139','148','157','238','247','256','346','490','580','670','689'],
  '4'=>['130','149','158','167','239','248','257','347','356','590','680','789'],
  '5'=>['140','159','168','230','249','258','267','348','357','456','690','780'],
  '6'=>['123','150','169','178','240','259','268','349','358','367','457','790'],
  '7'=>['124','160','178','179','250','269','340','359','368','458','467','890'],
  '8'=>['125','134','170','189','260','279','350','369','468','378','459','567'],
  '9'=>['126','135','180','234','270','289','360','379','450','469','478','568'],
];
// 4.c Double-Pana
$doublePana = [
  '0'=>['118','226','244','299','334','488','550','668','677'],
  '1'=>['100','119','155','227','335','344','399','588','669'],
  '2'=>['110','200','228','255','336','499','660','688','778'],
  '3'=>['166','229','300','337','355','445','599','779','788'],
  '4'=>['112','220','266','338','400','446','455','699','770'],
  '5'=>['113','122','177','339','366','447','500','799','889'],
  '6'=>['114','224','277','330','448','466','556','880','899'],
  '7'=>['115','133','188','223','377','449','557','566','700'],
  '8'=>['116','224','233','288','440','477','558','800','990'],
  '9'=>['117','144','199','225','388','559','577','667','900'],
];
// 4.d Triple-Pana
$triplePana = ['000','111','222','333','444','555','666','777','888','999'];

// assemble all
$fs_sets = [
  'Jodi Digit'  => $jodi,
  'Single Pana' => array_merge(...array_values($singlePana)),
  'Double Pana' => array_merge(...array_values($doublePana)),
  'Triple Pana' => $triplePana,
];

// 5) Seed fs_rewards **only** for today’s draws
$ins = $conn->prepare("
  INSERT IGNORE INTO fs_rewards
    (game_id,game_name,open_time,close_time,reward_date,created_at)
  SELECT
    id, game_name, open_time, close_time, ?, NOW()
  FROM game56
  WHERE DATE(open_time) = ?
");
$ins->bind_param('ss', $filter_date, $filter_date);
$ins->execute();
$ins->close();

// 6) Grab today’s table
$stmt = $conn->prepare("
  SELECT
    id, game_name, open_time, close_time,
    winning_open_op_fs,
    winning_open_cp_fs,
    winning_close_op_fs,
    winning_close_cp_fs
  FROM fs_rewards
  WHERE reward_date = ?
  ORDER BY open_time
");
$stmt->bind_param('s', $filter_date);
$stmt->execute();
$stmt->bind_result(
  $rid, $gname, $open_dt, $close_dt,
  $win_op_open, $win_cp_open,
  $win_op_close, $win_cp_close
);
$games = [];
while ($stmt->fetch()) {
  $games[] = [
    'id'            => $rid,
    'game_name'     => $gname,
    'open_time'     => $open_dt,
    'close_time'    => $close_dt,
    'win_op_open'   => $win_op_open,
    'win_cp_open'   => $win_cp_open,
    'win_op_close'  => $win_op_close,
    'win_cp_close'  => $win_cp_close,
  ];
}
$stmt->close();

// 7) Auto-fill only *after* each session’s start
if ($filter_date === $today) {
  // 1) prepare statements just once
  $upOpenOP  = $conn->prepare("
    UPDATE fs_rewards
       SET winning_open_op_fs = ?, created_at = NOW()
     WHERE id = ?");
  $upOpenCP  = $conn->prepare("
    UPDATE fs_rewards
       SET winning_open_cp_fs = ?, created_at = NOW()
     WHERE id = ?");
  $upCloseOP = $conn->prepare("
    UPDATE fs_rewards
       SET winning_close_op_fs = ?, created_at = NOW()
     WHERE id = ?");
  $upCloseCP = $conn->prepare("
    UPDATE fs_rewards
       SET winning_close_cp_fs = ?, created_at = NOW()
     WHERE id = ?");
  
  foreach ($games as &$g) {
    $oTs = strtotime($filter_date.' '.$g['open_time']);
    $cTs = strtotime($filter_date.' '.$g['close_time']);

    // OPEN-session draws
    if ($current_ts >= $oTs) {
      if ($g['win_op_open']  === null) {
        // pick a random category, then a random code from it
$cats = array_keys($fs_sets);
$cat  = $cats[array_rand($cats)];
$list = $fs_sets[$cat];
$d    = $list[array_rand($list)];

        $upOpenOP->bind_param('si',$d,$g['id']);
        $upOpenOP->execute();
        $g['win_op_open'] = $d;
      }
      if ($g['win_cp_open'] === null) {
        // pick a random category, then a random code from it
$cats = array_keys($fs_sets);
$cat  = $cats[array_rand($cats)];
$list = $fs_sets[$cat];
$d    = $list[array_rand($list)];

        $upOpenCP->bind_param('si',$d,$g['id']);
        $upOpenCP->execute();
        $g['win_cp_open'] = $d;
      }
    }

    // CLOSE-session draws
    if ($current_ts >= $cTs) {
      if ($g['win_op_close']  === null) {
        // pick a random category, then a random code from it
$cats = array_keys($fs_sets);
$cat  = $cats[array_rand($cats)];
$list = $fs_sets[$cat];
$d    = $list[array_rand($list)];

        $upCloseOP->bind_param('si',$d,$g['id']);
        $upCloseOP->execute();
        $g['win_op_close'] = $d;
      }
      if ($g['win_cp_close'] === null) {
        // pick a random category, then a random code from it
$cats = array_keys($fs_sets);
$cat  = $cats[array_rand($cats)];
$list = $fs_sets[$cat];
$d    = $list[array_rand($list)];

        $upCloseCP->bind_param('si',$d,$g['id']);
        $upCloseCP->execute();
        $g['win_cp_close'] = $d;
      }
    }
  }
  unset($g);

  // 2) close statements
  $upOpenOP->close();
  $upOpenCP->close();
  $upCloseOP->close();
  $upCloseCP->close();
}


// 8) Render
$current_display = date("d/m/Y g:i A");
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Full Sangam : Player Results</title>
  <style>
    body{background:#f5f6f7;font-family:Arial,sans-serif;margin:0;padding:2rem}
    .container{max-width:900px;margin:0 auto;background:#fff;
               padding:1.5rem;border-radius:8px;
               box-shadow:0 2px 8px rgba(0,0,0,.1)}
    h1{text-align:center;margin-bottom:1rem}
    .controls{margin-bottom:1rem;text-align:center}
    .controls select{padding:.5rem;font-size:1rem;margin-left:.5rem}
    table{width:100%;border-collapse:collapse;margin-top:1rem}
    th,td{padding:10px;border:1px solid #eee;text-align:center}
    th{background:#28a745;color:#fff}
  </style>
</head>
<body>
  <div class="container">
    <h1>Full Sangam : Player Results</h1>

    <div class="controls">
      <form>
        <label>Date:
          <select name="filter_date" onchange="this.form.submit()">
            <?php foreach($last5 as $d => $lbl): ?>
              <option value="<?=$d?>" <?= $d=== $filter_date ? 'selected' : '' ?>>
                <?=$lbl?>
              </option>
            <?php endforeach; ?>
          </select>
        </label>
      </form>
    </div>

    <div style="text-align:center;color:#28a745;margin-bottom:1rem">
      Current IST: <?=$current_display?>
    </div>

    <table>
      <thead>
        <tr>
          <th>Game</th>
          <th>Open</th>
          <th>Close</th>
          <th>Win (Open-OP)</th>
          <th>Win (Open-CP)</th>
          <th>Win (Close-OP)</th>
          <th>Win (Close-CP)</th>
          <th>Status</th>
        </tr>
      </thead>
     <tbody>
<?php foreach($games as $g):
    $oTs = strtotime("{$filter_date} {$g['open_time']}");
    $cTs = strtotime("{$filter_date} {$g['close_time']}");

    // flags for “we already have a value”
    $oo = $g['win_op_open']  !== null;
    $oc = $g['win_cp_open']  !== null;
    $co = $g['win_op_close'] !== null;
    $cc = $g['win_cp_close'] !== null;

    // flags for “time has passed & ready to show bare”
    $showOO = ($filter_date=== $today && $current_ts >= $oTs && $oo);
    $showOC = ($filter_date=== $today && $current_ts >= $oTs && $oc);
    $showCO = ($filter_date=== $today && $current_ts >= $cTs && $co);
    $showCC = ($filter_date=== $today && $current_ts >= $cTs && $cc);

    // status pill
    if      ($current_ts < $oTs - 600) $st='yet-open';
    elseif  ($current_ts < $oTs)       $st='bidding-closed';
    elseif  ($current_ts < $cTs - 600) $st='running';
    else                                $st='closed';
?>
  <tr data-game-id="<?=$g['id']?>">
    <td><?=htmlspecialchars($g['game_name'])?></td>
    <td><?=date('g:i A',$oTs)?></td>
    <td><?=date('g:i A',$cTs)?></td>

    <!-- Open-OP -->
    <td class="open-op" contenteditable="true" data-type="open_op">
      <?= $showOO
            ? $g['win_op_open']
            : ($oo ? "({$g['win_op_open']}) N/A" : 'N/A')
      ?>
    </td>

    <!-- Open-CP -->
    <td class="open-cp" contenteditable="true" data-type="open_cp">
      <?= $showOC
            ? $g['win_cp_open']
            : ($oc ? "({$g['win_cp_open']}) N/A" : 'N/A')
      ?>
    </td>

    <!-- Close-OP -->
    <td class="close-op" contenteditable="true" data-type="close_op">
      <?= $showCO
            ? $g['win_op_close']
            : ($co ? "({$g['win_op_close']}) N/A" : 'N/A')
      ?>
    </td>

    <!-- Close-CP -->
    <td class="close-cp" contenteditable="true" data-type="close_cp">
      <?= $showCC
            ? $g['win_cp_close']
            : ($cc ? "({$g['win_cp_close']}) N/A" : 'N/A')
      ?>
    </td>

    <td class="<?=$st?>"><?=ucwords(str_replace('-',' ',$st))?></td>
  </tr>
<?php endforeach; ?>
</tbody>

    </table>
  </div>
    <script>
  // reload page every 30 seconds
  setTimeout(() => window.location.reload(), 10000);
</script>
<script>
  ['open-op','open-cp','close-op','close-cp'].forEach(cls => {
    document.querySelectorAll(`td.${cls}`).forEach(cell => {
      cell.addEventListener('focus', () => {
        cell.dataset.orig = cell.innerText.trim();
      });
      cell.addEventListener('blur', () => {
        const orig = cell.dataset.orig;
        const val  = cell.innerText.trim();
        if (val === orig) return;

        // must be 2–3 digits
        if (!/^[0-9]{2,3}$/.test(val)) {
          alert("Enter 2–3 digits only.");
          cell.innerText = orig;
          return;
        }

        const row    = cell.closest('tr');
        const gameId = row.dataset.gameId;
        const type   = cell.dataset.type; // open_op, open_cp, close_op, close_cp

        fetch('rewards_fs.php', {
          method: 'POST',
          headers: { 'Content-Type':'application/x-www-form-urlencoded' },
          body: new URLSearchParams({
            action:    'update_fs_digit',
            game_id:   gameId,
            type:      type,
            digit:     val
          })
        })
        .then(r => r.json())
        .then(res => {
          if (!res.success) {
            alert(res.message || "Save failed");
            cell.innerText = orig;
          } else {
            // show (xx) N/A until time passes, then bare xx
            cell.innerText = `(${val}) N/A`;
            cell.contentEditable = "false";
          }
        })
        .catch(() => {
          cell.innerText = orig;
        });
      });
    });
  });
</script>


</body>
</html>
<?php $conn->close(); ?>
