<?php
/****************************************************************
 * rewards_hs.php — Half-Sangam rewards & live picker/updater
 ****************************************************************/

ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

session_start();
date_default_timezone_set('Asia/Kolkata');
require_once __DIR__.'/db.php';  // $conn = new mysqli(...)
$conn->set_charset('utf8mb4');

// ───────────────────────────────────────────────────────────────────
// Inline-edit AJAX handler for Half-Sangam
if ($_SERVER['REQUEST_METHOD'] === 'POST'
    && ($_POST['action'] ?? '') === 'update_hs_digit')
{
    header('Content-Type: application/json');
    if (empty($_SESSION['user_id'])) {
        echo json_encode(['success'=>false,'message'=>'Unauthorized']);
        exit;
    }

    $id    = intval($_POST['game_id']);
    $type  = $_POST['type'];       // open_od | open_cp | close_od | close_cp
    $digit = trim($_POST['digit']); // must be exactly 2 digits

    if (!preg_match('/^[0-9]{2}$/', $digit)) {
        echo json_encode(['success'=>false,'message'=>'Invalid code']);
        exit;
    }

    switch ($type) {
      case 'open_od':  $col = 'winning_open_time_od_hs';  break;
      case 'open_cp':  $col = 'winning_open_time_cp_hs';  break;
      case 'close_od': $col = 'winning_close_time_od_hs'; break;
      case 'close_cp': $col = 'winning_close_time_cp_hs'; break;
      default:
        echo json_encode(['success'=>false,'message'=>'Bad type']);
        exit;
    }

    $stmt = $conn->prepare("
      UPDATE hs_rewards
         SET {$col} = ?, created_at = NOW()
       WHERE id = ?
    ");
    $stmt->bind_param('si', $digit, $id);
    $ok = $stmt->execute();
    echo json_encode(['success'=> (bool)$ok]);
    exit;
}
// ───────────────────────────────────────────────────────────────────

// 1) Auth guard
if (empty($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// 2) Date filter (YYYY-MM-DD)
$filter_date = isset($_GET['filter_date']) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $_GET['filter_date'])
             ? $_GET['filter_date']
             : date('Y-m-d');
$today      = date('Y-m-d');
$current_ts = time();

// 3) Last-5-days for dropdown
$last5 = [];
for ($i=0; $i<5; $i++) {
    $d = date('Y-m-d', strtotime("-{$i} days"));
    $last5[$d] = date('d/m/Y', strtotime($d));
}

// 4) Half-Sangam sets
$hs_sets = [
  '0'=> ['00','01','02','03','04','05','06','07','08','09'],
  '1'=> ['11','10','12','13','14','15','16','17','18','19'],
  '2'=> ['22','20','21','23','24','25','26','27','28','29'],
  '3'=> ['33','30','31','32','34','35','36','37','38','39'],
  '4'=> ['44','40','41','42','43','45','46','47','48','49'],
  '5'=> ['55','50','51','52','53','54','56','57','58','59'],
  '6'=> ['66','60','61','62','63','64','65','67','68','69'],
  '7'=> ['77','70','71','72','73','74','75','76','78','79'],
  '8'=> ['88','80','81','82','83','84','85','86','87','89'],
  '9'=> ['99','90','91','92','93','94','95','96','97','98'],
];

// 5) Seed hs_rewards from game56
$ins = $conn->prepare("
  INSERT IGNORE INTO hs_rewards
    (game_id,game_name,open_time,close_time,reward_date,created_at)
  SELECT id,game_name,open_time,close_time,?,NOW()
    FROM game56
   WHERE DATE(open_time)=?
");
$ins->bind_param('ss',$filter_date,$filter_date);
$ins->execute();
$ins->close();

// 6) Fetch today’s hs_rewards (with new columns)
$stmt = $conn->prepare("
  SELECT id,game_name,open_time,close_time,
         winning_open_time_od_hs,
         winning_close_time_od_hs,
         winning_open_time_cp_hs,
         winning_close_time_cp_hs
    FROM hs_rewards
   WHERE reward_date=?
   ORDER BY open_time
");
$stmt->bind_param('s',$filter_date);
$stmt->execute();
$stmt->bind_result(
  $rid,$gname,$open_dt,$close_dt,
  $win_od_open,$win_od_close,
  $win_cp_open,$win_cp_close
);
$games = [];
while($stmt->fetch()){
  $games[] = [
    'id'             => $rid,
    'game_name'      => $gname,
    'open_time'      => $open_dt,
    'close_time'     => $close_dt,
    'win_od_open'    => $win_od_open,
    'win_od_close'   => $win_od_close,
    'win_cp_open'    => $win_cp_open,
    'win_cp_close'   => $win_cp_close,
  ];
}
$stmt->close();

// 7) Auto-fill missing winners (today only)
if ($filter_date === $today) {
    // prepare once
    $upOpenOD  = $conn->prepare("
      UPDATE hs_rewards
         SET winning_open_time_od_hs   = ?,
             created_at                = NOW()
       WHERE id = ?");
    $upOpenCP  = $conn->prepare("
      UPDATE hs_rewards
         SET winning_open_time_cp_hs   = ?,
             created_at                = NOW()
       WHERE id = ?");
    $upCloseOD = $conn->prepare("
      UPDATE hs_rewards
         SET winning_close_time_od_hs  = ?,
             created_at                = NOW()
       WHERE id = ?");
    $upCloseCP = $conn->prepare("
      UPDATE hs_rewards
         SET winning_close_time_cp_hs  = ?,
             created_at                = NOW()
       WHERE id = ?");

    foreach ($games as &$g) {
      $oTs = strtotime($g['open_time']);
      $cTs = strtotime($g['close_time']);

      // OPEN window
      if ($current_ts >= $oTs) {
        if ($g['win_od_open']  === null) {
          $d = $hs_sets[array_rand($hs_sets)][array_rand($hs_sets[array_rand($hs_sets)])];
          $upOpenOD->bind_param('si', $d, $g['id']);
          $upOpenOD->execute();
          $g['win_od_open'] = $d;
        }
        if ($g['win_cp_open']  === null) {
          $d2 = $hs_sets[array_rand($hs_sets)][array_rand($hs_sets[array_rand($hs_sets)])];
          $upOpenCP->bind_param('si', $d2, $g['id']);
          $upOpenCP->execute();
          $g['win_cp_open'] = $d2;
        }
      }

      // CLOSE window
      if ($current_ts >= $cTs) {
        if ($g['win_od_close'] === null) {
          $d3 = $hs_sets[array_rand($hs_sets)][array_rand($hs_sets[array_rand($hs_sets)])];
          $upCloseOD->bind_param('si', $d3, $g['id']);
          $upCloseOD->execute();
          $g['win_od_close'] = $d3;
        }
        if ($g['win_cp_close'] === null) {
          $d4 = $hs_sets[array_rand($hs_sets)][array_rand($hs_sets[array_rand($hs_sets)])];
          $upCloseCP->bind_param('si', $d4, $g['id']);
          $upCloseCP->execute();
          $g['win_cp_close'] = $d4;
        }
      }
    }
    unset($g);

    // close your statements
    $upOpenOD->close();
    $upOpenCP->close();
    $upCloseOD->close();
    $upCloseCP->close();
}


// 8) Ready to render
$current_display = date("d/m/Y g:i A");
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Half Sangam : Player Results</title>
  <style>
    body{background:#f5f6f7;font-family:Arial,sans-serif;margin:0;padding:2rem}
    .container{max-width:900px;margin:0 auto;background:#fff;
               padding:1.5rem;border-radius:8px;
               box-shadow:0 2px 8px rgba(0,0,0,.1)}
    h1{text-align:center;margin-bottom:1rem}
    .controls{margin-bottom:1rem;text-align:center}
    .controls select{padding:.5rem;font-size:1rem;margin-left:.5rem}
    table{width:100%;border-collapse:collapse;margin-top:1rem}
    th,td{padding:10px;border:1px solid #eee;text-align:center}
    th{background:#007bff;color:#fff}
    .status{font-weight:bold;}
    .yet-open{color:#555;}
    .bidding-closed{color:#e67e22;}
    .running{color:#28a745;}
    .closed{color:#dc3545;}
  </style>
</head>
<body>
  <div class="container">
    <h1>Half Sangam : Player Results</h1>

    <div class="controls">
      <form>
        <label>Date:
          <select name="filter_date" onchange="this.form.submit()">
            <?php foreach($last5 as $d=>$lbl): ?>
              <option value="<?=$d?>" <?=$d===$filter_date?'selected':''?>>
                <?=$lbl?>
              </option>
            <?php endforeach?>
          </select>
        </label>
      </form>
    </div>

    <div style="text-align:center;color:#007bff;margin-bottom:1rem">
      Current IST: <?=$current_display?>
    </div>

    <table>
      <thead>
        <tr>
          <th>Game</th>
          <th>Open</th>
          <th>Close</th>
          <th>Win (Open OD)</th>
          <th>Win (Open CP)</th>
          <th>Win (Close OD)</th>
          <th>Win (Close CP)</th>
          <th>Status</th>
        </tr>
      </thead>
      <tbody>
<?php foreach($games as $g):
    $oTs = strtotime($g['open_time']);
    $cTs = strtotime($g['close_time']);

    // did we already have a value?
    $openOD   = $g['win_od_open']   !== null;
    $openCP   = $g['win_cp_open']   !== null;
    $closeOD  = $g['win_od_close']  !== null;
    $closeCP  = $g['win_cp_close']  !== null;

    // should we show it now?
    $showODOpen  = ($filter_date=== $today && $current_ts >= $oTs && $openOD);
    $showCPOpen  = ($filter_date=== $today && $current_ts >= $oTs && $openCP);
    $showODClose = ($filter_date=== $today && $current_ts >= $cTs && $closeOD);
    $showCPClose = ($filter_date=== $today && $current_ts >= $cTs && $closeCP);

    // status pill
    if      ($current_ts < $oTs - 600) $st = 'yet-open';
    elseif  ($current_ts < $oTs)       $st = 'bidding-closed';
    elseif  ($current_ts < $cTs-600)   $st = 'running';
    else                                $st = 'closed';
?>
  <tr data-game-id="<?= $g['id'] ?>">
    <td><?=htmlspecialchars($g['game_name'])?></td>
    <td><?=date('d/m/Y g:i A',$oTs)?></td>
    <td><?=date('d/m/Y g:i A',$cTs)?></td>

    <!-- Win (Open OD) -->
    <td class="open-od-digit" contenteditable="true" data-type="open_od">
      <?= $showODOpen
           ? $g['win_od_open']
           : ($openOD 
               ? "({$g['win_od_open']}) N/A"
               : 'N/A'
             )
      ?>
    </td>

    <!-- Win (Open CP) -->
    <td class="open-cp-digit" contenteditable="true" data-type="open_cp">
      <?= $showCPOpen
           ? $g['win_cp_open']
           : ($openCP 
               ? "({$g['win_cp_open']}) N/A"
               : 'N/A'
             )
      ?>
    </td>

    <!-- Win (Close OD) -->
    <td class="close-od-digit" contenteditable="true" data-type="close_od">
      <?= $showODClose
           ? $g['win_od_close']
           : ($closeOD 
               ? "({$g['win_od_close']}) N/A"
               : 'N/A'
             )
      ?>
    </td>

    <!-- Win (Close CP) -->
    <td class="close-cp-digit" contenteditable="true" data-type="close_cp">
      <?= $showCPClose
           ? $g['win_cp_close']
           : ($closeCP 
               ? "({$g['win_cp_close']}) N/A"
               : 'N/A'
             )
      ?>
    </td>

    <td class="<?= $st ?>">
      <?= ucwords(str_replace('-', ' ', $st)) ?>
    </td>
  </tr>
<?php endforeach; ?>
</tbody>

    </table>
  </div>
    <script>
  // reload page every 30 seconds
  setTimeout(() => window.location.reload(), 10000);
</script>
<script>
  document.querySelectorAll(
    'td.open-od-digit, td.open-cp-digit, td.close-od-digit, td.close-cp-digit'
  ).forEach(cell => {
    cell.addEventListener('focus', () => {
      cell.dataset.orig = cell.innerText.trim();
    });
    cell.addEventListener('blur', () => {
      const orig  = cell.dataset.orig;
      const val   = cell.innerText.trim();
      if (val === orig) return;                       // no change

      // must be exactly two digits
      if (!/^[0-9]{2}$/.test(val)) {
        alert("Please enter exactly 2 digits (00–99).");
        cell.innerText = orig;
        return;
      }

      const row    = cell.closest('tr');
      const gameId = row.dataset.gameId;
      const type   = cell.dataset.type;  // open_od, open_cp, close_od, close_cp

      fetch('rewards_hs.php', {
        method: 'POST',
        headers:{'Content-Type':'application/x-www-form-urlencoded'},
        body: new URLSearchParams({
          action:    'update_hs_digit',
          game_id:   gameId,
          type:      type,
          digit:     val
        })
      })
      .then(r => r.json())
      .then(res => {
        if (!res.success) {
          alert(res.message || "Update failed");
          cell.innerText = orig;
        } else {
          cell.innerText = `(${val}) N/A`;
          cell.contentEditable = "false";
        }
      })
      .catch(err => {
        console.error(err);
        cell.innerText = orig;
      });
    });
  });
</script>

</body>
</html>
<?php $conn->close(); ?>
