<?php
// Show errors for debugging (remove in production)
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

session_start();
date_default_timezone_set('Asia/Kolkata');
include 'db.php'; // $conn = your mysqli connection

// ───────────────────────────────────────────────────────────────────
// Inline‐edit AJAX handler for Jodi digits
if ($_SERVER['REQUEST_METHOD'] === 'POST'
    && ($_POST['action'] ?? '') === 'update_jd_digit')
{
    header('Content-Type: application/json');
    if (empty($_SESSION['user_id'])) {
        echo json_encode(['success'=>false,'message'=>'Unauthorized']);
        exit;
    }

    $id    = intval($_POST['game_id']);
    $type  = $_POST['type'];       // "open" or "close"
    $digit = trim($_POST['digit']); // must be 2 digits

    if (!preg_match('/^[0-9]{2}$/', $digit)) {
        echo json_encode(['success'=>false,'message'=>'Invalid code']);
        exit;
    }

    $col = $type === 'open'
         ? 'winning_open_time_jd'
         : 'winning_close_time_jd';

    $stmt = $conn->prepare("
      UPDATE jd_rewards
         SET {$col} = ?, created_at = NOW()
       WHERE id = ?
    ");
    $stmt->bind_param('si', $digit, $id);
    $ok = $stmt->execute();
    echo json_encode(['success' => (bool)$ok]);
    exit;
}
// ───────────────────────────────────────────────────────────────────

// if you don’t already have it, add this helper:
function status_pill(int $now, int $open_ts, int $close_ts): string
{
    if ($now < $open_ts - 600)       return '<span class="status yet-open">Game is open</span>';
    if ($now < $open_ts)             return '<span class="status bidding-closed">Open bidding closed</span>';
    if ($now < $close_ts - 600)      return '<span class="status running">Running</span>';
                                     return '<span class="status closed">Closed</span>';
}


// 1) Ensure user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

// 2) Build last‑5‑days filter
$last5days = [];
for ($i = 0; $i < 5; $i++) {
    $d = date('Y-m-d', strtotime("-{$i} days"));
    $l = date('d/m/Y',   strtotime("-{$i} days"));
    $last5days[$d] = $l;
}
$filter_date = $_GET['filter_date'] ?? date('Y-m-d');
$today       = date('Y-m-d');
$current_ts  = time();

// build the 00–99 jodi list
$jd_nums = [];
for ($i = 0; $i < 100; $i++) {
    $jd_nums[] = str_pad($i, 2, '0', STR_PAD_LEFT);
}

// 3) Seed jd_rewards from game56 (one row per game)
$ins = $conn->prepare("
  INSERT IGNORE INTO jd_rewards
    (game_id, game_name, open_time, close_time, reward_date, created_at)
  SELECT id, game_name, open_time, close_time, ?, NOW()
    FROM game56
   WHERE DATE(open_time) = ?
");
$ins->bind_param("ss", $filter_date, $filter_date);
$ins->execute();
$ins->close();

// 4) Fetch jd_rewards for this date
$stmt = $conn->prepare("
  SELECT 
    id,
    game_name,
    open_time,
    close_time,
    winning_open_time_jd,
    winning_close_time_jd
  FROM jd_rewards
  WHERE reward_date = ?
  ORDER BY open_time
");
$stmt->bind_param("s", $filter_date);
$stmt->execute();
$stmt->bind_result(
  $rid,
  $gname,
  $open_dt,
  $close_dt,
  $win_open_jd,
  $win_close_jd
);
$games = [];
while ($stmt->fetch()) {
    $games[] = [
      'id'                     => $rid,
      'game_name'              => $gname,
      'open_time'              => $open_dt,
      'close_time'             => $close_dt,
      'winning_open_time_jd'   => $win_open_jd,
      'winning_close_time_jd'  => $win_close_jd,
    ];
}
$stmt->close();

// 5) Clear past‑day winners in memory
foreach ($games as &$g) {
    if ($filter_date !== $today) {
        $g['winning_open_time_jd']  = null;
        $g['winning_close_time_jd'] = null;
    }
}
unset($g);

// 6) At open/close times, fill in missing winners
if ($filter_date === $today) {
  // prepare once
  $upOpen  = $conn->prepare("
    UPDATE jd_rewards
       SET winning_open_time_jd = ?,
           created_at           = NOW()
     WHERE id = ?");
  $upClose = $conn->prepare("
    UPDATE jd_rewards
       SET winning_close_time_jd = ?,
           created_at            = NOW()
     WHERE id = ?");

  foreach ($games as &$g) {
    $open_ts  = strtotime($g['open_time']);
    $close_ts = strtotime($g['close_time']);

    if ($current_ts >= $open_ts && $g['winning_open_time_jd'] === null) {
      $digit = $jd_nums[array_rand($jd_nums)];
      $upOpen->bind_param("si", $digit, $g['id']);
      $upOpen->execute();
      $g['winning_open_time_jd'] = $digit;
    }
    if ($current_ts >= $close_ts && $g['winning_close_time_jd'] === null) {
      $digit2 = $jd_nums[array_rand($jd_nums)];
      $upClose->bind_param("si", $digit2, $g['id']);
      $upClose->execute();
      $g['winning_close_time_jd'] = $digit2;
    }
  }
  unset($g);
  $upOpen->close();
  $upClose->close();
}


// 7) Prepare display timestamp
$current_display = date("d/m/Y g:i A");
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Jodi Digit Bid Results</title>
  <style>
    body { font-family:Arial,sans-serif; background:#f4f9f9; margin:20px; color:#333 }
    h1 { text-align:center }
    .filter { text-align:center; margin-bottom:20px }
    table { width:90%; margin:0 auto; border-collapse:collapse }
    th, td { padding:12px; border:1px solid #ddd; text-align:center }
    th { background:#007BFF; color:#fff }
    tr:nth-child(even) { background:#f9f9f9 }
    .status { font-weight:bold; }
    .yet-open        { color:#555; }
    .bidding-closed  { color:#e67e22; }
    .running         { color:#28a745; }
    .closed          { color:#dc3545; }
  </style>
</head>
<body>
  <h1>Jodi Digit Bid Results</h1>

  <div style="text-align:center; margin-bottom:20px; color:#007BFF;">
    Current IST: <?= $current_display ?>
  </div>

  <div class="filter">
    <form>
      <label>View Rewards:</label>
      <select onchange="location.href=this.value;">
        <option value="rewards_sd.php">Single Digit Rewards</option>
        <option value="rewards_dp.php">Double Panna Rewards</option>
        <option value="rewards_tp.php">Triple Panna Rewards</option>
        <option value="rewards_jd.php" selected>Jodi Digit Rewards</option>
        <option value="rewards_sp.php">Single Panna Rewards</option>
      </select>
    </form>
  </div>

  <table>
    <thead>
      <tr>
        <th>Game Name</th>
        <th>Open Time</th>
        <th>Close Time</th>
        <th>Open Bid Result</th>
        <th>Close Bid Result</th>
        <th>Status</th>
      </tr>
    </thead>
    <tbody>
<?php foreach ($games as $g):
    $openTs   = strtotime($g['open_time']);
    $closeTs  = strtotime($g['close_time']);
    $openSet  = $g['winning_open_time_jd']  !== null;
    $closeSet = $g['winning_close_time_jd'] !== null;
    $showOpen  = ($filter_date === $today && $current_ts >= $openTs && $openSet);
    $showClose = ($filter_date === $today && $current_ts >= $closeTs && $closeSet);
?>
  <tr data-game-id="<?= $g['id'] ?>">
    <td><?= htmlspecialchars($g['game_name']) ?></td>
    <td><?= date('d/m/Y g:i A', $openTs)  ?></td>
    <td><?= date('d/m/Y g:i A', $closeTs) ?></td>

    <!-- Open Bid Result -->
    <td
      class="open-digit"
      contenteditable="true"
      data-type="open"
    >
      <?= $showOpen
           ? $g['winning_open_time_jd']
           : (
               $openSet
                 ? "({$g['winning_open_time_jd']}) N/A"
                 : 'N/A'
             )
      ?>
    </td>

    <!-- Close Bid Result -->
    <td
      class="close-digit"
      contenteditable="true"
      data-type="close"
    >
      <?= $showClose
           ? $g['winning_close_time_jd']
           : (
               $closeSet
                 ? "({$g['winning_close_time_jd']}) N/A"
                 : 'N/A'
             )
      ?>
    </td>

    <td><?= status_pill($current_ts, $openTs, $closeTs) ?></td>
  </tr>
<?php endforeach; ?>
</tbody>


  </table>
    <script>
  // reload page every 30 seconds
  setTimeout(() => window.location.reload(), 10000);
</script>
<script>
  document.querySelectorAll('td.open-digit, td.close-digit').forEach(cell => {
    cell.addEventListener('focus', () => {
      cell.dataset.orig = cell.innerText.trim();
    });
    cell.addEventListener('blur', () => {
      const orig  = cell.dataset.orig;
      const val   = cell.innerText.trim();
      if (val === orig) return;                            // no change
      if (!/^[0-9]{2}$/.test(val)) {                       // validation
        alert("Please enter exactly 2 digits (00–99).");
        cell.innerText = orig;
        return;
      }
      const row    = cell.closest('tr');
      const gameId = row.dataset.gameId;
      const type   = cell.dataset.type; // "open" or "close"

      fetch('rewards_jd.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
          action:    'update_jd_digit',
          game_id:   gameId,
          type:      type,
          digit:     val
        })
      })
      .then(r => r.json())
      .then(res => {
        if (!res.success) {
          alert(res.message || "Update failed");
          cell.innerText = orig;
        } else {
          cell.innerText = `(${val}) N/A`;  // lock UI until time passes
          cell.contentEditable = "false";
        }
      })
      .catch(err => {
        console.error(err);
        cell.innerText = orig;
      });
    });
  });
</script>


</body>
</html>
<?php $conn->close(); ?>
