<?php
// ─────────────────────────────────────────────────────────────────────────────
// rewards_new.php
// ─────────────────────────────────────────────────────────────────────────────

// 1) Hide errors from users, log them instead
ini_set('display_errors','0');
ini_set('log_errors','1');
ini_set('error_log', __DIR__.'/rewards_error.log');
error_reporting(E_ALL);

// 2) Session & timezone
session_start();
date_default_timezone_set('Asia/Kolkata');

// 3) Database
require_once __DIR__.'/db.php'; // must define $conn = new mysqli(...)

// 4) Auth guard
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// 5) Date filter
$filter_date = $_GET['filter_date'] ?? date('Y-m-d');
$now_ts      = time();

// 6) Seed dp_rewards with today’s double_panna games
$seed = "
  INSERT IGNORE INTO dp_rewards
    (game_id, game_name, open_time, close_time, reward_date, created_at)
  SELECT id, game_name, open_time, close_time, ?, NOW()
    FROM game56
   WHERE sub_games='double_panna'
     AND DATE(open_time)=?
";
if ($ins = $conn->prepare($seed)) {
    $ins->bind_param('ss',$filter_date,$filter_date);
    $ins->execute();
    $ins->close();
}

// 7) Fetch & backfill winners (today only)
function backfillWinners($conn,$games) {
    $now = time();
    foreach ($games as $g) {
        $ots = strtotime($g['open_time']);
        $cts = strtotime($g['close_time']);
        if ($now >= $ots && $g['winning_open_time_dp']===null) {
            $d = rand(0,9);
            $conn->query("UPDATE dp_rewards SET winning_open_time_dp=$d WHERE id={$g['id']}");
        }
        if ($now >= $cts && $g['winning_close_time_dp']===null) {
            $d2 = rand(0,9);
            $conn->query("UPDATE dp_rewards SET winning_close_time_dp=$d2 WHERE id={$g['id']}");
        }
    }
}

// 8) Load dp_rewards rows
$games = [];
$sqlG = "
  SELECT id, game_id, open_time, close_time,
         winning_open_time_dp, winning_close_time_dp
    FROM dp_rewards
   WHERE reward_date=?
   ORDER BY open_time
";
if ($st = $conn->prepare($sqlG)) {
    $st->bind_param('s',$filter_date);
    $st->execute();
    $st->bind_result($rid,$gid,$ot,$ct,$wo,$wc);
    while($st->fetch()) {
        $games[] = [
          'id'=>$rid,
          'game_id'=>$gid,
          'open_time'=>$ot,
          'close_time'=>$ct,
          'winning_open_time_dp'=>$wo,
          'winning_close_time_dp'=>$wc
        ];
    }
    $st->close();
}

// 9) If today, fill in missing winners and reload
if ($filter_date === date('Y-m-d')) {
    backfillWinners($conn,$games);
    $games = []; // reload
    if ($st = $conn->prepare($sqlG)) {
        $st->bind_param('s',$filter_date);
        $st->execute();
        $st->bind_result($rid,$gid,$ot,$ct,$wo,$wc);
        while($st->fetch()) {
            $games[] = [
              'id'=>$rid,
              'game_id'=>$gid,
              'open_time'=>$ot,
              'close_time'=>$ct,
              'winning_open_time_dp'=>$wo,
              'winning_close_time_dp'=>$wc
            ];
        }
        $st->close();
    }
}

// 10) Fetch every user bid for those games on that date
$bids = [];
$sqlB = "
  SELECT
    r.game_name,
    DATE_FORMAT(r.open_time,'%l:%i %p') AS open_fmt,
    DATE_FORMAT(r.close_time,'%l:%i %p') AS close_fmt,
    u.username,
    b.bid_details       AS bid_number,
    b.total_bid_amount  AS amt,
    b.bid_session       AS session,
    CASE b.bid_session
      WHEN 'open'  THEN r.winning_open_time_dp
      ELSE r.winning_close_time_dp
    END AS win_number
  FROM user_bids_dp AS b
  JOIN dp_rewards AS r
    ON b.game_id = r.game_id
   AND r.reward_date = ?
  JOIN users_data AS u
    ON b.user_id = u.user_id
  WHERE DATE(b.created_at)=?
  ORDER BY r.open_time, b.bid_session, u.username
";
if ($st = $conn->prepare($sqlB)) {
    $st->bind_param('ss',$filter_date,$filter_date);
    $st->execute();
    $st->bind_result(
      $gname,$open_fmt,$close_fmt,
      $username,$bid_number,$amt,$session,$win_number
    );
    while($st->fetch()) {
        $bids[] = [
          'game_name'=>$gname,
          'open_fmt'=>$open_fmt,
          'close_fmt'=>$close_fmt,
          'username'=>$username,
          'bid_number'=>$bid_number,
          'amt'=>$amt,
          'session'=>$session,
          'win_number'=>$win_number
        ];
    }
    $st->close();
}

// 11) Render HTML + CSS
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Double Panna Bid Results</title>
  <style>
    body {
      font-family:Arial,sans-serif;
      background:#f0f2f5;
      margin:0; padding:2rem;
    }
    .card {
      background:#fff;
      max-width:900px;
      margin:0 auto;
      border-radius:8px;
      box-shadow:0 2px 8px rgba(0,0,0,0.1);
      padding:1.5rem;
    }
    h1 {
      margin:0; font-size:1.8rem;
      text-align:center;
    }
    .subtitle {
      text-align:center;
      color:#007bff;
      margin-top:0.2rem;
      margin-bottom:1rem;
    }
    table {
      width:100%;
      border-collapse:collapse;
      margin-top:1rem;
    }
    th, td {
      padding:12px 8px;
      border-bottom:1px solid #eee;
      text-align:left;
    }
    th {
      background:#007bff;
      color:#fff;
      text-align:center;
    }
    td.center { text-align:center; }
    .icon-win { color:gold; margin-right:4px; }
    .icon-lose { color:#e74c3c; margin-right:4px; }
  </style>
</head>
<body>
  <div class="card">
    <h1>Double Panna Bid Results</h1>
    <p class="subtitle">Current IST: <?=date('d/m/Y g:i A')?></p>

    <table>
      <tr>
        <th>Game</th>
        <th>Open</th>
        <th>Close</th>
        <th>User</th>
        <th>Bid #</th>
        <th>Amt</th>
        <th>Session</th>
        <th>Win #</th>
        <th>Result</th>
      </tr>
      <?php foreach($bids as $r): 
        $winAmt = $r['amt'] * 30;           // 1:30 payout
        $isWin  = ($r['bid_number']==$r['win_number']);
      ?>
      <tr>
        <td class="center"><?=htmlspecialchars($r['game_name'])?></td>
        <td class="center"><?=$r['open_fmt']?></td>
        <td class="center"><?=$r['close_fmt']?></td>
        <td><?=htmlspecialchars($r['username'])?></td>
        <td class="center"><?=$r['bid_number']?></td>
        <td class="center">₹<?=number_format($r['amt'],2)?></td>
        <td class="center"><?=ucfirst($r['session'])?></td>
        <td class="center"><?=$r['win_number']!==null ? $r['win_number']:'N/A'?></td>
        <td class="center">
          <?php if($isWin): ?>
            <span class="icon-win">🏆</span>₹<?=number_format($winAmt,2)?>
          <?php else: ?>
            <span class="icon-lose">❌</span>Better luck
          <?php endif; ?>
        </td>
      </tr>
      <?php endforeach; ?>
      <?php if(empty($bids)): ?>
      <tr>
        <td colspan="9" style="text-align:center; color:#555;">
          No bids placed for <?=htmlspecialchars($filter_date)?>.
        </td>
      </tr>
      <?php endif; ?>
    </table>
  </div>
</body>
</html>
