<?php
/**********************************************************************
 *  rewards_sd.php  –  Single-digit reward generator / viewer
 *********************************************************************/

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();
date_default_timezone_set('Asia/Kolkata');

require 'db.php'; // provides $conn (mysqli)

// ───────────────────────────────────────────────────────────────────
// if AJAX is setting a digit, process & return JSON
if ($_SERVER['REQUEST_METHOD'] === 'POST' 
    && ($_POST['action'] ?? '') === 'update_digit') 
{
    session_start();
    if (empty($_SESSION['user_id'])) {
        echo json_encode(['success'=>false,'message'=>'Unauthorized']);
        exit;
    }

    $game_id     = intval($_POST['game_id']);
    $type        = $_POST['type'];    // 'open' or 'close'
    $digit       = intval($_POST['digit']);
    $filter_date = $_POST['filter_date'] ?? date('Y-m-d');

    if ($digit < 0 || $digit > 9) {
        echo json_encode(['success'=>false,'message'=>'Invalid digit']);
        exit;
    }
    $col = $type==='open' 
         ? 'winning_open_time_sd' 
         : 'winning_close_time_sd';

    $stmt = $conn->prepare("
      UPDATE sd_rewards
         SET $col = ?, created_at = NOW()
       WHERE game_id = ? AND reward_date = ?
    ");
    $stmt->bind_param('iis', $digit, $game_id, $filter_date);
    $success = $stmt->execute();
    $stmt->close();

    echo json_encode(['success'=>$success]);
    exit;
}
// ───────────────────────────────────────────────────────────────────



/* ------------------------------------------------------------------ */
/* 1 – Auth guard                                                     */
/* ------------------------------------------------------------------ */
if (empty($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

/* ------------------------------------------------------------------ */
/* 2 – Date filter (default = today, or ?filter_date=YYYY-MM-DD)      */
/* ------------------------------------------------------------------ */
$filter_date = $_GET['filter_date'] ?? date('Y-m-d');
$today       = date('Y-m-d');
$now         = time();

/* ------------------------------------------------------------------ */
/* 3 – Debug: Check what's in the database first                      */
/* ------------------------------------------------------------------ */
echo "<!-- DEBUG: Checking database content -->";
echo "<!-- Filter date: $filter_date -->";
echo "<!-- Today: $today -->";

// Debug: Check all games in database
$debug_stmt = $conn->prepare("SELECT id, game_name, open_time, close_time, DATE(open_time) as open_date, DATE(close_time) as close_date FROM game56 ORDER BY open_time LIMIT 10");
$debug_stmt->execute();
$debug_result = $debug_stmt->get_result();
echo "<!-- DEBUG: All games in database: -->";
while ($debug_row = $debug_result->fetch_assoc()) {
    echo "<!-- Game: {$debug_row['game_name']}, Open: {$debug_row['open_time']}, Close: {$debug_row['close_time']}, Open Date: {$debug_row['open_date']}, Close Date: {$debug_row['close_date']} -->";
}
$debug_stmt->close();

/* ------------------------------------------------------------------ */
/* 3 – Fetch all games - same logic as games.php                      */
/* ------------------------------------------------------------------ */
$games = [];

// Simply fetch all games for the selected date - same as games.php
$stmt = $conn->prepare("
    SELECT id, game_name, open_time, close_time
      FROM game56
     WHERE DATE(open_time) = ? OR DATE(close_time) = ?
");
$stmt->bind_param('ss', $filter_date, $filter_date);

$stmt->execute();
$stmt->bind_result($gid, $gname, $gopen, $gclose);
while ($stmt->fetch()) {
    $games[$gid] = [
        'id'                    => $gid,
        'game_name'             => $gname,
        'open_time'             => $gopen,
        'close_time'            => $gclose,
        'winning_open_time_sd'  => null,
        'winning_close_time_sd' => null,
    ];
}
$stmt->close();

/* ------------------------------------------------------------------ */
/* 4 – Ensure a rewards row exists (INSERT IGNORE)                   */
/* ------------------------------------------------------------------ */
if (!empty($games)) {
    $ins = $conn->prepare("
        INSERT IGNORE INTO sd_rewards
            (game_id, game_name, open_time, close_time, reward_date)
        VALUES (?, ?, ?, ?, ?)
    ");
    foreach ($games as $g) {
        $ins->bind_param(
            'issss',
            $g['id'],
            $g['game_name'],
            $g['open_time'],
            $g['close_time'],
            $filter_date
        );
        $ins->execute();
    }
    $ins->close();
}

/* ------------------------------------------------------------------ */
/* 5 – Read back any already-generated digits                         */
/* ------------------------------------------------------------------ */
if (!empty($games)) {
    $rw = $conn->prepare("
        SELECT game_id, winning_open_time_sd, winning_close_time_sd
          FROM sd_rewards
         WHERE reward_date = ?
    ");
    $rw->bind_param('s', $filter_date);
    $rw->execute();
    $rw->bind_result($rid, $winOpen, $winClose);
    while ($rw->fetch()) {
        if (isset($games[$rid])) {
            $games[$rid]['winning_open_time_sd']  = $winOpen;
            $games[$rid]['winning_close_time_sd'] = $winClose;
        }
    }
    $rw->close();
}

/* ------------------------------------------------------------------ */
/* 6 – If "today," generate digits once open/close times pass         */
/* ------------------------------------------------------------------ */
if ($filter_date === $today && !empty($games)) {
    $upOpen  = $conn->prepare("
        UPDATE sd_rewards
           SET winning_open_time_sd = ?,
               created_at           = NOW()
         WHERE game_id = ? 
           AND reward_date = ?
    ");
    $upClose = $conn->prepare("
        UPDATE sd_rewards
           SET winning_close_time_sd = ?,
               created_at            = NOW()
         WHERE game_id = ?
           AND reward_date = ?
    ");

    foreach ($games as &$_g) {
        $open_ts  = strtotime($_g['open_time']);
        $close_ts = strtotime($_g['close_time']);

        // — Generate Open digit
        if ($now >= $open_ts && $_g['winning_open_time_sd'] === null) {
            $d = random_int(0, 9);
            $upOpen->bind_param('iis', $d, $_g['id'], $filter_date);
            $upOpen->execute();
            $_g['winning_open_time_sd'] = $d;
            error_log("Open Winning Number for Game ID {$_g['id']}: $d");
        }

        // — Generate Close digit
        if ($now >= $close_ts && $_g['winning_close_time_sd'] === null) {
            $d = random_int(0, 9);
            $upClose->bind_param('iis', $d, $_g['id'], $filter_date);
            $upClose->execute();
            $_g['winning_close_time_sd'] = $d;
            error_log("Close Winning Number for Game ID {$_g['id']}: $d");
        }
    }
    unset($_g);

    $upOpen->close();
    $upClose->close();
}

/* ------------------------------------------------------------------ */
/* 7 – Status pill logic                                              */
/* ------------------------------------------------------------------ */
function status_pill(int $now, int $open_ts, int $close_ts): string
{
    // >10 min before open
    if ($now < $open_ts - 600) {
        return '<span class="status yet-open">Game is open</span>';
    }
    // within last 10 min before open
    if ($now < $open_ts) {
        return '<span class="status bidding-closed">Open bidding closed</span>';
    }
    // from open until 10 min before close
    if ($now < $close_ts - 600) {
        return '<span class="status running">Running</span>';
    }
    // within 10 min before close or after
    return '<span class="status closed">Closed</span>';
}

/* ------------------------------------------------------------------ */
/* 8 – Render HTML                                                    */
/* ------------------------------------------------------------------ */
$current_display = date('d/m/Y g:i A');
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Single-Digit Rewards</title>
  <style>
    body { font-family:Arial,Helvetica,sans-serif; background:#f4f9f9; margin:20px; }
    h1 { text-align:center; }
    table { width:90%; margin:0 auto; border-collapse:collapse; }
    th, td { padding:12px; border:1px solid #ddd; text-align:center; }
    th { background:#007BFF; color:#fff; }
    tr:nth-child(even) { background:#f9f9f9; }
    .status { font-weight:bold; }
    .yet-open       { color:#555; }
    .bidding-closed { color:#e67e22; }
    .running        { color:#28a745; }
    .closed         { color:#dc3545; }
    .filter { text-align:center; margin-bottom:20px; color:#007BFF; }
    .countdown { font-size: 18px; color: #dc3545; font-weight: bold; }
  </style>
</head>
<body>

  <h1>Single-Digit Rewards</h1>

  <div class="filter">
    <form>
      <label>View Rewards:</label>
      <select onchange="location.href=this.value;">
        <option value="rewards_sd.php" selected>Single Digit Rewards</option>
        <option value="rewards_dp.php">Double Panna Rewards</option>
        <option value="rewards_tp.php">Triple Panna Rewards</option>
        <option value="rewards_jd.php">Jodi Digit Rewards</option>
        <option value="rewards_sp.php">Single Panna Rewards</option>
        <option value="rewards_hs.php">Half Sangam Rewards</option>
        <option value="rewards_fs.php">Full Sangam Rewards</option>
      </select>

      &nbsp;&nbsp;
      <label for="filter_date">Date:</label>
      <input
        type="date"
        id="filter_date"
        name="filter_date"
        value="<?= htmlspecialchars($filter_date) ?>"
        onchange="location.href='rewards_sd.php?filter_date='+this.value"
      />
    </form>
  </div>

  <div style="text-align:center; margin-bottom:20px;">
    Current IST: <strong><?= $current_display ?></strong>
  </div>

  <table>
    <thead>
      <tr>
        <th>Game</th>
        <th>Open</th>
        <th>Close</th>
        <th>Win # (Open)</th>
        <th>Win # (Close)</th>
        <th>Status</th>
      </tr>
    </thead>
    <tbody>
<?php if (empty($games)): ?>
    <tr>
        <td colspan="6" style="text-align:center; color:#666; padding:20px;">
            No games found for <?= htmlspecialchars($filter_date) ?>
        </td>
    </tr>
<?php else: ?>
    <?php foreach ($games as $g): 
        // 1) timestamps
        $openTs  = strtotime($g['open_time']);
        $closeTs = strtotime($g['close_time']);

        // 2) has a value been stored?
        $openSet  = $g['winning_open_time_sd']  !== null;
        $closeSet = $g['winning_close_time_sd'] !== null;

        // 3) should we reveal it? (only once time has passed)
        $showOpen  = $openSet  && ($now >= $openTs);
        $showClose = $closeSet && ($now >= $closeTs);

        // 4) can we still edit? (only before its time AND if not yet set)
        $editOpen  = !$openSet  && ($now < $openTs);
        $editClose = !$closeSet && ($now < $closeTs);
    ?>
      <tr data-game-id="<?= $g['id'] ?>">
        <td><?= htmlspecialchars($g['game_name']) ?></td>
        <td><?= date('d/m/Y g:i A', $openTs)  ?></td>
        <td><?= date('d/m/Y g:i A', $closeTs) ?></td>

          <!-- Win # (Open) -->
      <td 
        class="open-digit"
        contenteditable="true"
        data-type="open"
      >
        <?= $showOpen
             // time passed → show just the digit
             ? $g['winning_open_time_sd']
             // you typed one, but time not passed → show "(digit) N/A"
             : ($openSet
                 ? "({$g['winning_open_time_sd']}) N/A"
                 : 'N/A'
               )
        ?>
      </td>

      <!-- Win # (Close) -->
      <td 
        class="close-digit"
        contenteditable="true"
        data-type="close"
      >
        <?= $showClose
             ? $g['winning_close_time_sd']
             : ($closeSet
                 ? "({$g['winning_close_time_sd']}) N/A"
                 : 'N/A'
               )
        ?>
      </td>

        <td>
          <?= status_pill($now, $openTs, $closeTs) ?>
        </td>
      </tr>
    <?php endforeach; ?>
<?php endif; ?>
</tbody>

  </table>

  <script>
    // Poll every 30 seconds for any newly-generated digits:
    function fetchLatestResults() {
      fetch('get_latest_sd_results.php?filter_date=<?= $filter_date ?>')
        .then(r => r.json())
        .then(data => {
          data.forEach(res => {
            const row = document.querySelector(`tr[data-game-id="${res.game_id}"]`);
            if (!row) return;
            row.querySelector('.open-digit').innerText  = res.open_digit  ?? 'N/A';
            row.querySelector('.close-digit').innerText = res.close_digit ?? 'N/A';
          });
        })
        .catch(console.error);
    }

    setInterval(fetchLatestResults, 30000);
  </script>

<script>
// reload page every 10 seconds
setTimeout(()=>window.location.reload(),10000);
</script>

<script>
  document.querySelectorAll('td.open-digit, td.close-digit').forEach(cell => {
    cell.addEventListener('focus', () => {
      cell.dataset.original = cell.innerText.trim();
    });
    cell.addEventListener('blur', () => {
      const orig   = cell.dataset.original;
      const value  = cell.innerText.trim();
      if (value === orig) return;             // no change
      if (!/^[0-9]$/.test(value)) {           // validate
        alert("Enter a single digit (0–9).");
        cell.innerText = orig;
        return;
      }
      const row    = cell.closest('tr');
      const gameId = row.dataset.gameId;
      const type   = cell.dataset.type;

      fetch('rewards_sd.php', {
        method: 'POST',
        headers: { 'Content-Type':'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
          action:      'update_digit',
          game_id:     gameId,
          type:        type,
          digit:       value,
          filter_date: '<?= $filter_date ?>'
        })
      })
      .then(r => r.json())
      .then(res => {
        if (!res.success) {
          alert(res.message || "Update failed");
          cell.innerText = orig;
        } else {
          cell.innerText = `(${value}) N/A`;
          cell.contentEditable = "false";
        }
      })
      .catch(err => {
        console.error(err);
        cell.innerText = orig;
      });
    });
  });
</script>

</body>
</html>
<?php
$conn->close();
?>
