<?php
// Show errors for debugging (remove in production)
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

session_start();
date_default_timezone_set('Asia/Kolkata');
include 'db.php'; // $conn = your mysqli connection

// ───────────────────────────────────────────────────────────────────
// Inline‐edit AJAX handler for Single‐Panna (3-digit) cells
if ($_SERVER['REQUEST_METHOD']==='POST'
    && ($_POST['action'] ?? '')==='update_sp_digit')
{
    header('Content-Type: application/json');
    if (empty($_SESSION['user_id'])) {
        echo json_encode(['success'=>false,'message'=>'Unauthorized']);
        exit;
    }

    $id    = intval($_POST['game_id']);
    $type  = $_POST['type'];       // "open" or "close"
    $digit = trim($_POST['digit']); // must be exactly 3 digits

    if (!preg_match('/^[0-9]{3}$/', $digit)) {
        echo json_encode(['success'=>false,'message'=>'Invalid code']);
        exit;
    }

    $col = $type==='open'
         ? 'winning_open_time_sp'
         : 'winning_close_time_sp';

    $stmt = $conn->prepare("
      UPDATE sp_rewards
         SET {$col} = ?, created_at = NOW()
       WHERE id = ?
    ");
    $stmt->bind_param('si', $digit, $id);
    $ok = $stmt->execute();
    echo json_encode(['success'=> (bool)$ok]);
    exit;
}
// ───────────────────────────────────────────────────────────────────

// 1) Ensure user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

// 2) Build last‑5‑days filter
$last5days = [];
for ($i = 0; $i < 5; $i++) {
    $d = date('Y-m-d', strtotime("-{$i} days"));
    $l = date('d/m/Y',   strtotime("-{$i} days"));
    $last5days[$d] = $l;
}
$filter_date = $_GET['filter_date'] ?? date('Y-m-d');
$today       = date('Y-m-d');
$current_ts  = time();

// **Single‑Panna number sets (12 combos each)**
$sp_nums = [
  '1'=> ['128','137','146','236','245','290','380','470','489','560','579','678'],
  '2'=> ['129','138','147','156','237','246','345','390','480','570','589','679'],
  '3'=> ['120','139','148','157','238','247','256','346','490','580','670','689'],
  '4'=> ['130','149','158','167','239','248','257','347','356','590','680','789'],
  '5'=> ['140','159','168','230','249','258','267','348','357','456','690','780'],
  '6'=> ['123','150','169','178','240','259','268','349','358','367','457','790'],
  '7'=> ['124','160','278','179','250','269','340','359','368','458','467','890'],
  '8'=> ['125','134','170','189','260','279','350','369','468','378','459','567'],
  '9'=> ['126','135','180','234','270','289','360','379','450','469','478','568'],
  '0'=> ['127','136','145','190','235','280','370','389','460','479','569','578'],
];

// 3) Seed sp_rewards from game56 (one row per game)
$ins = $conn->prepare("
  INSERT IGNORE INTO sp_rewards
    (game_id, game_name, open_time, close_time, reward_date, created_at)
  SELECT id, game_name, open_time, close_time, ?, NOW()
    FROM game56
   WHERE DATE(open_time) = ?
");
$ins->bind_param("ss", $filter_date, $filter_date);
$ins->execute();
$ins->close();

// 4) Fetch sp_rewards for this date
$stmt = $conn->prepare("
  SELECT 
    id,
    game_name,
    open_time,
    close_time,
    winning_open_time_sp,
    winning_close_time_sp
  FROM sp_rewards
  WHERE reward_date = ?
  ORDER BY open_time
");
$stmt->bind_param("s", $filter_date);
$stmt->execute();
$stmt->bind_result(
  $rid,
  $gname,
  $open_dt,
  $close_dt,
  $win_open_sp,
  $win_close_sp
);
$games = [];
while ($stmt->fetch()) {
    $games[] = [
      'id'                     => $rid,
      'game_name'              => $gname,
      'open_time'              => $open_dt,
      'close_time'             => $close_dt,
      'winning_open_time_sp'   => $win_open_sp,
      'winning_close_time_sp'  => $win_close_sp,
    ];
}
$stmt->close();

// 5) Clear past‑day winners in memory
foreach ($games as &$g) {
    if ($filter_date !== $today) {
        $g['winning_open_time_sp']  = null;
        $g['winning_close_time_sp'] = null;
    }
}
unset($g);

// 6) At open/close times, fill in missing winners
if ($filter_date === $today) {
    // prepare once
    $upOpen  = $conn->prepare("
      UPDATE sp_rewards
         SET winning_open_time_sp = ?,
             created_at           = NOW()
       WHERE id = ?");
    $upClose = $conn->prepare("
      UPDATE sp_rewards
         SET winning_close_time_sp = ?,
             created_at            = NOW()
       WHERE id = ?");

    foreach ($games as &$g) {
        $open_ts  = strtotime($g['open_time']);
        $close_ts = strtotime($g['close_time']);

        // OPEN
        if ($current_ts >= $open_ts && $g['winning_open_time_sp'] === null) {
            $key   = array_rand($sp_nums);
            $digit = $sp_nums[$key][ array_rand($sp_nums[$key]) ];
            $upOpen->bind_param("si", $digit, $g['id']);
            $upOpen->execute();
            $g['winning_open_time_sp'] = $digit;
        }

        // CLOSE
        if ($current_ts >= $close_ts && $g['winning_close_time_sp'] === null) {
            $key    = array_rand($sp_nums);
            $digit2 = $sp_nums[$key][ array_rand($sp_nums[$key]) ];
            $upClose->bind_param("si", $digit2, $g['id']);
            $upClose->execute();
            $g['winning_close_time_sp'] = $digit2;
        }
    }
    unset($g);
    $upOpen->close();
    $upClose->close();
}


// 7) Prepare display timestamp
$current_display = date("d/m/Y g:i A");
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Single Panna Bid Results</title>
  <style>
    body { font-family:Arial,sans-serif; background:#f4f9f9; margin:20px; color:#333 }
    h1 { text-align:center }
    .filter { text-align:center; margin-bottom:20px }
    table { width:90%; margin:0 auto; border-collapse:collapse }
    th, td { padding:12px; border:1px solid #ddd; text-align:center }
    th { background:#007BFF; color:#fff }
    tr:nth-child(even) { background:#f9f9f9 }
    .status { font-weight:bold; }
    .yet-open        { color:#555; }
    .bidding-closed  { color:#e67e22; }
    .running         { color:#28a745; }
    .closed          { color:#dc3545; }
  </style>
</head>
<body>
  <h1>Single Panna Bid Results</h1>

  <div style="text-align:center; margin-bottom:20px; color:#007BFF;">
    Current IST: <?= $current_display ?>
  </div>

  <div class="filter">
    <form>
      <label>View Rewards:</label>
      <select onchange="location.href=this.value">
        <option value="rewards_sd.php">Single Digit Rewards</option>
        <option value="rewards_dp.php">Double Panna Rewards</option>
        <option value="rewards_tp.php">Triple Panna Rewards</option>
        <option value="rewards_jd.php">Jodi Digit Rewards</option>
        <option value="rewards_sp.php" selected>Single Panna Rewards</option>
      </select>
    </form>
  </div>

  <table>
    <thead>
      <tr>
        <th>Game Name</th>
        <th>Open Time</th>
        <th>Close Time</th>
        <th>Open Bid Result</th>
        <th>Close Bid Result</th>
        <th>Status</th>
      </tr>
    </thead>
   <tbody>
<?php foreach ($games as $g):
    $openTs   = strtotime($g['open_time']);
    $closeTs  = strtotime($g['close_time']);

    // has a SP code been set?
    $openSet  = $g['winning_open_time_sp']  !== null;
    $closeSet = $g['winning_close_time_sp'] !== null;

    // should we reveal the code now?
    $showOpen  = ($filter_date === $today && $current_ts >= $openTs && $openSet);
    $showClose = ($filter_date === $today && $current_ts >= $closeTs && $closeSet);

    // build status
         if ($current_ts < $openTs - 600) $status = '<span class="status yet-open">Game is open</span>';
    elseif ($current_ts < $openTs)       $status = '<span class="status bidding-closed">Open bidding closed</span>';
    elseif ($current_ts < $closeTs - 600)$status = '<span class="status running">Running</span>';
    else                                  $status = '<span class="status closed">Closed</span>';

    $od = date("d/m/Y g:i A", $openTs);
    $cd = date("d/m/Y g:i A", $closeTs);
?>
  <tr data-game-id="<?= $g['id'] ?>">
    <td><?= htmlspecialchars($g['game_name']) ?></td>
    <td><?= $od ?></td>
    <td><?= $cd ?></td>

    <!-- OPEN RESULT -->
    <td
      class="open-digit"
      contenteditable="true"
      data-type="open"
    >
      <?= $showOpen
           ? $g['winning_open_time_sp']
           : (
               $openSet
                 ? "({$g['winning_open_time_sp']}) N/A"
                 : 'N/A'
             )
      ?>
    </td>

    <!-- CLOSE RESULT -->
    <td
      class="close-digit"
      contenteditable="true"
      data-type="close"
    >
      <?= $showClose
           ? $g['winning_close_time_sp']
           : (
               $closeSet
                 ? "({$g['winning_close_time_sp']}) N/A"
                 : 'N/A'
             )
      ?>
    </td>

    <td><?= $status ?></td>
  </tr>
<?php endforeach; ?>
</tbody>

  </table>
  <script>
  // reload page every 30 seconds
  setTimeout(() => window.location.reload(), 10000);
</script>
<script>
  document.querySelectorAll('td.open-digit, td.close-digit').forEach(cell => {
    cell.addEventListener('focus', () => {
      cell.dataset.orig = cell.innerText.trim();
    });
    cell.addEventListener('blur', () => {
      const orig = cell.dataset.orig.trim();
      const val  = cell.innerText.trim();
      if (val === orig) return;               // no change

      // validate exactly 3 digits
      if (!/^[0-9]{3}$/.test(val)) {
        alert("Please enter exactly 3 digits (000–999).");
        cell.innerText = orig;
        return;
      }

      const row    = cell.closest('tr');
      const gameId = row.dataset.gameId;
      const type   = cell.dataset.type;  // "open" or "close"

      fetch('rewards_sp.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
          action:    'update_sp_digit',
          game_id:   gameId,
          type:      type,
          digit:     val
        })
      })
      .then(r => r.json())
      .then(res => {
        if (!res.success) {
          alert(res.message || "Update failed");
          cell.innerText = orig;
        } else {
          // lock cell & show "(val) N/A" until its time passes
          cell.innerText = `(${val}) N/A`;
          cell.contentEditable = "false";
        }
      })
      .catch(err => {
        console.error(err);
        cell.innerText = orig;
      });
    });
  });
</script>

</body>
</html>
<?php $conn->close(); ?>
