<?php
// rewards_tp.php – Triple-Panna rewards & live updater

// 1) show errors while debugging
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

session_start();
date_default_timezone_set('Asia/Kolkata');
require_once __DIR__ . '/db.php';  // creates $conn = new mysqli(…)

// ───────────────────────────────────────────────────────────────────
// Handle inline-TP edits
if ($_SERVER['REQUEST_METHOD'] === 'POST'
  && ($_POST['action'] ?? '') === 'update_tp_digit')
{
    header('Content-Type: application/json');
    // session already started up top
    if (empty($_SESSION['user_id'])) {
        echo json_encode(['success'=>false,'message'=>'Unauthorized']);
        exit;
    }

    $id    = intval($_POST['game_id']);
    $type  = $_POST['type'];       // "open" or "close"
    $digit = trim($_POST['digit']); // should be one of "000".."999"

    // validate exactly three digits
    if (!preg_match('/^[0-9]{3}$/', $digit)) {
        echo json_encode(['success'=>false,'message'=>'Invalid code']);
        exit;
    }

    // pick correct column
    $col = $type === 'open'
         ? 'winning_open_time_tp'
         : 'winning_close_time_tp';

    $stmt = $conn->prepare("
      UPDATE tp_rewards
         SET {$col} = ?, created_at = NOW()
       WHERE id = ?
    ");
    $stmt->bind_param('si', $digit, $id);
    $ok = $stmt->execute();
    echo json_encode(['success' => (bool)$ok]);
    exit;
}
// ───────────────────────────────────────────────────────────────────

// 2) auth
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

// 3) date logic
$filter_date = $_GET['filter_date'] ?? date('Y-m-d');
$today       = date('Y-m-d');
$current_ts  = time();

// 4) our Triple-Panna pool
$tp_nums = ['000','111','222','333','444','555','666','777','888','999'];

// 5) seed one row per game into tp_rewards
$ins = $conn->prepare("
  INSERT IGNORE INTO tp_rewards
    (game_id, game_name, open_time, close_time, reward_date, created_at)
  SELECT id, game_name, open_time, close_time, ?, NOW()
    FROM game56
   WHERE DATE(open_time)=?
");
$ins->bind_param("ss", $filter_date, $filter_date);
$ins->execute();
$ins->close();

// 6) pull them back (no winners yet)
$stmt = $conn->prepare("
  SELECT id, game_name, open_time, close_time,
         winning_open_time_tp, winning_close_time_tp
    FROM tp_rewards
   WHERE reward_date=?
   ORDER BY open_time
");
$stmt->bind_param("s", $filter_date);
$stmt->execute();
$stmt->bind_result(
  $rid, $gname, $open_dt, $close_dt,
  $win_open_tp, $win_close_tp
);
$games = [];
while ($stmt->fetch()) {
    $games[] = [
      'id'                    => $rid,
      'game_name'             => $gname,
      'open_time'             => $open_dt,
      'close_time'            => $close_dt,
      'winning_open_time_tp'  => $win_open_tp,
      'winning_close_time_tp' => $win_close_tp,
    ];
}
$stmt->close();

// 7) forget winners if looking at a past day
if ($filter_date !== $today) {
  foreach ($games as &$g) {
    $g['winning_open_time_tp']  = null;
    $g['winning_close_time_tp'] = null;
  }
  unset($g);
}

// 8) if today’s games are due, pick & write a random triple
if ($filter_date === $today) {
  $upOpen  = $conn->prepare("
    UPDATE tp_rewards
       SET winning_open_time_tp = ?, created_at = NOW()
     WHERE id = ?
  ");
  $upClose = $conn->prepare("
    UPDATE tp_rewards
       SET winning_close_time_tp = ?, created_at = NOW()
     WHERE id = ?
  ");

  foreach ($games as &$g) {
    $open_ts  = strtotime($g['open_time']);
    $close_ts = strtotime($g['close_time']);

    // OPEN
    if ($current_ts >= $open_ts && $g['winning_open_time_tp'] === null) {
      $digit = $tp_nums[array_rand($tp_nums)];
      $upOpen->bind_param("si", $digit, $g['id']);
      $upOpen->execute();
      $g['winning_open_time_tp'] = $digit;
    }

    // CLOSE
    if ($current_ts >= $close_ts && $g['winning_close_time_tp'] === null) {
      $digit2 = $tp_nums[array_rand($tp_nums)];
      $upClose->bind_param("si", $digit2, $g['id']);
      $upClose->execute();
      $g['winning_close_time_tp'] = $digit2;
    }
  }
  unset($g);

  $upOpen->close();
  $upClose->close();
}

// 9) **Re-fetch** so that our display _always_ reflects what’s in the DB
$stmt2 = $conn->prepare("
  SELECT id, game_name, open_time, close_time,
         winning_open_time_tp, winning_close_time_tp
    FROM tp_rewards
   WHERE reward_date=?
   ORDER BY open_time
");
$stmt2->bind_param("s", $filter_date);
$stmt2->execute();
$stmt2->bind_result(
  $rid, $gname, $open_dt, $close_dt,
  $win_open_tp, $win_close_tp
);
$games = [];
while ($stmt2->fetch()) {
    $games[] = [
      'id'                    => $rid,
      'game_name'             => $gname,
      'open_time'             => $open_dt,
      'close_time'            => $close_dt,
      'winning_open_time_tp'  => $win_open_tp,
      'winning_close_time_tp' => $win_close_tp,
    ];
}
$stmt2->close();

// 10) render exactly like dp does
$current_display = date("d/m/Y g:i A");
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Triple Panna Bid Results</title>
  <style>
    body { font-family:Arial,sans-serif; background:#f4f9f9; margin:20px; color:#333 }
    h1 { text-align:center }
    .filter { text-align:center; margin-bottom:20px }
    table { width:90%; margin:0 auto; border-collapse:collapse }
    th, td { padding:12px; border:1px solid #ddd; text-align:center }
    th { background:#007BFF; color:#fff }
    tr:nth-child(even) { background:#f9f9f9 }
    .status { font-weight:bold; }
    .yet-open       { color:#555; }
    .bidding-closed { color:#e67e22; }
    .running        { color:#28a745; }
    .closed         { color:#dc3545; }
  </style>
</head>
<body>
  <h1>Triple Panna Bid Rewards</h1>
  
  
  <div style="text-align:center; margin-bottom:20px; color:#007BFF;">
    Current IST: <?= $current_display ?>
  </div>

  <div class="filter">
    <form>
      <label>View Rewards:</label>
      <select onchange="location.href=this.value;">
        <option value="rewards_sd.php">Single Digit Rewards</option>
        <option value="rewards_dp.php" selected>Double Panna Rewards</option>
        <option value="rewards_tp.php">Triple Panna Rewards</option>
        <option value="rewards_jd.php">Jodi Digit Rewards</option>
        <option value="rewards_sp.php">Single Panna Rewards</option>
      </select>
    </form>
  </div>

  <table>
    <thead>
      <tr>
        <th>Game</th><th>Open</th><th>Close</th>
        <th>Open Result</th><th>Close Result</th><th>Status</th>
      </tr>
    </thead>
    <tbody>
<?php foreach($games as $g):
    $open_ts   = strtotime($g['open_time']);
    $close_ts  = strtotime($g['close_time']);
    $openSet   = $g['winning_open_time_tp']  !== null;
    $closeSet  = $g['winning_close_time_tp'] !== null;
    $showOpen  = ($filter_date === $today && time() >= $open_ts && $openSet);
    $showClose = ($filter_date === $today && time() >= $close_ts && $closeSet);
?>
  <tr data-game-id="<?= $g['id'] ?>">
    <td><?= htmlspecialchars($g['game_name']) ?></td>
    <td><?= date("d/m/Y g:i A", $open_ts) ?></td>
    <td><?= date("d/m/Y g:i A", $close_ts) ?></td>

    <!-- Open Result -->
    <td
      class="open-digit"
      contenteditable="true"
      data-type="open"
    >
      <?= $showOpen
           ? $g['winning_open_time_tp']
           : (
               $openSet
                 ? "({$g['winning_open_time_tp']}) N/A"
                 : 'N/A'
             )
      ?>
    </td>

    <!-- Close Result -->
    <td
      class="close-digit"
      contenteditable="true"
      data-type="close"
    >
      <?= $showClose
           ? $g['winning_close_time_tp']
           : (
               $closeSet
                 ? "({$g['winning_close_time_tp']}) N/A"
                 : 'N/A'
             )
      ?>
    </td>

    <td class="<?= ($current_ts < $open_ts - 600 ? 'yet-open'
               : ($current_ts < $open_ts     ? 'bidding-closed'
               : ($current_ts < $close_ts-600 ? 'running' : 'closed')
             )) ?>">
      <?= ucwords(str_replace('-', ' ',
         ($current_ts < $open_ts - 600 ? 'yet-open'
           : ($current_ts < $open_ts     ? 'bidding-closed'
           : ($current_ts < $close_ts-600 ? 'running' : 'closed'))
         )
      )) ?>
    </td>
  </tr>
<?php endforeach; ?>
</tbody>

  </table>
  <script>setTimeout(()=>location.reload(),10000);</script>
  <script>
  document.querySelectorAll('td.open-digit, td.close-digit').forEach(cell => {
    cell.addEventListener('focus', () => {
      cell.dataset.orig = cell.innerText.trim();
    });
    cell.addEventListener('blur', () => {
      const orig  = cell.dataset.orig;
      const value = cell.innerText.trim();
      if (value === orig) return;                      // no change
      if (!/^[0-9]{3}$/.test(value)) {                 // validate
        alert("Please enter exactly 3 digits (000–999).");
        cell.innerText = orig;
        return;
      }

      const row    = cell.closest('tr');
      const gameId = row.dataset.gameId;
      const type   = cell.dataset.type; // "open" or "close"

      fetch('rewards_tp.php', {
        method: 'POST',
        headers: {'Content-Type':'application/x-www-form-urlencoded'},
        body: new URLSearchParams({
          action:    'update_tp_digit',
          game_id:   gameId,
          type:      type,
          digit:     value
        })
      })
      .then(r => r.json())
      .then(res => {
        if (!res.success) {
          alert(res.message || "Update failed");
          cell.innerText = orig;
        } else {
          // lock & show “(xxx) N/A” until time passes
          cell.innerText = `(${value}) N/A`;
          cell.contentEditable = "false";
        }
      })
      .catch(err => {
        console.error(err);
        cell.innerText = orig;
      });
    });
  });
</script>

</body>
</html>
<?php $conn->close(); ?>
