<?php
/**********************************************************************
 * rews.php — ONE FILE to generate rewards for ALL subgames (SD/DP/SP/TP/JD/HS/FS)
 * - Safe to run repeatedly (idempotent).
 * - Fills winners only when each session’s open/close time has passed.
 * - Works as a CLI daemon (runs forever) or single HTTP sweep.
 *
 * Master table expected: game56(id, game_name, open_time, close_time [DATETIME])
 * Reward tables (per your screenshots):
 *   sd_rewards(id, game_id, game_name, open_time TIME, close_time TIME, reward_date DATE,
 *              winning_open_time_sd TINYINT, winning_close_time_sd TINYINT, created_at)
 *   dp_rewards(..., winning_open_time_dp VARCHAR(10), winning_close_time_dp VARCHAR(10), ...)
 *   sp_rewards(..., winning_open_time_sp VARCHAR(10), winning_close_time_sp VARCHAR(10), ...)
 *   tp_rewards(..., winning_open_time_tp VARCHAR(10), winning_close_time_tp VARCHAR(10), ...)
 *   jd_rewards(..., winning_open_time_jd VARCHAR(10), winning_close_time_jd VARCHAR(10), ...)
 *   hs_rewards(..., winning_open_time_od_hs VARCHAR(10), winning_close_time_od_hs VARCHAR(10),
 *                    winning_open_time_cp_hs VARCHAR(10), winning_close_time_cp_hs VARCHAR(10), ...)
 *   fs_rewards(..., winning_open_op_fs, winning_open_cp_fs, winning_close_op_fs, winning_close_cp_fs)
 * Each rewards table should have UNIQUE (game_id, reward_date) so INSERT IGNORE works.
 *********************************************************************/

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

date_default_timezone_set('Asia/Kolkata');
require_once __DIR__.'/db.php';           // provides $conn = new mysqli(...)
$conn->set_charset('utf8mb4');

// ───────────────────────────────── helpers
function now_ts()         { return time(); }
function today_date()     { return date('Y-m-d'); }
function to_ts(string $date, string $timeLike): int {
  // If $timeLike already contains a date/time, trust strtotime; else attach $date
  if (preg_match('/\d{4}-\d{2}-\d{2}/', $timeLike)) return strtotime($timeLike);
  return strtotime("$date $timeLike");
}
function log_line(string $s) {
  // Echo and flush so you can tail output when running in CLI/daemon mode.
  echo '['.date('d-m-Y H:i:s')."] $s\n";
  @flush();
}

// Pools (exactly as used across your existing per-game pages)
$DOUBLE_PANNA = [
  0 => ['118','226','244','299','334','488','550','668','677'],
  1 => ['100','119','155','227','335','344','399','588','669'],
  2 => ['110','200','228','255','336','499','660','688','778'],
  3 => ['166','229','300','337','355','445','599','779','788'],
  4 => ['112','220','266','338','400','446','455','699','770'],
  5 => ['113','122','177','339','366','447','500','799','889'],
  6 => ['600','114','277','330','448','466','556','880','899'],
  7 => ['115','133','188','223','377','449','557','566','700'],
  8 => ['116','224','233','288','440','477','558','800','990'],
  9 => ['117','144','199','225','388','559','577','667','900'],
];
$SINGLE_PANNA = [
  '1'=> ['128','137','146','236','245','290','380','470','489','560','579','678'],
  '2'=> ['129','138','147','156','237','246','345','390','480','570','589','679'],
  '3'=> ['120','139','148','157','238','247','256','346','490','580','670','689'],
  '4'=> ['130','149','158','167','239','248','257','347','356','590','680','789'],
  '5'=> ['140','159','168','230','249','258','267','348','357','456','690','780'],
  '6'=> ['123','150','169','178','240','259','268','349','358','367','457','790'],
  '7'=> ['124','160','278','179','250','269','340','359','368','458','467','890'],
  '8'=> ['125','134','170','189','260','279','350','369','468','378','459','567'],
  '9'=> ['126','135','180','234','270','289','360','379','450','469','478','568'],
  '0'=> ['127','136','145','190','235','280','370','389','460','479','569','578'],
];
$TRIPLE_PANNA = ['000','111','222','333','444','555','666','777','888','999'];

// HS: simple flat 00–99 pool (2 digits)
$HS_POOL = (function(){
  $a=[]; for($i=0;$i<100;$i++) $a[] = str_pad((string)$i,2,'0',STR_PAD_LEFT); return $a;
})();

// FS pools (mix of jodi+single+double+triple panna)
$JODI_POOL = $HS_POOL;
$SP_FLAT   = array_merge(...array_values($SINGLE_PANNA));
$DP_FLAT   = array_merge(...array_values($DOUBLE_PANNA));
$FS_SETS   = [
  'Jodi Digit'  => $JODI_POOL,
  'Single Pana' => $SP_FLAT,
  'Double Pana' => $DP_FLAT,
  'Triple Pana' => $TRIPLE_PANNA,
];
function fs_pick(array $FS_SETS): string {
  $cats = array_keys($FS_SETS);
  $cat  = $cats[array_rand($cats)];
  $list = $FS_SETS[$cat];
  return $list[array_rand($list)];
}
function pick_3_from_group(array $groups): string {
  $k = array_rand($groups);
  $lst = $groups[$k];
  return $lst[array_rand($lst)];
}

// ─────────────────────────────── seeding (INSERT IGNORE from game56)
function seed_for_date(mysqli $conn, string $table, string $reward_date): void {
  $sql = "
    INSERT IGNORE INTO {$table}
      (game_id, game_name, open_time, close_time, reward_date, created_at)
    SELECT id, game_name, TIME(open_time), TIME(close_time), ?, NOW()
      FROM game56
     WHERE DATE(open_time)=? OR DATE(close_time)=?
  ";
  $stmt = $conn->prepare($sql);
  $stmt->bind_param('sss', $reward_date, $reward_date, $reward_date);
  $stmt->execute();
  $stmt->close();
}

// ─────────────────────────────── per-table generators
function run_sd(mysqli $conn, string $date, int $now): array {
  seed_for_date($conn, 'sd_rewards', $date);

  $stmt = $conn->prepare("
    SELECT id, open_time, close_time,
           winning_open_time_sd, winning_close_time_sd
      FROM sd_rewards WHERE reward_date=? ORDER BY open_time
  ");
  $stmt->bind_param('s', $date);
  $stmt->execute();
  $stmt->bind_result($id,$ot,$ct,$wo,$wc);

  $upO = $conn->prepare("
    UPDATE sd_rewards
       SET winning_open_time_sd=?, created_at=NOW()
     WHERE id=? AND winning_open_time_sd IS NULL
  ");
  $upC = $conn->prepare("
    UPDATE sd_rewards
       SET winning_close_time_sd=?, created_at=NOW()
     WHERE id=? AND winning_close_time_sd IS NULL
  ");

  $cO=$cC=0;
  while($stmt->fetch()){
    $oTs = to_ts($date,$ot); $cTs = to_ts($date,$ct);
    if ($wo === null && $now >= $oTs) { $d = random_int(0,9); $upO->bind_param('ii',$d,$id); $upO->execute(); $cO++; }
    if ($wc === null && $now >= $cTs) { $d = random_int(0,9); $upC->bind_param('ii',$d,$id); $upC->execute(); $cC++; }
  }
  $stmt->close(); $upO->close(); $upC->close();
  return ['open'=>$cO,'close'=>$cC];
}

function run_jd(mysqli $conn, string $date, int $now, array $pool): array {
  seed_for_date($conn, 'jd_rewards', $date);
  $stmt=$conn->prepare("
    SELECT id, open_time, close_time,
           winning_open_time_jd, winning_close_time_jd
      FROM jd_rewards WHERE reward_date=? ORDER BY open_time
  ");
  $stmt->bind_param('s',$date);
  $stmt->execute();
  $stmt->bind_result($id,$ot,$ct,$wo,$wc);
  $upO=$conn->prepare("
    UPDATE jd_rewards
       SET winning_open_time_jd=?, created_at=NOW()
     WHERE id=? AND winning_open_time_jd IS NULL
  ");
  $upC=$conn->prepare("
    UPDATE jd_rewards
       SET winning_close_time_jd=?, created_at=NOW()
     WHERE id=? AND winning_close_time_jd IS NULL
  ");
  $cO=$cC=0;
  while($stmt->fetch()){
    $oTs = to_ts($date,$ot); $cTs = to_ts($date,$ct);
    if ($wo===null && $now >= $oTs){ $d=$pool[array_rand($pool)]; $upO->bind_param('si',$d,$id); $upO->execute(); $cO++; }
    if ($wc===null && $now >= $cTs){ $d=$pool[array_rand($pool)]; $upC->bind_param('si',$d,$id); $upC->execute(); $cC++; }
  }
  $stmt->close(); $upO->close(); $upC->close();
  return ['open'=>$cO,'close'=>$cC];
}

function run_dp(mysqli $conn, string $date, int $now, array $groups): array {
  seed_for_date($conn, 'dp_rewards', $date);
  $stmt=$conn->prepare("
    SELECT id, open_time, close_time,
           winning_open_time_dp, winning_close_time_dp
      FROM dp_rewards WHERE reward_date=? ORDER BY open_time
  ");
  $stmt->bind_param('s',$date); $stmt->execute();
  $stmt->bind_result($id,$ot,$ct,$wo,$wc);
  $upO=$conn->prepare("
    UPDATE dp_rewards
       SET winning_open_time_dp=?, created_at=NOW()
     WHERE id=? AND winning_open_time_dp IS NULL
  ");
  $upC=$conn->prepare("
    UPDATE dp_rewards
       SET winning_close_time_dp=?, created_at=NOW()
     WHERE id=? AND winning_close_time_dp IS NULL
  ");
  $cO=$cC=0;
  while($stmt->fetch()){
    $oTs=to_ts($date,$ot); $cTs=to_ts($date,$ct);
    if ($wo===null && $now >= $oTs){ $d=pick_3_from_group($groups); $upO->bind_param('si',$d,$id); $upO->execute(); $cO++; }
    if ($wc===null && $now >= $cTs){ $d=pick_3_from_group($groups); $upC->bind_param('si',$d,$id); $upC->execute(); $cC++; }
  }
  $stmt->close(); $upO->close(); $upC->close();
  return ['open'=>$cO,'close'=>$cC];
}

function run_sp(mysqli $conn, string $date, int $now, array $groups): array {
  seed_for_date($conn, 'sp_rewards', $date);
  $stmt=$conn->prepare("
    SELECT id, open_time, close_time,
           winning_open_time_sp, winning_close_time_sp
      FROM sp_rewards WHERE reward_date=? ORDER BY open_time
  ");
  $stmt->bind_param('s',$date); $stmt->execute();
  $stmt->bind_result($id,$ot,$ct,$wo,$wc);
  $upO=$conn->prepare("
    UPDATE sp_rewards
       SET winning_open_time_sp=?, created_at=NOW()
     WHERE id=? AND winning_open_time_sp IS NULL
  ");
  $upC=$conn->prepare("
    UPDATE sp_rewards
       SET winning_close_time_sp=?, created_at=NOW()
     WHERE id=? AND winning_close_time_sp IS NULL
  ");
  $cO=$cC=0;
  while($stmt->fetch()){
    $oTs=to_ts($date,$ot); $cTs=to_ts($date,$ct);
    if ($wo===null && $now >= $oTs){ $d=pick_3_from_group($groups); $upO->bind_param('si',$d,$id); $upO->execute(); $cO++; }
    if ($wc===null && $now >= $cTs){ $d=pick_3_from_group($groups); $upC->bind_param('si',$d,$id); $upC->execute(); $cC++; }
  }
  $stmt->close(); $upO->close(); $upC->close();
  return ['open'=>$cO,'close'=>$cC];
}

function run_tp(mysqli $conn, string $date, int $now, array $pool): array {
  seed_for_date($conn, 'tp_rewards', $date);
  $stmt=$conn->prepare("
    SELECT id, open_time, close_time,
           winning_open_time_tp, winning_close_time_tp
      FROM tp_rewards WHERE reward_date=? ORDER BY open_time
  ");
  $stmt->bind_param('s',$date); $stmt->execute();
  $stmt->bind_result($id,$ot,$ct,$wo,$wc);
  $upO=$conn->prepare("
    UPDATE tp_rewards
       SET winning_open_time_tp=?, created_at=NOW()
     WHERE id=? AND winning_open_time_tp IS NULL
  ");
  $upC=$conn->prepare("
    UPDATE tp_rewards
       SET winning_close_time_tp=?, created_at=NOW()
     WHERE id=? AND winning_close_time_tp IS NULL
  ");
  $cO=$cC=0;
  while($stmt->fetch()){
    $oTs=to_ts($date,$ot); $cTs=to_ts($date,$ct);
    if ($wo===null && $now >= $oTs){ $d=$pool[array_rand($pool)]; $upO->bind_param('si',$d,$id); $upO->execute(); $cO++; }
    if ($wc===null && $now >= $cTs){ $d=$pool[array_rand($pool)]; $upC->bind_param('si',$d,$id); $upC->execute(); $cC++; }
  }
  $stmt->close(); $upO->close(); $upC->close();
  return ['open'=>$cO,'close'=>$cC];
}

function run_hs(mysqli $conn, string $date, int $now, array $pool): array {
  seed_for_date($conn, 'hs_rewards', $date);
  $stmt=$conn->prepare("
    SELECT id, open_time, close_time,
           winning_open_time_od_hs,
           winning_open_time_cp_hs,
           winning_close_time_od_hs,
           winning_close_time_cp_hs
      FROM hs_rewards WHERE reward_date=? ORDER BY open_time
  ");
  $stmt->bind_param('s',$date); $stmt->execute();
  $stmt->bind_result($id,$ot,$ct,$oo,$oc,$co,$cc);

  $up = [
    'open_od'  => $conn->prepare("UPDATE hs_rewards SET winning_open_time_od_hs=?,  created_at=NOW() WHERE id=? AND winning_open_time_od_hs  IS NULL"),
    'open_cp'  => $conn->prepare("UPDATE hs_rewards SET winning_open_time_cp_hs=?,  created_at=NOW() WHERE id=? AND winning_open_time_cp_hs  IS NULL"),
    'close_od' => $conn->prepare("UPDATE hs_rewards SET winning_close_time_od_hs=?, created_at=NOW() WHERE id=? AND winning_close_time_od_hs IS NULL"),
    'close_cp' => $conn->prepare("UPDATE hs_rewards SET winning_close_time_cp_hs=?, created_at=NOW() WHERE id=? AND winning_close_time_cp_hs IS NULL"),
  ];
  $c = ['open_od'=>0,'open_cp'=>0,'close_od'=>0,'close_cp'=>0];

  while($stmt->fetch()){
    $oTs=to_ts($date,$ot); $cTs=to_ts($date,$ct);
    if ($oo===null && $now >= $oTs){ $d=$pool[array_rand($pool)]; $up['open_od']->bind_param('si',$d,$id); $up['open_od']->execute(); $c['open_od']++; }
    if ($oc===null && $now >= $oTs){ $d=$pool[array_rand($pool)]; $up['open_cp']->bind_param('si',$d,$id); $up['open_cp']->execute(); $c['open_cp']++; }
    if ($co===null && $now >= $cTs){ $d=$pool[array_rand($pool)]; $up['close_od']->bind_param('si',$d,$id); $up['close_od']->execute(); $c['close_od']++; }
    if ($cc===null && $now >= $cTs){ $d=$pool[array_rand($pool)]; $up['close_cp']->bind_param('si',$d,$id); $up['close_cp']->execute(); $c['close_cp']++; }
  }
  $stmt->close();
  foreach($up as $p){ $p->close(); }
  return $c;
}

function run_fs(mysqli $conn, string $date, int $now, array $FS_SETS): array {
  seed_for_date($conn, 'fs_rewards', $date);
  $stmt=$conn->prepare("
    SELECT id, open_time, close_time,
           winning_open_op_fs, winning_open_cp_fs,
           winning_close_op_fs, winning_close_cp_fs
      FROM fs_rewards WHERE reward_date=? ORDER BY open_time
  ");
  $stmt->bind_param('s',$date); $stmt->execute();
  $stmt->bind_result($id,$ot,$ct,$oo,$oc,$co,$cc);

  $up = [
    'open_op'  => $conn->prepare("UPDATE fs_rewards SET winning_open_op_fs=?,  created_at=NOW() WHERE id=? AND winning_open_op_fs  IS NULL"),
    'open_cp'  => $conn->prepare("UPDATE fs_rewards SET winning_open_cp_fs=?,  created_at=NOW() WHERE id=? AND winning_open_cp_fs  IS NULL"),
    'close_op' => $conn->prepare("UPDATE fs_rewards SET winning_close_op_fs=?, created_at=NOW() WHERE id=? AND winning_close_op_fs IS NULL"),
    'close_cp' => $conn->prepare("UPDATE fs_rewards SET winning_close_cp_fs=?, created_at=NOW() WHERE id=? AND winning_close_cp_fs IS NULL"),
  ];
  $c = ['open_op'=>0,'open_cp'=>0,'close_op'=>0,'close_cp'=>0];

  while($stmt->fetch()){
    $oTs=to_ts($date,$ot); $cTs=to_ts($date,$ct);
    if ($oo===null && $now >= $oTs){ $d=fs_pick($FS_SETS); $up['open_op']->bind_param('si',$d,$id); $up['open_op']->execute(); $c['open_op']++; }
    if ($oc===null && $now >= $oTs){ $d=fs_pick($FS_SETS); $up['open_cp']->bind_param('si',$d,$id); $up['open_cp']->execute(); $c['open_cp']++; }
    if ($co===null && $now >= $cTs){ $d=fs_pick($FS_SETS); $up['close_op']->bind_param('si',$d,$id); $up['close_op']->execute(); $c['close_op']++; }
    if ($cc===null && $now >= $cTs){ $d=fs_pick($FS_SETS); $up['close_cp']->bind_param('si',$d,$id); $up['close_cp']->execute(); $c['close_cp']++; }
  }
  $stmt->close();
  foreach($up as $p){ $p->close(); }
  return $c;
}

// ─────────────────────────────── one full sweep
function do_one_sweep(mysqli $conn): array {
  $date = today_date();
  $now  = now_ts();
  // keep connection healthy in daemons
  if (method_exists($conn,'ping')) { @$conn->ping(); }

  $out = [];
  $out['sd'] = run_sd($conn,$date,$now);
  $out['jd'] = run_jd($conn,$date,$now, $GLOBALS['JODI_POOL']);
  $out['dp'] = run_dp($conn,$date,$now, $GLOBALS['DOUBLE_PANNA']);
  $out['sp'] = run_sp($conn,$date,$now, $GLOBALS['SINGLE_PANNA']);
  $out['tp'] = run_tp($conn,$date,$now, $GLOBALS['TRIPLE_PANNA']);
  $out['hs'] = run_hs($conn,$date,$now, $GLOBALS['HS_POOL']);
  $out['fs'] = run_fs($conn,$date,$now, $GLOBALS['FS_SETS']);
  return $out;
}

// ─────────────────────────────── runner (CLI daemon or HTTP once)
$daemonRequested = (PHP_SAPI === 'cli') || (isset($_GET['daemon']) && $_GET['daemon']=='1');

if ($daemonRequested) {
  ignore_user_abort(true);
  set_time_limit(0);
  log_line("REWARDS DAEMON start (date=".today_date().")");
  while (true) {
    try {
      $res = do_one_sweep($conn);
      log_line("Sweep: "
        ."SD[o{$res['sd']['open']}/c{$res['sd']['close']}] "
        ."JD[o{$res['jd']['open']}/c{$res['jd']['close']}] "
        ."DP[o{$res['dp']['open']}/c{$res['dp']['close']}] "
        ."SP[o{$res['sp']['open']}/c{$res['sp']['close']}] "
        ."TP[o{$res['tp']['open']}/c{$res['tp']['close']}] "
        ."HS[oo{$res['hs']['open_od']}, oc{$res['hs']['open_cp']}, co{$res['hs']['close_od']}, cc{$res['hs']['close_cp']}] "
        ."FS[oo{$res['fs']['open_op']}, oc{$res['fs']['open_cp']}, co{$res['fs']['close_op']}, cc{$res['fs']['close_cp']}]"
      );
    } catch (Throwable $e) {
      log_line("ERROR: ".$e->getMessage());
    }
    sleep(20); // wake every 20s
  }
} else {
  // Single web request: one sweep and a tiny JSON status
  header('Content-Type: application/json');
  try {
    $res = do_one_sweep($conn);
    echo json_encode([
      'date' => today_date(),
      'ok'   => true,
      'updated' => $res
    ]);
  } catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['ok'=>false,'error'=>$e->getMessage()]);
  }
}
