<?php
// settings.php
session_start();
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

// 1) Your existing mysqli connection
require_once __DIR__ . '/db.php';  // defines $conn = new mysqli(...)

if (!isset($_SESSION['user_id'])) {
  header('Location: login.php');
  exit;
}
$userId = (int)$_SESSION['user_id'];

// 2) Fetch current profile + plain password
$stmt = $conn->prepare("
  SELECT username,email,mobile,password
    FROM users_data
   WHERE user_id = ?
   LIMIT 1
");
$stmt->bind_param('i', $userId);
$stmt->execute();
$stmt->bind_result($curUser, $curEmail, $curMobile, $curPassword);
if (!$stmt->fetch()) {
  die("User not found");
}
$stmt->close();

$errors  = [];
$success = '';

// 3a) Profile update
if (isset($_POST['update_profile'])) {
  $u = trim($_POST['username'] ?? '');
  $e = trim($_POST['email']    ?? '');
  $m = trim($_POST['mobile']   ?? '');

  if (strlen($u) < 3) {
    $errors[] = "Username must be at least 3 characters.";
  }
  if (!filter_var($e, FILTER_VALIDATE_EMAIL)) {
    $errors[] = "Invalid email address.";
  }
  if (!preg_match('/^\d{10}$/', $m)) {
    $errors[] = "Mobile must be exactly 10 digits.";
  }

  if (empty($errors)) {
    $up = $conn->prepare("
      UPDATE users_data
         SET username=?, email=?, mobile=?
       WHERE user_id=?
    ");
    $up->bind_param('sssi', $u, $e, $m, $userId);
    $up->execute();
    $up->close();

    $success    = "Profile updated successfully.";
    $curUser    = $u;
    $curEmail   = $e;
    $curMobile  = $m;
  }
}

// 3b) Password change (plain text)
if (isset($_POST['change_password'])) {
  $current = $_POST['current_password'] ?? '';
  $new     = $_POST['new_password']     ?? '';
  $confirm = $_POST['confirm_password'] ?? '';

  if ($current !== $curPassword) {
    $errors[] = "Current password is incorrect.";
  }
  if (strlen($new) < 6) {
    $errors[] = "New password must be at least 6 characters.";
  }
  if ($new !== $confirm) {
    $errors[] = "New password and confirmation do not match.";
  }

  if (empty($errors)) {
    $up2 = $conn->prepare("
      UPDATE users_data
         SET password=?
       WHERE user_id=?
    ");
    $up2->bind_param('si', $new, $userId);
    $up2->execute();
    $up2->close();

    $success    = "Password changed successfully.";
    // reflect the new plain-text password
    $curPassword = $new;
  }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Settings</title>
  <link
    href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css"
    rel="stylesheet">
</head>
<body class="bg-light">
  <div class="container py-5">
    <h1 class="mb-4">Account Settings</h1>

    <?php if ($success): ?>
      <div class="alert alert-success"><?=htmlspecialchars($success)?></div>
    <?php endif; ?>

    <?php if ($errors): ?>
      <div class="alert alert-danger">
        <ul class="mb-0">
          <?php foreach ($errors as $err): ?>
            <li><?=htmlspecialchars($err)?></li>
          <?php endforeach; ?>
        </ul>
      </div>
    <?php endif; ?>

    <!-- Profile Card -->
    <div class="card mb-4">
      <div class="card-header bg-primary text-white">
        Edit Profile
      </div>
      <div class="card-body">
        <form method="post">
          <div class="mb-3">
            <label class="form-label">Username</label>
            <input
              type="text"
              name="username"
              class="form-control"
              value="<?=htmlspecialchars($curUser)?>">
          </div>
          <div class="mb-3">
            <label class="form-label">Email</label>
            <input
              type="email"
              name="email"
              class="form-control"
              value="<?=htmlspecialchars($curEmail)?>">
          </div>
          <div class="mb-3">
            <label class="form-label">Mobile</label>
            <input
              type="text"
              name="mobile"
              class="form-control"
              value="<?=htmlspecialchars($curMobile)?>">
          </div>
          <button
            type="submit"
            name="update_profile"
            class="btn btn-primary">
            Save Profile
          </button>
        </form>
      </div>
    </div>

    <!-- Change Password Card -->
    <div class="card">
      <div class="card-header bg-warning text-dark">
        Change Password
      </div>
      <div class="card-body">
        <form method="post">
          <div class="mb-3">
            <label class="form-label">Current Password</label>
            <input
              type="password"
              name="current_password"
              class="form-control"
              required>
          </div>
          <div class="mb-3">
            <label class="form-label">New Password</label>
            <input
              type="password"
              name="new_password"
              class="form-control"
              required>
          </div>
          <div class="mb-3">
            <label class="form-label">Confirm New Password</label>
            <input
              type="password"
              name="confirm_password"
              class="form-control"
              required>
          </div>
          <button
            type="submit"
            name="change_password"
            class="btn btn-warning">
            Change Password
          </button>
        </form>
      </div>
    </div>

  </div>
</body>
</html>
