<?php
// single_digit.php

// DEV: show errors (remove in prod)
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

session_start();
include 'db.php'; // your mysqli connection in $conn

// 1) Auth
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}
$user_id = (int)$_SESSION['user_id'];

// 2) Get & validate game_id
if (empty($_GET['game_id'])) {
    die("No game specified.");
}
$game_id = (int)$_GET['game_id'];

// 3) Mark sub_game
$stmt = $conn->prepare("UPDATE game56 SET sub_games = 'single_digit' WHERE id = ?");
$stmt->bind_param("i", $game_id);
$stmt->execute();
$stmt->close();

// 4) Fetch username
$stmt = $conn->prepare("SELECT username FROM users_data WHERE user_id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$stmt->bind_result($username);
$stmt->fetch();
$stmt->close();

// 5) Fetch game details
$stmt = $conn->prepare("
  SELECT game_name, open_time, close_time
    FROM game56
   WHERE id = ?
");
$stmt->bind_param("i", $game_id);
$stmt->execute();
$stmt->bind_result($game_name, $open_time, $close_time);
if (!$stmt->fetch()) {
    die("Game not found.");
}
$stmt->close();

// 6) Compute toggle logic with 10 min buffers
date_default_timezone_set('Asia/Kolkata');
$now     = new DateTime();
$openDT  = DateTime::createFromFormat('H:i:s', $open_time);
$closeDT = DateTime::createFromFormat('H:i:s', $close_time);

// handle overnight close
if ($closeDT <= $openDT) {
    $closeDT->modify('+1 day');
}

// 10 min before open → start running; 10 min before close → end running
$runStart = (clone $openDT)->modify('-10 minutes');
$runEnd   = (clone $closeDT)->modify('-10 minutes');

if ($now < $runStart) {
    // before game window
    $openBtnClass  = 'btn-blue';
    $closeBtnClass = 'btn-default';
} elseif ($now >= $runStart && $now <= $runEnd) {
    // during game window
    $openBtnClass  = 'btn-default';
    $closeBtnClass = 'btn-blue';
} else {
    // after window
    $openBtnClass  = 'btn-default';
    $closeBtnClass = 'btn-default';
}

// 7) Fetch wallet balance
$stmt = $conn->prepare("SELECT balance FROM wallet WHERE user_id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$stmt->bind_result($wallet_balance);
$stmt->fetch();
$stmt->close();

// 8) Bidding rules
$min_bid = 10;

// 9) Handle submission
$viewLink = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $bids  = [];
    $total = 0;
    for ($i = 0; $i <= 9; $i++) {
        $field = "bid_{$i}";
        if (isset($_POST[$field]) && $_POST[$field] !== '') {
            $amt = floatval($_POST[$field]);
            if ($amt >= $min_bid) {
                $bids[$i] = $amt;
                $total   += $amt;
            }
        }
    }
    if ($total > 0 && $total <= $wallet_balance) {
        $conn->begin_transaction();
        try {
            $details = json_encode($bids);
            // determine session (open vs close)
            $session = ($now >= $openDT && $now <= $closeDT) ? 'close' : 'open';

            // insert bid
            $ins = $conn->prepare("
              INSERT INTO user_bids
                (user_id, game_id, bid_details, total_bid_amount, bid_session, game_name)
              VALUES (?, ?, ?, ?, ?, ?)
            ");
            $ins->bind_param(
              "iisdss",
              $user_id,
              $game_id,
              $details,
              $total,
              $session,
              $game_name
            );
            $ins->execute();
            $ins->close();

            // deduct wallet
            $upd = $conn->prepare("UPDATE wallet SET balance = balance - ? WHERE user_id = ?");
            $upd->bind_param("di", $total, $user_id);
            $upd->execute();
            $upd->close();

            $conn->commit();
            $wallet_balance -= $total;

            // only show the View Results link
            $viewLink = "<p style='text-align:center; margin:20px 0;'>
                           <a href='random.php?game_id={$game_id}'>View Results</a>
                         </p>";
        } catch (Exception $e) {
            $conn->rollback();
            // silently fail or log $e->getMessage()
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Single Digit Bidding</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <style>
    body { background:#f4f4f9; margin:0; font-family:Arial,sans-serif; color:#333 }
    .back { display:inline-block; margin:20px; color:#1E90FF; text-decoration:none }
    .back:hover { text-decoration:underline }
    .container {
      max-width:600px; margin:0 auto; background:#fff; padding:20px;
      border-radius:8px; box-shadow:0 2px 5px rgba(0,0,0,0.1);
    }
    h1 { text-align:center; color:#1E90FF }
    .info { margin:8px 0; font-size:1rem; }
    .info strong { display:inline-block; width:100px; }
    .button-group { text-align:center; margin:15px 0; }
    .btn {
      padding:8px 16px; margin:0 5px; border:none; border-radius:4px;
      cursor:pointer; background:#ccc; color:#333;
    }
    .btn-default { background:#ccc; }
    .btn-blue    { background:#007BFF; color:#fff; }
    .bid-row { display:flex; align-items:center; margin-bottom:10px; }
    .bid-row label { width:40px; font-weight:bold; }
    .bid-row input {
      flex:1; padding:6px; border:1px solid #ccc; border-radius:4px;
    }
    .submit {
      width:100%; padding:10px; background:#007BFF; color:#fff;
      border:none; border-radius:4px; font-size:1rem; cursor:pointer;
    }
    .submit:hover { background:#0056b3 }
    a.view { display:block; text-align:center; color:#007BFF; text-decoration:none }
    a.view:hover { text-decoration:underline }
  </style>
</head>
<body>

  <a href="games.php" class="back">← Back to Games</a>
  <div class="container">
    <h1>Single Digit Bidding</h1>

    <!-- Open/Close toggle -->
    <div class="button-group">
      <button type="button" class="btn <?= $openBtnClass ?>">Open</button>
      <button type="button" class="btn <?= $closeBtnClass ?>">Close</button>
    </div>

    <p class="info"><strong>User:</strong> <?= htmlspecialchars($username) ?></p>
    <p class="info">
      <strong>Game:</strong> <?= htmlspecialchars($game_name) ?><br>
      <strong>Open:</strong> <?= date("g:i A", strtotime($open_time)) ?> IST<br>
      <strong>Close:</strong> <?= date("g:i A", strtotime($close_time)) ?> IST
    </p>
    <p class="info"><strong>Wallet:</strong> ₹<?= number_format($wallet_balance,2) ?></p>

    <!-- only after a successful bid -->
    <?= $viewLink ?>

    <form method="POST">
      <p class="info">Min bid per digit: ₹<?= $min_bid ?></p>
      <?php for ($i = 0; $i <= 9; $i++): ?>
        <div class="bid-row">
          <label for="bid_<?= $i ?>"><?= $i ?>:</label>
          <input
            type="number"
            name="bid_<?= $i ?>"
            id="bid_<?= $i ?>"
            min="<?= $min_bid ?>"
            placeholder="₹<?= $min_bid ?>"
          >
        </div>
      <?php endfor; ?>
      <button class="submit" type="submit">Submit Bids</button>
    </form>
  </div>

</body>
</html>
<?php
$conn->close();
?>
