<?php
// single_digit_api.php

// 1) DEV: show all errors (disable in prod)
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

// 2) JSON response header
header('Content-Type: application/json; charset=UTF-8');

// 3) Database connection
require_once __DIR__ . '/db.php';

// 4) Decode JSON body (or fallback to $_REQUEST)
$raw = file_get_contents('php://input');
$input = json_decode($raw, true) ?: $_REQUEST;

// 5) Pull & validate
$mobile      = trim($input['mobile']      ?? '');
$gameId      = (int)($input['game_id']     ?? 0);
$gameName    = trim($input['game_name']    ?? '');
$bids        = $input['bids']              ?? [];

if ( ! $mobile
   || $gameId <= 0
   || $gameName === ''
   || ! is_array($bids)
 )  {
    http_response_code(400);
    exit(json_encode([
        'status'  => false,
        'message' => 'Required: mobile (string), game_id (int), game_name (string), bids (array).'
    ]));
}

try {
    // 6) Lookup user_id
    $stmt = $conn->prepare("SELECT user_id FROM users_data WHERE mobile = ?");
    $stmt->bind_param('s', $mobile);
    $stmt->execute();
    $stmt->bind_result($user_id);
    if (!$stmt->fetch()) {
        http_response_code(404);
        exit(json_encode(['status'=>false,'message'=>'User not found.']));
    }
    $stmt->close();

    // 7) Lookup game by game_number
    // 7) Lookup game by game_number
$stmt = $conn->prepare("
     SELECT 
      id,
      open_time,
      close_time,
      open_bid_last_time,
      close_bid_last_time
    FROM game56 
   WHERE id = ?
     AND LOWER(game_name) = LOWER(?)
   LIMIT 1
");
$stmt->bind_param('is', $gameId, $gameName);
$stmt->execute();
$stmt->bind_result(
  $game_id,
  $open_time,
  $close_time,
  $open_bid_last_time,
  $close_bid_last_time
);

    if (!$stmt->fetch()) {
        http_response_code(404);
        exit(json_encode(['status'=>false,'message'=>'Game not found.']));
    }
    $stmt->close();

    // 8) Determine session (open vs close) with 10-minute buffer
    date_default_timezone_set('Asia/Kolkata');
    $now     = new DateTime();
    $openDT  = DateTime::createFromFormat('H:i:s', $open_time);
    $closeDT = DateTime::createFromFormat('H:i:s', $close_time);
    if ($closeDT <= $openDT) {
        // overnight close
        $closeDT->modify('+1 day');
    }
    $runStart = (clone $openDT)->modify('-10 minutes');
    $runEnd   = (clone $closeDT)->modify('-10 minutes');

    if ($now < $runStart) {
        $session = 'open';
    } elseif ($now <= $runEnd) {
        $session = 'close';
    } else {
        // after close window — no more bids
        http_response_code(403);
        exit(json_encode([
            'status'  => false,
            'message' => 'Bidding period has ended.'
        ]));
    }

    // 9) Sum & validate bids
    $minBid = 10;
    $total  = 0;
    $clean  = [];
    foreach ($bids as $digit => $amt) {
        $d = (int)$digit;
        $a = floatval($amt);
        if ($d < 0 || $d > 9) continue;
        if ($a >= $minBid) {
            $clean[$d] = $a;
            $total += $a;
        }
    }
    if ($total <= 0) {
        http_response_code(400);
        exit(json_encode(['status'=>false,'message'=>"Minimum bid is ₹{$minBid}."]));
    }

    // 10) Check wallet balance
    $stmt = $conn->prepare("SELECT balance FROM wallet WHERE user_id = ?");
    $stmt->bind_param('i', $user_id);
    $stmt->execute();
    $stmt->bind_result($walletBalance);
    $stmt->fetch();
    $stmt->close();

    if ($total > $walletBalance) {
        http_response_code(400);
        exit(json_encode([
            'status'=>false,
            'message'=>'Insufficient wallet balance.',
            'data' => ['wallet_balance'=>$walletBalance]
        ]));
    }

    // 11) Insert bids and deduct wallet
    $conn->begin_transaction();

    $details = json_encode($clean, JSON_NUMERIC_CHECK);
    $ins = $conn->prepare("
        INSERT INTO user_bids
          (user_id, game_id, game_name, bid_details, total_bid_amount, bid_session)
        VALUES (?,?,?,?,?,?)
    ");
    $ins->bind_param('iissds', 
        $user_id, $game_id, $gameName, $details, $total, $session
    );
    $ins->execute();
    $ins->close();

    $upd = $conn->prepare("UPDATE wallet SET balance = balance - ? WHERE user_id = ?");
    $upd->bind_param('di', $total, $user_id);
    $upd->execute();
    $upd->close();

    $conn->commit();
    $newBalance = $walletBalance - $total;

    echo json_encode([
        'status'         => true,
        'message'        => 'Bids placed successfully.',
        'data'           => [
            'game_id'             => $game_id,
          'session'             => $session,
          'wallet_balance'      => $newBalance,
          'open_time'           => $open_time,
          'close_time'          => $close_time,
          'open_bid_last_time'  => $open_bid_last_time,
          'close_bid_last_time' => $close_bid_last_time,
        ]
    ]);

} catch (Exception $e) {
    $conn->rollback();
    http_response_code(500);
    echo json_encode([
        'status'=>false,
        'message'=>'Server error: '.$e->getMessage()
    ]);
}

$conn->close();
