<?php
// sp_api.php — accepts multiple SP bids in one call

// ─── 0) Turn off on-screen errors (send them to your logs instead) ───────────
ini_set('display_errors',   0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL & ~E_NOTICE);

// ─── 1) JSON header ─────────────────────────────────────────────────────────
header('Content-Type: application/json; charset=utf-8');

// ─── 2) DB connection ───────────────────────────────────────────────────────
require_once __DIR__ . '/db.php';

// ─── 3) Parse & validate input ──────────────────────────────────────────────
$raw  = file_get_contents('php://input');
$in   = json_decode($raw, true) ?? [];

$mobile      = trim($in['mobile']      ?? '');
$game_id = intval($in['game_id'] ?? 0);
$game_name   = trim($in['game_name']   ?? '');
$bidsArr     = $in['bids']            ?? [];

if (
    $mobile === '' ||
    $game_id <= 0 ||
    $game_name === '' ||
    !is_array($bidsArr)
) {
    http_response_code(400);
    exit(json_encode([
      'status'  => false,
      'message' => 'Required: mobile, game_id, game_name & bids(array).'
    ]));
}

// ─── 4) Lookup user_id ──────────────────────────────────────────────────────
$stmt = $conn->prepare("SELECT user_id FROM users_data WHERE mobile = ?");
$stmt->bind_param('s', $mobile);
$stmt->execute();
$stmt->bind_result($user_id);
if (!$stmt->fetch()) {
    http_response_code(404);
    exit(json_encode(['status'=>false,'message'=>'User not found.']));
}
$stmt->close();

// ─── 5) Lookup game_id + times ───────────────────────────────────────────────
$stmt = $conn->prepare("
  SELECT id, open_time, close_time
    FROM game56
   WHERE id = ?
     AND LOWER(game_name) = LOWER(?)
");
$stmt->bind_param('is', $game_id, $game_name);
$stmt->execute();
$stmt->bind_result($game_id, $open_time, $close_time);
if (!$stmt->fetch()) {
    http_response_code(404);
    exit(json_encode(['status'=>false,'message'=>'Game not found or name mismatch.']));
}
$stmt->close();

// ─── 6) Check wallet balance ─────────────────────────────────────────────────
$stmt = $conn->prepare("SELECT balance FROM wallet WHERE user_id = ?");
$stmt->bind_param('i', $user_id);
$stmt->execute();
$stmt->bind_result($wallet_balance);
$stmt->fetch();
$stmt->close();

// ─── 7) Clean & total the bids ────────────────────────────────────────────────
$minBid    = 10;
$total     = 0.0;
$cleanBids = [];  // must match JSON key for later storage

foreach ($bidsArr as $item) {
    if (!is_array($item) ||
        !isset($item['number'], $item['amount'], $item['session'], $item['date'])) {
      continue;
    }
    $num = preg_replace('/\D/', '', (string)$item['number']);
    $amt = floatval($item['amount']);
    if ($num === '' || $amt < $minBid) continue;

    // bundle up exactly what you need
    $cleanBids[] = [
      'number'  => $num,
      'amount'  => $amt,
      'session' => $item['session'],
      'date'    => preg_match('/^\d{4}-\d{2}-\d{2}$/', $item['date'])
                    ? $item['date']
                    : date('Y-m-d'),
    ];
    $total += $amt;
}

if (empty($cleanBids)) {
    http_response_code(400);
    exit(json_encode([
      'status'  => false,
      'message' => "Please place at least one bid of ₹{$minBid}."
    ]));
}

if ($total > $wallet_balance) {
    http_response_code(400);
    exit(json_encode([
      'status'  => false,
      'message' => 'Insufficient balance.',
      'data'    => ['wallet_balance' => $wallet_balance],
    ]));
}

// ─── 8) Determine session (open/close) ──────────────────────────────────────
date_default_timezone_set('Asia/Kolkata');
$nowTs   = time();
$openTs  = strtotime(date('Y-m-d').' '.$open_time);
$closeTs = strtotime(date('Y-m-d').' '.$close_time);
if ($closeTs <= $openTs) $closeTs += 86400;
$session = ($nowTs >= $openTs && $nowTs <= $closeTs) ? 'close' : 'open';

// ─── 9) Insert + deduct in one transaction ──────────────────────────────────
$conn->begin_transaction();
try {
    $details = json_encode($cleanBids, JSON_NUMERIC_CHECK);

    $ins = $conn->prepare("
      INSERT INTO user_bids_sp
        (user_id, game_id, bid_details, total_bid_amount, bid_session, game_name, created_at)
      VALUES (?,?,?,?,?,?, NOW())
    ");
    // i = user_id, i = game_id, s = JSON details, d = double total, s = session, s = game_name
    $ins->bind_param('iisdss',
      $user_id,
      $game_id,
      $details,
      $total,
      $session,
      $game_name
    );
    $ins->execute();
    $ins->close();

    $upd = $conn->prepare("UPDATE wallet SET balance = balance - ? WHERE user_id = ?");
    $upd->bind_param('di', $total, $user_id);
    $upd->execute();
    $upd->close();

    $conn->commit();

    echo json_encode([
      'status'  => true,
      'message' => "Placed ₹{$total} on single panna ({$session}).",
      'data'    => ['wallet_balance' => $wallet_balance - $total],
    ]);
} catch (Exception $e) {
    $conn->rollback();
    http_response_code(500);
    echo json_encode([
      'status'  => false,
      'message' => 'Server error: ' . $e->getMessage()
    ]);
}

$conn->close();
