<?php
// tp_api.php — Triple Panna JSON API
date_default_timezone_set('Asia/Kolkata');
// 1) DEV: show all PHP errors (disable in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// 2) JSON response header
header('Content-Type: application/json; charset=utf-8');

// 3) DB connection
require_once __DIR__ . '/db.php';

try {
    // 4) Parse JSON POST body
    $body = file_get_contents('php://input');
    $json = json_decode($body, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Invalid JSON: ' . json_last_error_msg(), 400);
    }

    // 5) Extract & validate inputs
    $mobile    = trim($json['mobile']       ?? '');
    $game_id  = intval($json['game_id'] ?? 0);
    $game_name = trim($json['game_name']    ?? '');
    $bids      = $json['bids']             ?? [];

    if ($mobile === '' || $game_id <= 0 || $game_name === '' || !is_array($bids)) {
        throw new Exception('`mobile`, `game_id`, `game_name` and `bids` (assoc array) are required.', 400);
    }

    // 6) Lookup user_id
    $stmt = $conn->prepare("SELECT user_id FROM users_data WHERE mobile = ? LIMIT 1");
    $stmt->bind_param('s', $mobile);
    $stmt->execute();
    $stmt->bind_result($user_id);
    if (!$stmt->fetch()) {
        throw new Exception('User not found.', 404);
    }
    $stmt->close();

    // 7) Verify game and fetch open/close times
    $stmt = $conn->prepare("
        SELECT id, open_time, close_time
          FROM game56
         WHERE id = ?
           AND LOWER(game_name) = LOWER(?)
         LIMIT 1
    ");
    $stmt->bind_param('is', $game_id, $game_name);
    $stmt->execute();
    $stmt->bind_result($game_id, $open_time, $close_time);
    if (!$stmt->fetch()) {
        throw new Exception('Game not found or name mismatch.', 404);
    }
    $stmt->close();
    
    $fmt = function($t){
    return date('g:i A', strtotime($t)) . ' IST';
};

// format them
$open_f        = $fmt($open_time);
$close_f       = $fmt($close_time);
// if you have last-bid fields in your SELECT, format them too:
$obl_f         = isset($open_bid_last_time)
               ? $fmt($open_bid_last_time)
               : null;
$cbl_f         = isset($close_bid_last_time)
               ? $fmt($close_bid_last_time)
               : null;

    // 8) Fetch wallet balance
    $stmt = $conn->prepare("SELECT balance FROM wallet WHERE user_id = ?");
    $stmt->bind_param('i', $user_id);
    $stmt->execute();
    $stmt->bind_result($wallet_balance);
    $stmt->fetch();
    $stmt->close();

    // 9) Clean & sum bids (only 3-digit keys, amount ≥ ₹10)
    $min_bid   = 10;
    $total     = 0.0;
    $cleanBids = [];
    foreach ($bids as $num => $amt) {
        if (preg_match('/^[0-9]{3}$/', (string)$num) && is_numeric($amt)) {
            $a = floatval($amt);
            if ($a >= $min_bid) {
                $cleanBids[$num] = $a;
                $total += $a;
            }
        }
    }
    if (empty($cleanBids)) {
        throw new Exception("Please place at least one bid of ₹{$min_bid}.", 400);
    }
    if ($total > $wallet_balance) {
        http_response_code(400);
        echo json_encode([
            'status'  => false,
            'message' => 'Insufficient wallet balance.',
            'data'    => ['wallet_balance' => $wallet_balance]
        ]);
        exit;
    }

    // 10) Determine session (open/close) in IST
    date_default_timezone_set('Asia/Kolkata');
    $now    = date('H:i:s');
    if ($open_time < $close_time) {
        $session = ($now >= $open_time && $now <= $close_time) ? 'close' : 'open';
    } else {
        $session = ($now >= $open_time || $now <= $close_time) ? 'close' : 'open';
    }

    // 11) Insert + deduct in one transaction
    $conn->begin_transaction();

    // 11a) Insert the clean JSON map
    $details = json_encode($cleanBids, JSON_NUMERIC_CHECK);
    $ins = $conn->prepare("
        INSERT INTO user_bids_tp
          (user_id, game_id, bid_details, total_bid_amount, bid_session, game_name, created_at)
        VALUES (?, ?, ?, ?, ?, ?, NOW())
    ");
    $ins->bind_param(
      'iisdss',
      $user_id,
      $game_id,
      $details,
      $total,
      $session,
      $game_name
    );
    $ins->execute();
    $ins->close();

    // 11b) Deduct wallet balance
    $upd = $conn->prepare("UPDATE wallet SET balance = balance - ? WHERE user_id = ?");
    $upd->bind_param('di', $total, $user_id);
    $upd->execute();
    $upd->close();

    $conn->commit();

    // 12) Return success
    echo json_encode([
        'status'  => true,
        'message' => "Placed ₹{$total} on Triple Panna ({$session}).",
        'data'    => [
             'wallet_balance'      => $wallet_balance - $total,
    'session'             => $session,
    'open_time'           => $open_f,    // ← formatted
    'close_time'          => $close_f,   // ← formatted
    'open_bid_last_time'  => $obl_f,     // ← if available
    'close_bid_last_time' => $cbl_f,     // ← if available
        ]
    ]);

} catch (Exception $e) {
    // rollback on any error
    if ($conn->errno) {
        $conn->rollback();
    }
    $code = $e->getCode() >= 400 && $e->getCode() < 600 ? $e->getCode() : 500;
    http_response_code($code);
    echo json_encode([
        'status'  => false,
        'message' => $e->getMessage()
    ]);
}

$conn->close();
