<?php
session_start();
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include 'db.php';

// Redirect if not logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

date_default_timezone_set('Asia/Kolkata');

/**
 * Resolve target date (YYYY-MM-DD). Defaults to today. Sanitizes bad input.
 */
$today = date('Y-m-d');
$targetDate = isset($_GET['d']) ? trim($_GET['d']) : $today;
$dt = DateTime::createFromFormat('Y-m-d', $targetDate);
if (!$dt || $dt->format('Y-m-d') !== $targetDate) {
    $targetDate = $today; // fallback if invalid
}

/**
 * Determine reference "now" for status logic:
 * - viewing today  -> real now
 * - viewing past   -> 23:59:59 of that date (so it's closed)
 * - viewing future -> 00:00:00 of that date (so it's not started)
 */
if ($targetDate === $today) {
    $refNow = time();
} elseif ($targetDate < $today) {
    $refNow = strtotime($targetDate . ' 23:59:59');
} else {
    $refNow = strtotime($targetDate . ' 00:00:00');
}

/**
 * Detect which date column to use for filtering user_bids_tp.
 * Prefers created_at, then bid_date; fallback = no date filter if neither exists.
 */
function detect_date_column(mysqli $conn): ?string {
    $col = null;
    if ($res = $conn->query("SHOW COLUMNS FROM user_bids_tp LIKE 'created_at'")) {
        if ($res->num_rows > 0) { $col = 'created_at'; }
        $res->free();
    }
    if (!$col && ($res2 = $conn->query("SHOW COLUMNS FROM user_bids_tp LIKE 'bid_date'"))) {
        if ($res2->num_rows > 0) { $col = 'bid_date'; }
        $res2->free();
    }
    return $col;
}
$dateCol = detect_date_column($conn);

/**
 * Normalise a number to a 3-digit string "000".."999". Return null if non-numeric/out of range.
 */
function norm_tp($num): ?string {
    if ($num === null) return null;
    if (is_string($num)) { $num = trim($num); }
    if (!is_numeric($num)) return null;
    $n = (int)$num;
    if ($n < 0 || $n > 999) return null;
    return str_pad((string)$n, 3, '0', STR_PAD_LEFT);
}

/**
 * Currency formatting helper (no decimals if integer).
 */
$fmt = function($v) {
    if (!is_numeric($v)) return $v;
    return (fmod((float)$v, 1.0) === 0.0) ? number_format((int)$v) : number_format((float)$v, 2);
};

// Triple Panna fixed numbers
$triplePannaNumbers = ['000', '111', '222', '333', '444', '555', '666', '777', '888', '999'];

// Fetch all games, sorted by open_time in ascending order
$sql = "SELECT * FROM game56 ORDER BY open_time ASC";
$result = $conn->query($sql);
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Triple Panna - Highest & Lowest Bids</title>
<style>
body { background:#f8f9fa; font-family:Arial,sans-serif; margin:0; }
.navbar { background:#007BFF; color:#fff; text-align:center; padding:12px; position:relative; }
.navbar h2 { margin:0; font-size:1.1rem; }
.back-button {
  position:absolute; left:10px; top:50%; transform:translateY(-50%);
  background:#28a745; color:#fff; padding:6px 12px; text-decoration:none; border-radius:4px;
}
.filter-bar {
  background:#fff; border-bottom:1px solid #e5e5e5; padding:10px 12px;
  display:flex; align-items:center; justify-content:center; gap:8px; flex-wrap:wrap;
}
.filter-bar form { display:flex; align-items:center; gap:8px; flex-wrap:wrap; }
.filter-bar label { font-weight:bold; }
.filter-bar input[type="date"] { padding:6px 8px; border:1px solid #ccc; border-radius:4px; }
.btn { padding:6px 10px; border:none; border-radius:4px; cursor:pointer; font-size:0.9rem; text-decoration:none; display:inline-block; }
.btn-primary { background:#007BFF; color:#fff; }
.btn-secondary { background:#6c757d; color:#fff; }
.meta { text-align:center; font-size:0.9rem; color:#333; margin:10px 0; }

.content {
  display:grid; grid-template-columns:repeat(auto-fit,minmax(260px,1fr));
  gap:16px; max-width:1000px; margin:20px auto; padding:0 12px;
}
.game-card {
  background:#fff; border-radius:8px; border:2px solid #007BFF;
  box-shadow:0 2px 6px rgba(0,0,0,0.1); overflow:hidden;
}
.game-card h3 {
  background:#007BFF; color:#fff; padding:10px; margin:0; text-align:center;
}
.body { padding:12px; text-align:center; }
.status-message { color:#d9534f; font-weight:bold; margin-bottom:10px; }
.minmax { display:flex; flex-direction:column; gap:8px; }
.highest, .lowest, .nobid {
  padding:8px; border-radius:4px; display:flex; justify-content:space-between; align-items:center;
}
.highest { background:#d4edda; }
.lowest  { background:#f8d7da; }
.nobid   { background:#fff3cd; }
@media (max-width:640px){ .back-button { font-size:0.85rem; padding:5px 9px; } }
</style>
</head>
<body>

<div class="navbar">
  <a href="dashboard.php" class="back-button">Back</a>
  <h2>Triple Panna - Highest & Lowest Bids</h2>
</div>

<div class="filter-bar">
  <form method="get" action="">
    <label for="d">Select Date:</label>
    <input type="date" id="d" name="d" value="<?php echo htmlspecialchars($targetDate); ?>" required>
    <button type="submit" class="btn btn-primary">Apply</button>
    <a class="btn btn-secondary" href="?d=<?php echo htmlspecialchars($today); ?>">Today</a>
  </form>
</div>

<div class="meta">
  Showing data for: <strong><?php echo htmlspecialchars($targetDate); ?></strong>
  <?php if (!$dateCol): ?>
    <div style="color:#a94442;margin-top:6px;">
      Note: No date column (created_at/bid_date) found in <code>user_bids_tp</code>; totals include all bids for each game.
    </div>
  <?php endif; ?>
</div>

<div class="content">
<?php if ($result && $result->num_rows > 0): ?>
  <?php while ($row = $result->fetch_assoc()): ?>
    <?php
      // Build timestamps for the selected date using game's HH:MM:SS
      $open_ts  = strtotime($targetDate . ' ' . $row['open_time']);
      $close_ts = strtotime($targetDate . ' ' . $row['close_time']);

      // Status relative to selected date
      if ($targetDate > $today) {
          $status_text = "Game has not started";
      } elseif ($targetDate < $today) {
          $status_text = "Game has Closed";
      } else {
          $status_text = ($refNow >= $open_ts && $refNow <= $close_ts) ? "Game is Open" : "Game has Closed";
      }

      // Initialize totals for triple panna numbers only
      $totals = array_fill_keys($triplePannaNumbers, 0.0);

      // Fetch bids for this game (filtered by date if available)
      if ($dateCol) {
          $stmt = $conn->prepare("SELECT bid_details FROM user_bids_tp WHERE game_name = ? AND DATE($dateCol) = ?");
          $stmt->bind_param('ss', $row['game_name'], $targetDate);
      } else {
          $stmt = $conn->prepare("SELECT bid_details FROM user_bids_tp WHERE game_name = ?");
          $stmt->bind_param('s', $row['game_name']);
      }

      if ($stmt && $stmt->execute()) {
          $bidsRes = $stmt->get_result();
          while ($b = $bidsRes->fetch_assoc()) {
              $details = json_decode($b['bid_details'], true);
              if (!is_array($details)) { continue; }

              // Case A: Key-value object like {"000":100,"111":50}
              $is_assoc = array_keys($details) !== range(0, count($details) - 1);
              if ($is_assoc) {
                  foreach ($details as $num => $amt) {
                      $n = norm_tp($num);
                      if ($n === null) { continue; }
                      if (isset($totals[$n])) {
                          $totals[$n] += (float)$amt;
                      }
                  }
              } else {
                  // Case B1: Array of objects [{number:"000", amount:"100"}, ...]
                  if (!empty($details) && is_array($details[0]) && (isset($details[0]['number']) || isset($details[0]['num']))) {
                      foreach ($details as $entry) {
                          $rawNum = $entry['number'] ?? ($entry['num'] ?? null);
                          $rawAmt = $entry['amount'] ?? ($entry['amt'] ?? null);
                          $n = norm_tp($rawNum);
                          if ($n === null || $rawAmt === null) { continue; }
                          if (isset($totals[$n])) {
                              $totals[$n] += (float)$rawAmt;
                          }
                      }
                  }
                  // Case B2: Plain array of amounts where index is the number: amounts[0]=X -> "000", amounts[111]=Y -> "111"
                  else {
                      foreach ($details as $idx => $amt) {
                          $n = norm_tp($idx);
                          if ($n === null) { continue; }
                          if (isset($totals[$n])) {
                              $totals[$n] += (float)$amt;
                          }
                      }
                  }
              }
          }
          $bidsRes->free();
      }
      if ($stmt) { $stmt->close(); }

      // Highest & Lowest (non-zero lowest)
      $maxVal   = max($totals);
      $maxIndex = array_search($maxVal, $totals, true);

      $nonZero  = array_filter($totals, fn($v) => $v > 0);
      $minVal   = $nonZero ? min($nonZero) : 0;
      $minIndex = $nonZero ? array_search($minVal, $totals, true) : '-';

      // Example no-bid number among triple set
      $noBidNumbers = array_keys(array_filter($totals, fn($v) => $v == 0));
      $exampleNoBid = !empty($noBidNumbers) ? reset($noBidNumbers) : '-';
    ?>
    <div class="game-card">
      <h3><?= htmlspecialchars($row['game_name']) ?> (<?= htmlspecialchars($row['game_number']) ?>)</h3>
      <div class="body">
        <p class="status-message"><?= htmlspecialchars($status_text) ?></p>

        <div class="minmax">
          <div class="highest">
            <span>Highest bidding number:</span>
            <span><?= htmlspecialchars((string)$maxIndex) ?> (₹<?= $fmt($maxVal) ?>)</span>
          </div>
          <div class="lowest">
            <span>Lowest bidding number:</span>
            <span><?= htmlspecialchars((string)$minIndex) ?> (₹<?= $fmt($minVal) ?>)</span>
          </div>
          <div class="nobid">
            <span>Example no-bid number:</span>
            <span><?= htmlspecialchars((string)$exampleNoBid) ?></span>
          </div>
        </div>

        <!-- Display open and close times -->
        <div class="minmax" style="margin-top:8px;">
          <div class="highest">
            <span>Open Time:</span>
            <span><?= date('g:i A', $open_ts) ?> IST</span>
          </div>
          <div class="lowest">
            <span>Close Time:</span>
            <span><?= date('g:i A', $close_ts) ?> IST</span>
          </div>
        </div>
      </div>
    </div>
  <?php endwhile; ?>
<?php else: ?>
  <p style="text-align:center;margin-top:32px;">No games found.</p>
<?php endif; ?>
</div>

</body>
</html>
