<?php
session_start();
include 'db.php'; // $conn

// 1) Must be logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}
$user_id = (int)$_SESSION['user_id'];

// 2) game_id + mark sub_game
if (empty($_GET['game_id'])) {
    die("No game selected.");
}
$game_id = (int)$_GET['game_id'];
$stmt = $conn->prepare("
    UPDATE game56
       SET sub_games = 'triple_panna'
     WHERE id = ?
");
$stmt->bind_param("i", $game_id);
$stmt->execute();
$stmt->close();

// 3) Fetch this game’s open_time & close_time
$stmt = $conn->prepare("
    SELECT open_time, close_time
      FROM game56
     WHERE id = ?
");
$stmt->bind_param("i", $game_id);
$stmt->execute();
$stmt->bind_result($open_time, $close_time);
$stmt->fetch();
$stmt->close();

// 4) Determine bid_session (open vs close) in IST
date_default_timezone_set('Asia/Kolkata');
$now = date('H:i:s');

if ($open_time < $close_time) {
    // same‑day window
    $bid_session = ($now >= $open_time && $now <= $close_time)
                 ? 'close'
                 : 'open';
} else {
    // overnight window
    $bid_session = ($now >= $open_time || $now <= $close_time)
                 ? 'close'
                 : 'open';
}

// 5) Fetch wallet balance
$wallet_balance = 0;
$res = $conn->prepare("
    SELECT balance
      FROM wallet
     WHERE user_id = ?
");
$res->bind_param("i", $user_id);
$res->execute();
$res->bind_result($wallet_balance);
$res->fetch();
$res->close();

// 6) Define Triple Panna numbers
$triplePanas = ["000","111","222","333","444","555","666","777","888","999"];
$allNumbers  = $triplePanas; // dropdown uses same list

// 7) Handle bid submission
$message = "";
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['place_bid'])) {
    $selected = $_POST['selected_number'] ?? '';
    $bid      = floatval($_POST['bid_amount'] ?? 0);

    if (!$selected) {
        $message = "<div class='alert alert-danger'>Please select a number.</div>";
    } elseif ($bid < 10) {
        $message = "<div class='alert alert-danger'>Minimum bid is \$10.</div>";
    } elseif ($bid > $wallet_balance) {
        $message = "<div class='alert alert-danger'>Insufficient wallet balance.</div>";
    } else {
        $conn->begin_transaction();
        try {
            // Insert into user_bids_tp
            $ins = $conn->prepare("
              INSERT INTO user_bids_tp
                (user_id, game_id, bid_details, total_bid_amount, bid_session, game_name)
              VALUES (?,?,?,?,?, (SELECT game_name FROM game56 WHERE id=?))
            ");
            $ins->bind_param("iisdsi",
                $user_id,
                $game_id,
                $selected,
                $bid,
                $bid_session,
                $game_id
            );
            $ins->execute();
            $ins->close();

            // Deduct from wallet
            $upd = $conn->prepare("
              UPDATE wallet
                 SET balance = balance - ?
               WHERE user_id = ?
            ");
            $upd->bind_param("di", $bid, $user_id);
            $upd->execute();
            $upd->close();

            $conn->commit();

            // Redirect to avoid re‑POST on reload
            header("Location: " . $_SERVER['PHP_SELF'] . "?game_id={$game_id}");
            exit;
        } catch (Exception $e) {
            $conn->rollback();
            $message = "<div class='alert alert-danger'>
                          Error placing bid: {$e->getMessage()}
                        </div>";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Triple Panna Bidding</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <style>
    body { background:#f8f9fa; margin:0; font-family:Arial,sans-serif; color:#333; }
    .back-button {
      position:absolute; top:20px; left:20px;
      background:#28a745; color:#fff; padding:8px 12px;
      text-decoration:none; border-radius:4px; font-size:14px;
    }
    .back-button:hover { background:#218838; }
    .header { text-align:center; padding:20px; }
    .wallet { font-size:1.2rem; margin-bottom:10px; }
    .grid {
      display:grid;
      grid-template-columns:1fr;
      gap:20px; padding:20px;
    }
    .card {
      background:#fff; border:2px solid #f36; border-radius:8px;
      padding:15px; box-shadow:0 2px 5px rgba(0,0,0,0.1);
    }
    .card h2 { margin-top:0; color:#007BFF; font-size:1.2rem; }
    .numbers { margin:10px 0; font-size:1rem; }
    .form-container {
      max-width:400px; margin:30px auto; background:#fff; padding:20px;
      border-radius:8px; box-shadow:0 2px 5px rgba(0,0,0,0.1);
    }
    .form-container h3 { margin-top:0; color:#007BFF; text-align:center; }
    .alert { padding:0.75rem; margin-bottom:1rem; border-radius:4px; }
    .alert-success { background:#d4edda; color:#155724; }
    .alert-danger { background:#f8d7da; color:#721c24; }
    label { display:block; margin:0.5rem 0; font-weight:bold; }
    select,input[type=number] {
      width:100%; padding:0.5rem; font-size:1rem; border:1px solid #ccc;
      border-radius:4px; margin-bottom:1rem;
    }
    button {
      width:100%; padding:0.75rem; font-size:1rem; color:#fff;
      background:#007BFF; border:none; border-radius:4px; cursor:pointer;
    }
    button:hover { background:#0056b3; }
  </style>
</head>
<body>

  <a href="dashboard.php" class="back-button">← Dashboard</a>

  <div class="header">
    <h1>Triple Panna Bidding</h1>
    <div class="wallet">Wallet Balance: $<?php echo number_format($wallet_balance,2) ?></div>
  </div>

  <div class="grid">
    <div class="card">
      <h2>Triple Panna’s</h2>
      <div class="numbers"><?php echo implode(", ", $triplePanas) ?></div>
    </div>
  </div>

  <div class="form-container">
    <h3>Place Your Bid</h3>
    <?php echo $message ?>
    <form method="POST">
      <label for="selected_number">Select Number:</label>
      <select name="selected_number" id="selected_number" required>
        <option value="">-- Choose --</option>
        <?php foreach($allNumbers as $n): ?>
          <option value="<?=htmlspecialchars($n)?>"><?=htmlspecialchars($n)?></option>
        <?php endforeach; ?>
      </select>

      <label for="bid_amount">Bid Amount ($):</label>
      <input type="number" name="bid_amount" id="bid_amount"
             step="0.01" min="10" required>

      <button type="submit" name="place_bid">Place Bid</button>
    </form>
  </div>

</body>
</html>
<?php
$conn->close();
?>
